/**
 * BeteiligtenHinzufuegenDialog.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.Frame;
import java.awt.GridBagConstraints;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;

import eu.gronos.kostenrechner.controller.ComponentBeschrifter;
import eu.gronos.kostenrechner.controller.baumbach.BeteiligtenPruefer;
import eu.gronos.kostenrechner.controller.baumbach.WiderklageCheckboxTextfieldHorcher;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.Vorsorgend;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.beschriftungen.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.view.GitterBeutelBeschraenkungen;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;

/**
 * Dialog zum Hinzufügen eines (einfachen) BaumbachBeteiligten
 * 
 */
public class BeteiligtenHinzufuegenDialog extends HinzufuegenDialog<BaumbachBeteiligter> {

	private static final long serialVersionUID = 2242125254034647717L;
	JComboBox<Beteiligter> cmbTypGenusNumerus;
	JCheckBox cbAnWiderklageBeteiligt;
	JFormattedTextField ftfStreitwert;
	JFormattedTextField ftfUnterliegen;
	final int beteiligtenTyp;
	final boolean widerKlageAnzeigen;
	private BaumbachBeteiligter beteiligter;
	private JLabel lblStreitwert;
	private JLabel lblUnterliegen;
	private ParsendUndBauend<BaumbachBeteiligter> pruefer = new BeteiligtenPruefer();

	/**
	 * Create the dialog.
	 * 
	 * @param owner              the JFrame from which the dialog is displayed
	 * @param beteiligter        ein BaumbachBeteiligter mit den voreingestellten
	 *                           Werten für den Beteiligtentyp, Genus, Numerus,
	 *                           Streitwert, Unterliegen und ob er an der Widerklage
	 *                           beteiligt ist.
	 * @param widerKlageAnzeigen wenn true, werden zusätzliche Felder für die
	 *                           Widerklage angezeigt
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#HinzufuegenDialog(JFrame,String)
	 * @see javax.swing.JDialog#JDialog(Frame,String,boolean)
	 */
	public BeteiligtenHinzufuegenDialog(JFrame owner, BaumbachBeteiligter beteiligter, boolean widerKlageAnzeigen) {
		super(owner, (Beschriftung) new ComponentBeschrifter()
				.changeTo(((Vorsorgend<?>) NameContainerSammlung.BESCHRIFTUNGEN.get(35000)), beteiligter.getTyp()));
		this.widerKlageAnzeigen = widerKlageAnzeigen;
		this.beteiligter = beteiligter;
		this.beteiligtenTyp = beteiligter.getTyp();
	}

	/**
	 * Methode zum Plausis checken, wird vom OK-Button aufgerufen.
	 * 
	 * @return true, wenn alle Plausis eingehalten werden
	 */
	@Override
	public boolean parseEingabe() {
		return pruefer.leseComponents(cmbTypGenusNumerus, cbAnWiderklageBeteiligt, ftfStreitwert, ftfUnterliegen)
				.parseEingabe();
	}

	/**
	 * @return beteiligtenTyp gibt den beteiligtenTyp als int zurück.
	 */
	int getBeteiligtenTyp() {
		return beteiligtenTyp;
	}

	/**
	 * Die Methode füllt den Inhaltsbereich des Dialogs mit den Eingabe- und
	 * Auswahlkomponenten
	 * 
	 */
	@Override
	protected void fuelleContentPane() {
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		JLabel lblBeteiligtentyp = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(35001));
		lblBeteiligtentyp.setHorizontalAlignment(SwingConstants.RIGHT);
		cmbTypGenusNumerus = fillBeteiligtenCombo(beteiligtenTyp);
		beschrifter.beschrifte(cmbTypGenusNumerus, NameContainerSammlung.BESCHRIFTUNGEN.get(35005));
		lblBeteiligtentyp.setLabelFor(cmbTypGenusNumerus);

		cbAnWiderklageBeteiligt = (JCheckBox) beschrifter.beschrifte(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(35011));
		cbAnWiderklageBeteiligt.setSelected(beteiligter.isAnWiderklageBeteiligt());
		/*
		 * Die CheckBox soll nicht erscheinen, wenn Widerklagen nicht angezeigt werden.
		 * Der BeteiligtenPruefer nimmt sich daraus die Info, ob Widerklagen angezeigt
		 * werden sollen.
		 */
		cbAnWiderklageBeteiligt.setVisible(widerKlageAnzeigen);

		lblStreitwert = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(35021));
		lblStreitwert.setHorizontalAlignment(SwingConstants.RIGHT);
		ftfStreitwert = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(35025));
		ftfStreitwert.setText(Double.toString(beteiligter.getStreitwert()));
		setEnterAction(ftfStreitwert, getOkAction());
		if (!widerKlageAnzeigen && beteiligtenTyp == Beteiligter.KLAEGER) {
			ftfStreitwert.setText("0");
			lblStreitwert.setVisible(false);
			ftfStreitwert.setVisible(false);
		}
		lblStreitwert.setLabelFor(ftfStreitwert);

		lblUnterliegen = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(35031));
		lblUnterliegen.setHorizontalAlignment(SwingConstants.RIGHT);
		ftfUnterliegen = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(35035));
		ftfUnterliegen.setText(Double.toString(beteiligter.getUnterliegen()));
		setEnterAction(ftfUnterliegen, getOkAction());
		if (!widerKlageAnzeigen && beteiligtenTyp == Beteiligter.KLAEGER) {
			ftfUnterliegen.setText("0");
			lblUnterliegen.setVisible(false);
			ftfUnterliegen.setVisible(false);
		}
		lblUnterliegen.setLabelFor(ftfUnterliegen);

		contentPanel.add(lblBeteiligtentyp,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(cmbTypGenusNumerus,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(cbAnWiderklageBeteiligt,
				GitterBeutelBeschraenkungen.getInstance(2, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(lblStreitwert,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(ftfStreitwert,
				GitterBeutelBeschraenkungen.getInstance(1, 2, 2, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(lblUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(ftfUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(1, 3, 2, 1, GridBagConstraints.HORIZONTAL, false));

		final WiderklageCheckboxTextfieldHorcher horcher = new WiderklageCheckboxTextfieldHorcher(cmbTypGenusNumerus,
				cbAnWiderklageBeteiligt, ftfStreitwert, ftfUnterliegen,
				(VorsorgendeBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(35011));
		cmbTypGenusNumerus.addItemListener(horcher);
		cbAnWiderklageBeteiligt.addChangeListener(horcher);
		// Sonst gibt es IndexOutOfBounds!
		cmbTypGenusNumerus.setSelectedIndex(beteiligter.getGenusNumerus() % Beteiligter.PLURAL);
		// beteiligter.getGenusNumerus() > Beteiligter.PLURAL ?
		// beteiligter.getGenusNumerus() - Beteiligter.PLURAL
		// : beteiligter.getGenusNumerus());
		horcher.stateChanged(new ChangeEvent(this));
		horcher.itemStateChanged(null);
	}

	@Override
	protected BaumbachBeteiligter baueRueckgabewert() {
		return pruefer.leseComponents(cmbTypGenusNumerus, cbAnWiderklageBeteiligt, ftfStreitwert, ftfUnterliegen)
				.getRueckgabewert();
	}

	/**
	 * Die Methode fillBeteiligtenCombo dient dazu, eine JComboBox mit Beteiligten
	 * eines Typs zu füllen
	 * 
	 * @param beteiligtenTyp
	 * @return
	 */
	private JComboBox<Beteiligter> fillBeteiligtenCombo(int beteiligtenTyp) {
		Beteiligter[] beteiligtenAuswahlListe = BaumbachBeteiligter.getBeteiligtenAuswahlListe(beteiligtenTyp, false);
		JComboBox<Beteiligter> jComboBox = new JComboBox<Beteiligter>(beteiligtenAuswahlListe);
		jComboBox.setEnabled(true);
		jComboBox.setEditable(false);
		jComboBox.setRenderer(BeteiligtenCellEditor.BETEILIGTEN_RENDERER_OHNE_LFDNR);
		return jComboBox;
	}

}

// public static final String TEXT_VERURTEILUNG = "Höhe der Verurteilung /
// darüber hinaus";
// public static final String TEXT_STREITWERT = "Höhe der Inanspruchnahme /
// Klageantrag";
// ;"Beteiligtentyp"
// super(owner, String.format("%s hinzufügen",
// beteiligter.kurzBezeichner(Beteiligter.AKKUSATIV)));
// lblBeteiligtentyp.setToolTipText(Beschriftung.baueHtmlTooltip(lblBeteiligtentyp.getText(),
// "Bitte wählen Sie das Geschlecht aus!"));
// cmbTypGenusNumerus.setName("cmbTypGenusNumerus");
// setMnemonicLabelFor(, , KeyEvent.VK_B);
// ItemListener itemListener = new WiderklageGenusItemHorcher();
// cbAnWiderklageBeteiligt.setName("cbAnWiderklageBeteiligt");
// cbAnWiderklageBeteiligt.setMnemonic(KeyEvent.VK_W);
// TEXT_STREITWERT
// lblStreitwert.setToolTipText(Beschriftung.baueHtmlTooltip(lblStreitwert.getText(),
// "Bitte geben Sie ein, auf welche Höhe der Klageantrag beziffert ist!"));
// ftfStreitwert.setName("ftfStreitwert");
// setMnemonicLabelFor(, , KeyEvent.VK_I);
// TEXT_VERURTEILUNG
// lblUnterliegen.setToolTipText(Beschriftung.baueHtmlTooltip(lblUnterliegen.getText(),
// "Bitte geben Sie ein, in welcher Höhe Sie (einzeln) verurteilen!"));
// ftfUnterliegen.setName("ftfUnterliegen");
// setMnemonicLabelFor(, , KeyEvent.VK_V);
// itemListener);
// new WiderklageVeraenderbarkeitsHorcher());
// adjustWiderklageCheckBox();
// adjustTextfieldEditability();
