/**
 * GesamtschuldnerDialog.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.Frame;
import java.awt.GridBagConstraints;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;

import eu.gronos.kostenrechner.controller.baumbach.GesamtschuldPruefer;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenExtractAndCancelAction;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenListModel;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.view.GitterBeutelBeschraenkungen;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;

/**
 * Dialog zum Hinzufügen eines BaumbachGesamtschuldners
 * 
 * @author Peter Schuster (setrok)
 * @date 16.06.2014
 */
public class GesamtschuldnerDialog extends HinzufuegenDialog<BaumbachGesamtschuldnerschaft> {
	private static final long serialVersionUID = 7906671533792996231L;
	private final AbstractAction extrahierenAction;

	private BaumbachBeteiligtenListe baumbachBeteiligtenListe;
	private JScrollPane scpBeteiligtenAuswahlListe;

	private JList<BaumbachBeteiligter> beteiligtenAuswahlListe;

	private JFormattedTextField ftfUnterliegen;
	private ParsendUndBauend<BaumbachGesamtschuldnerschaft> pruefer = new GesamtschuldPruefer();

	/**
	 * Erzeugt einen modalen Dialog zum Hinzufügen eines Gesamtschuldners
	 * 
	 * @param owner the JFrame from which the dialog is displayed
	 * @param kl    die BaumbachBeteiligtenListe mit den Klägern
	 * @param dwbk  die BaumbachBeteiligtenListe mit den Drittwiderbeklagten, zu der
	 *              ein Beteiligter hinzugefügt werden soll
	 */
	public GesamtschuldnerDialog(JFrame owner, BaumbachBeteiligtenListe kl, BaumbachBeteiligtenListe dwbk,
			AbstractAction extrahierenAction) {
		this(owner, new BaumbachBeteiligtenListe(kl, dwbk), extrahierenAction);
	}

	/**
	 * Erzeugt einen modalen Dialog zum Hinzufügen eines Gesamtschuldners
	 * 
	 * @param owner                    the JFrame from which the dialog is displayed
	 * @param baumbachBeteiligtenListe die BaumbachBeteiligtenListe, zu der ein
	 *                                 Beteiligter hinzugefügt werden soll
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#HinzufuegenDialog(JFrame,String)
	 * @see javax.swing.JDialog#JDialog(Frame, String, boolean)
	 * @throws IllegalArgumentException wenn die
	 *                                  <code>baumbachBeteiligtenListe</code> leer
	 *                                  ist
	 */
	public GesamtschuldnerDialog(JFrame owner, BaumbachBeteiligtenListe baumbachBeteiligtenListe,
			AbstractAction extrahierenAction) throws IllegalArgumentException {
		super(owner, (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(36000));
		if (baumbachBeteiligtenListe.size() < 1)
			throw new IllegalArgumentException("Keine Beteiligten für den angegebenen Beteiligtentyp vorhanden, "
					+ "\ndie gesamtschuldnerisch verurteilt werden könnten. "
					+ "\nBitte zunächst \"einfache\" Beteiligte der Liste hinzufügen.");
		else
			this.baumbachBeteiligtenListe = baumbachBeteiligtenListe;
		LangBeschriftung beschriftung = (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(36031);
		beschriftung.setActionCommandKey((String) extrahierenAction.getValue(Action.ACTION_COMMAND_KEY));
		this.extrahierenAction = new StreitgenossenExtractAndCancelAction(beschriftung, extrahierenAction,
				super.getCancelAction());
	}

	/**
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		return pruefer.leseComponents(beteiligtenAuswahlListe, ftfUnterliegen).parseEingabe();
	}

	/**
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#baueRueckgabewert()
	 */
	@Override
	protected BaumbachGesamtschuldnerschaft baueRueckgabewert() {
		return pruefer.leseComponents(beteiligtenAuswahlListe, ftfUnterliegen).getRueckgabewert();
	}

	/**
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#fuelleContentPane()
	 */
	@Override
	protected void fuelleContentPane() {
		JLabel lblBeteiligtentyp = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(36001));
		beteiligtenAuswahlListe = new JList<BaumbachBeteiligter>(new BeteiligtenListModel(baumbachBeteiligtenListe));
		beschrifter.beschrifte(beteiligtenAuswahlListe, NameContainerSammlung.BESCHRIFTUNGEN.get(36011));
		beteiligtenAuswahlListe.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		beteiligtenAuswahlListe.setCellRenderer(new BeteiligtenListRenderer());
		scpBeteiligtenAuswahlListe = (JScrollPane) beschrifter.beschrifte(new JScrollPane(beteiligtenAuswahlListe),
				NameContainerSammlung.BESCHRIFTUNGEN.get(36010));
		lblBeteiligtentyp.setLabelFor(scpBeteiligtenAuswahlListe);

		JLabel lblUnterliegen = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(36021));
		ftfUnterliegen = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(36025));
		ftfUnterliegen.setText(Double.toString(getLetztenBeteiligten().getUnterliegen()));
		ftfUnterliegen.setColumns(10);
		setEnterAction(ftfUnterliegen, getOkAction());
		lblUnterliegen.setLabelFor(ftfUnterliegen);

		contentPanel.add(lblBeteiligtentyp,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(scpBeteiligtenAuswahlListe,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 2, 1, GridBagConstraints.BOTH, true));
		contentPanel.add(lblUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		contentPanel.add(ftfUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));

		getButtonPane().add(new JButton(extrahierenAction));
		addToRootInputMap(extrahierenAction);
	}

	/**
	 * Die Methode greift den letzten Eintrag aus der übergebenen
	 * baumbachBeteiligtenListe
	 * 
	 * @return
	 */
	private BaumbachBeteiligter getLetztenBeteiligten() {
		return this.baumbachBeteiligtenListe.get(this.baumbachBeteiligtenListe.size() - 1);
	}
}
// class GesamtschuldnerDialog extends JDialog {
// public static final String TEXT_UNTERLIEGEN = "Höhe der Verurteilung";
// super(owner, "Gesamtschuldner hinzufügen", true);
// "Gesamtschuldnerische Verurteilung hinzufügen"
//"Automatisch extrahieren",KeyEvent.VK_E,
//"Geht alle Beteiligte aus der Liste durch und guckt, ob sie z.T. gesamtschuldnerisch verurteilt wurden",
//(String) extrahierenAction.getValue(Action.ACTION_COMMAND_KEY), 
// StreitgenossenPanel.BETEILIGTEN_RENDERER);
//baumbachBeteiligtenListe.toArray(new BaumbachBeteiligter[baumbachBeteiligtenListe.size()]));
// "Gesamtschuldnerisch verurteilte Beteiligte auswählen"
// lblBeteiligtentyp.setToolTipText(Beschriftung.baueHtmlTooltip(lblBeteiligtentyp.getText(),
// "Bitte wählen Sie die Beklagten/Widerbeklagten aus, die gesamtschuldnerisch
// verurteilt werden!"));
// beteiligtenAuswahlListe.setName("beteiligtenAuswahlListe");
// setMnemonicLabelFor(, , KeyEvent.VK_B);
// TEXT_UNTERLIEGEN
// lblUnterliegen.setToolTipText(Beschriftung.baueHtmlTooltip(lblUnterliegen.getText(),
// "Bitte geben Sie ein, in welcher Höhe Sie die Beklagten/Widerbeklagten
// gesamtschuldnerisch verurteilen!"));
// ftfUnterliegen.setName("ftfUnterliegen");
// setMnemonicLabelFor(, , KeyEvent.VK_V);
