/**
 * BaumbachPanel.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.BeschriebeneAktion;
import eu.gronos.kostenrechner.controller.EintragEntfernenAktion;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenAssistentAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenBaumbachAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenBeteiligterNeuAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldBkExtractAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldDwbkExtractAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldnerBkAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldnerDwbkAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenPruefer;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenWiderklageAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenXJustizLesenAction;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.StreitgenossenTabellenBehandelnd;
import eu.gronos.kostenrechner.interfaces.WerteLesbar;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenTableModel;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.baumbach.Widerklage;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.GitterBeutelBeschraenkungen;
import eu.gronos.kostenrechner.view.KostenJPanel;
import eu.gronos.kostenrechner.view.menus.PopupMenuBuilder;

/**
 * Diese von {@link KostenJPanel} abgeleitete Klasse nimmt die
 * Oberflächenelemente für die Berechnung der Baumbachschen Formel auf.
 *
 * @author Peter Schuster (setrok)
 * @date 13.06.2018
 *
 */
public class StreitgenossenPanel extends KostenJPanel
		implements WerteLesbar<Streitgenossen>, StreitgenossenTabellenBehandelnd {

	private static final long serialVersionUID = 1265694381131490890L;
	private static final StreitgenossenPanel instance = new StreitgenossenPanel();
	private Kostenrechner kostenrechner;
	private boolean widerKlageAnzeigen;
	public JTable tabKlaeger;
	public BeteiligtenTableModel btmKlaeger;
	public JTable tabBeklagte;
	public BeteiligtenTableModel btmBeklagte;

	private final AbstractAction neuerKlaegerAction = new StreitgenossenBeteiligterNeuAction(this, Beteiligter.KLAEGER,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(31021));
	private final AbstractAction neuerBeklagterAction = new StreitgenossenBeteiligterNeuAction(this,
			Beteiligter.BEKLAGTE, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32021));
	private final AbstractAction neuerDrittwiderbeklagterAction = new StreitgenossenBeteiligterNeuAction(this,
			Beteiligter.DRITTWIDERBEKLAGTE, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33121));

	public JTable tabDrittwiderbeklagte;
	public BeteiligtenTableModel btmDrittwiderbeklagte;
	private JScrollPane scpDwbk;
	private JButton btnWiderklageAnzeigen;
	private JButton btnDrittwiderbeklagterNeu;
	private JButton btnKlDwblGesamtsch;
	private AbstractAction bkGesamtschuldnerAction = new StreitgenossenGesamtschuldnerBkAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32031));
	private final AbstractAction widerKlageAktivierenAction = new StreitgenossenWiderklageAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33001));
	private final AbstractAction klDwbkGesamtschAction = new StreitgenossenGesamtschuldnerDwbkAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33131));
	private final AbstractAction baumbachAction = new StreitgenossenBaumbachAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(34001));

	public final AbstractAction gesamtschuldnerschaftDwbkExtrahierenAction = new StreitgenossenGesamtschuldDwbkExtractAction(
			this, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39111));
	public final AbstractAction gesamtschuldnerschaftExtrahierenAction = new StreitgenossenGesamtschuldBkExtractAction(
			this, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39211));
	public AbstractAction xmlLesenAction = new StreitgenossenXJustizLesenAction(this, getKostenrechner(),
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(61031));
	public AbstractAction baumbachAssistentAction = new StreitgenossenAssistentAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(62051));

	private boolean wirtschaftlichIdentisch = false;
	public static final String FEHLER_BERECHNUNG = "Fehler bei der Berechnung! ";

	private static final String FEHLER_GESAMTSCHULD_NIX_LOESCHEN = "Einträge können nicht entfernt werden, wenn bereits eine Gesamtschuldnerschaft vorhanden ist.";
	public static final String FEHLER_GESAMTSCHULDNERSCHAFT = "Fehler beim Hinzufügen einer Gesamtschuldnerschaft";
	public static final String FEHLER_NIX_PARTEIEN = "Fehler: Keine Kläger und/oder Beklagten vorhanden";
	public static final String FEHLER_NIX_PARTEIEN_LANG = "Die Berechnung erfordert einen Kläger und mindestens einen Beklagten.";
	private static final WiderklageTableRendererEditor WIDERKLAGE_RENDERER_EDITOR = new WiderklageTableRendererEditor();
	public static final BeteiligtenTableRenderer BETEILIGTEN_RENDERER = new BeteiligtenTableRenderer();
	public static final int TAB_STREITGENOSSEN = 1;
	private ParsendUndBauend<Streitgenossen> pruefer = new StreitgenossenPruefer();

	/**
	 * Der Konstruktor ruft nur die Oberklasse auf.
	 * 
	 */
	private StreitgenossenPanel() {
		super();
	}

	/**
	 * @return gibt {@link #instance} als BaumbachPanel zurück.
	 */
	public static StreitgenossenPanel getInstance() {
		return instance;
	}

	/**
	 * Die Methode setzt die Oberflächenelemente für die Registerkarte 2
	 * Streitgenossen
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#buildPanel()
	 */
	@Override
	public void buildPanel() {
		setWiderKlageAnzeigen(false);

		leereWerte();

		JPanel pnKlaeger = createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(31000),
				new GridBagLayout());
		tabKlaeger = (JTable) beschrifter.beschrifte(new JTable(btmKlaeger),
				NameContainerSammlung.BESCHRIFTUNGEN.get(31011));
		setzeBeteiligtenRenderer(tabKlaeger);
		JScrollPane scpKlaeger = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabKlaeger),
				NameContainerSammlung.BESCHRIFTUNGEN.get(31010));
		entferneTastaturFalle(tabKlaeger);

		JPanel pnBeklagte = createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32000),
				new GridBagLayout());
		tabBeklagte = (JTable) beschrifter.beschrifte(new JTable(btmBeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(32011));
		setzeBeteiligtenRenderer(tabBeklagte);
		JScrollPane scpBeklagte = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabBeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(32010));
		entferneTastaturFalle(tabBeklagte);
		JButton btnKlgerNeu = ((BeschriebeneAktion) neuerKlaegerAction).toButton();

		JPanel pnWiderklage = createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33000),
				new GridBagLayout());
		tabDrittwiderbeklagte = (JTable) beschrifter.beschrifte(new JTable(btmDrittwiderbeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(33111));
		setzeBeteiligtenRenderer(tabDrittwiderbeklagte);
		scpDwbk = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabDrittwiderbeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(33110));
		entferneTastaturFalle(tabDrittwiderbeklagte);
		btnDrittwiderbeklagterNeu = ((BeschriebeneAktion) neuerDrittwiderbeklagterAction).toButton();

		JButton btnBeklagterNeu = ((BeschriebeneAktion) neuerBeklagterAction).toButton();

		btnKlDwblGesamtsch = ((BeschriebeneAktion) klDwbkGesamtschAction).toButton();
		btnWiderklageAnzeigen = ((BeschriebeneAktion) widerKlageAktivierenAction).toButton();

		JButton btnBkGesamtsch = ((BeschriebeneAktion) bkGesamtschuldnerAction).toButton();

		JButton btnBerechnen = ((BeschriebeneAktion) baumbachAction).toButton();

		// zum Panel hinzufügen
		pnKlaeger.add(scpKlaeger, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		pnKlaeger.add(btnKlgerNeu,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnWiderklage.add(scpDwbk, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		pnWiderklage.add(btnDrittwiderbeklagterNeu,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnWiderklage.add(btnKlDwblGesamtsch,
				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnWiderklage.add(btnWiderklageAnzeigen,
				GitterBeutelBeschraenkungen.getInstance(0, 4, 1, 1, GridBagConstraints.NONE, false));
		setWiderKlageAnzeigen(false);
		pnBeklagte.add(scpBeklagte,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		pnBeklagte.add(btnBeklagterNeu,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnBeklagte.add(btnBkGesamtsch,
				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		add(pnKlaeger, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.BOTH, false));
		add(pnWiderklage, GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
		add(pnBeklagte, GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 2, GridBagConstraints.BOTH, false));
		add(btnBerechnen, GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.HORIZONTAL, false));

		PopupMenuBuilder builder = new PopupMenuBuilder();

		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39100), tabKlaeger, new Integer[] { 1 },
				(BeschriebeneAktion) neuerKlaegerAction, (BeschriebeneAktion) klDwbkGesamtschAction,
				(BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39200), tabBeklagte, new Integer[] { 1 },
				(BeschriebeneAktion) neuerBeklagterAction, (BeschriebeneAktion) bkGesamtschuldnerAction,
				(BeschriebeneAktion) gesamtschuldnerschaftExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39300), tabDrittwiderbeklagte,
				new Integer[] { 1 }, (BeschriebeneAktion) neuerDrittwiderbeklagterAction,
				(BeschriebeneAktion) klDwbkGesamtschAction,
				(BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);

		logDauer();
	}

	/**
	 * Die Methode leert die Baumbach-Tabellen und füllt sie mit den Daten aus dem
	 * Untercontainer
	 * {@link eu.gronos.kostenrechner.model.baumbach.Streitgenossen}
	 * 
	 * @param container ein Untercontainer
	 *                  {@link eu.gronos.kostenrechner.model.baumbach.Streitgenossen}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.StreitgenossenTabellenBehandelnd#setzeWerte(eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer.Streitgenossen)
	 */
	@Override
	public void setzeWerte(final Streitgenossen container) {
		setWiderKlageAnzeigen(container.widerklage);
		setWirtschaftlichIdentisch(container.wirtschaftlichIdentisch);
		btmKlaeger = new BeteiligtenTableModel(Beteiligter.KLAEGER, isWiderKlageAnzeigen(), container.klaeger);
		tabKlaeger.setModel(btmKlaeger);
		btmDrittwiderbeklagte = new BeteiligtenTableModel(Beteiligter.DRITTWIDERBEKLAGTE, isWiderKlageAnzeigen(),
				container.drittwiderbeklagte);
		tabDrittwiderbeklagte.setModel(btmDrittwiderbeklagte);
		btmBeklagte = new BeteiligtenTableModel(Beteiligter.BEKLAGTE, isWiderKlageAnzeigen(), container.beklagte);
		tabBeklagte.setModel(btmBeklagte);
		Kostenrechner.getLogger().info("Fertig!");
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#streitgenossen}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.WerteLesbar#baueRueckgabewert()
	 */
	@Override
	public Streitgenossen baueRueckgabewert() {
		((StreitgenossenPruefer) pruefer)
				.setzeSonstigeWerte(new Widerklage(isWiderKlageAnzeigen(), isWirtschaftlichIdentisch()))
				.leseComponents(tabKlaeger, tabBeklagte, tabDrittwiderbeklagte).parseEingabe();
		return pruefer.getRueckgabewert();
	}

	/**
	 * Die Methode setWiderklageAnzeigen dient zum Aktivieren der Widerklagen. Bei
	 * allen TableModels widerKlageAnzeigen setzen, Dwbk-Tabelle und Schaltflächen
	 * wieder auf Visible setzen, eigene Schaltflächen ausblenden.
	 * 
	 * @param widerKlageAnzeigen der Wert für widerKlageAnzeigen, der gesetzt werden
	 *                           soll.
	 */
	public void setWiderKlageAnzeigen(boolean widerKlageAnzeigen) {
		this.widerKlageAnzeigen = widerKlageAnzeigen;
		if (scpDwbk != null)
			scpDwbk.setVisible(widerKlageAnzeigen);
		if (btnDrittwiderbeklagterNeu != null)
			btnDrittwiderbeklagterNeu.setVisible(widerKlageAnzeigen);
		if (btnKlDwblGesamtsch != null)
			btnKlDwblGesamtsch.setVisible(widerKlageAnzeigen);
		if (btnWiderklageAnzeigen != null)
			btnWiderklageAnzeigen.setVisible(!widerKlageAnzeigen);
	}

	/**
	 * Die Methode leert die Tabellen der BaumbachBeteiligten
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#leereWerte()
	 */
	@Override
	public void leereWerte() {
		btmKlaeger = new BeteiligtenTableModel(Beteiligter.KLAEGER, isWiderKlageAnzeigen(),
				new BaumbachBeteiligtenListe());
		btmDrittwiderbeklagte = new BeteiligtenTableModel(Beteiligter.DRITTWIDERBEKLAGTE, isWiderKlageAnzeigen());
		btmBeklagte = new BeteiligtenTableModel(Beteiligter.BEKLAGTE, isWiderKlageAnzeigen(),
				new BaumbachBeteiligtenListe());
		if (tabKlaeger != null)
			tabKlaeger.setModel(btmKlaeger);
		if (tabBeklagte != null)
			tabBeklagte.setModel(btmBeklagte);
		if (tabDrittwiderbeklagte != null)
			tabDrittwiderbeklagte.setModel(btmDrittwiderbeklagte);
	}

	/**
	 * Die Methode setzt die DefaultRenderer für Beteiligten-JTables
	 * (BeteiligtenRenderer und EuroRenderer) und bestimmt, dass sie den Viewport
	 * ausfüllen müssen.
	 * 
	 * @param table JTable für Beteiligtendaten
	 * 
	 */
	private void setzeBeteiligtenRenderer(JTable table) {
		table.setDefaultRenderer(Beteiligter.class, BETEILIGTEN_RENDERER);
		table.setDefaultEditor(Beteiligter.class,
				new BeteiligtenCellEditor(((BeteiligtenTableModel) table.getModel()).getBeteiligtenTyp()));
		// DefaultCellEditor(comboBox)
		table.setDefaultRenderer(Double.class, EURO_RENDERER);
		table.setDefaultEditor(Double.class, EURO_EDITOR);
		table.setDefaultRenderer(Boolean.class, WIDERKLAGE_RENDERER_EDITOR);
		table.setDefaultEditor(Boolean.class, WIDERKLAGE_RENDERER_EDITOR);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.getTableHeader().setDefaultRenderer(HEADER_RENDERER);
	}

	/**
	 * Die Methode prüft, ob die Tabellen für die BaumbachBeteiligten auf Kläger-
	 * und auf Beklagtenseite Werte enthalten.
	 * 
	 * @return true, wenn btmKlaeger und btmBeklagte Beteiligte enthalten; false,
	 *         wenn auch nur eine der beiden leer ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TabellenWertePruefend#pruefeTabellen()
	 */
	@Override
	public boolean pruefeTabellen() {
		return btmKlaeger.getRowCount() > 0 && btmBeklagte.getRowCount() > 0;
	}

	/**
	 * Die Methode dient dazu, eine Referenz auf das Hauptfenster zu übermitteln
	 * 
	 * @param kostenrechner
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#setKostenrechner(eu.gronos.kostenrechner.Kostenrechner)
	 */
	@Override
	public void setKostenrechner(Kostenrechner kostenrechner) {
		this.kostenrechner = kostenrechner;
	}

	/**
	 * Entfernt eine Zeile aus einer Beteiligten-Tabelle. Es wird zuvor geprüft, ob
	 * die entsprechende Tabelle bereits Gesamtschuldnerschaften enthält; dann ist
	 * ein Entfernen aus Gründen möglicher Verweise nicht möglicht. Es sei denn: die
	 * Gesamtschuldnerschaft selbst wird entfernt.
	 * 
	 * @param table die JTable, aus der die Zeile gelöscht werden soll
	 * @param model das BeteiligtenTableModel, aus der die Zeile gelöscht werden
	 *              soll.
	 * 
	 */
	public void entferneBeteiligtenEintrag(JTable table, BeteiligtenTableModel model) {
		if (model == null)
			return;
		if (model.getAllValues() != null && model.getAllValues().enthaeltGesamtschuldner()) {
			if (table.getSelectedRow() > -1 && !model.getRow(table.getSelectedRow()).isGesamtschuldnerschaft()) {
				FehlerHelper.zeigeFehler(FEHLER_GESAMTSCHULD_NIX_LOESCHEN,
						new IllegalArgumentException(FEHLER_GESAMTSCHULD_NIX_LOESCHEN));
				return;
			}
		} else if (model.getBeteiligtenTyp() == Beteiligter.KLAEGER && (btmDrittwiderbeklagte.getAllValues() != null
				&& btmDrittwiderbeklagte.getAllValues().enthaeltGesamtschuldner())) {
			if (table.getSelectedRow() > -1 && !model.getRow(table.getSelectedRow()).isGesamtschuldnerschaft()) {
				FehlerHelper.zeigeFehler(FEHLER_GESAMTSCHULD_NIX_LOESCHEN,
						new IllegalArgumentException(FEHLER_GESAMTSCHULD_NIX_LOESCHEN));
				return;
			}
		}
		((EintragEntfernenAktion) eintragEntfernenAktion).entferneTabellenEintrag(table, model);
	}

	/**
	 * @return gibt {@link #kostenrechner} als Kostenrechner zurück.
	 */
	public Kostenrechner getKostenrechner() {
		return kostenrechner;
	}

	/**
	 * @return gibt {@link #wirtschaftlichIdentisch} als boolean zurück.
	 */
	public boolean isWirtschaftlichIdentisch() {
		return wirtschaftlichIdentisch;
	}

	/**
	 * @param wirtschaftlichIdentisch d. {@link #wirtschaftlichIdentisch}, d.
	 *                                gesetzt werden soll als boolean
	 */
	public void setWirtschaftlichIdentisch(boolean wirtschaftlichIdentisch) {
		this.wirtschaftlichIdentisch = wirtschaftlichIdentisch;
	}

	/**
	 * @return gibt {@link #widerKlageAnzeigen} als boolean zurück.
	 */
	public boolean isWiderKlageAnzeigen() {
		return widerKlageAnzeigen;
	}

	/**
	 * Die Methode gibt das zu einem {@link Beteiligter#getTyp()} zugehörige
	 * {@link BeteiligtenTableModel} zurück.
	 * 
	 * @param typ {@link Beteiligter#KLAEGER}, {@link Beteiligter#BEKLAGTE} oder
	 *            {@link Beteiligter#DRITTWIDERBEKLAGTE}
	 * @return das dazu gehörige {@link BeteiligtenTableModel}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TableModelGebend#getTableModel(int)
	 */
	@Override
	public BeteiligtenTableModel getTableModel(int typ) {
		switch (typ) {
		case Beteiligter.KLAEGER:
			return btmKlaeger;
		case Beteiligter.DRITTWIDERBEKLAGTE:
			return btmDrittwiderbeklagte;
		case Beteiligter.BEKLAGTE:
			return btmBeklagte;
		default:
			return null;
		}
	}

	/**
	 * Die Methode dient dazu, dass eine {@link BeschriebeneAktion} den Focus auf
	 * die {@link JComponent} legen kann, die sie aufruft.
	 * 
	 * @return <code>true</code>, wenn der Aufruf wahrscheinlich Erfolg hat;
	 *         <code>false</code>, wenn er zum Scheitern verurteilt ist.
	 * 
	 * @see javax.swing.JComponent#requestFocusInWindow()
	 */
	@Override
	public boolean requestFocusInWindow() {
		getKostenrechner().getTabbedPane().setSelectedIndex(TAB_STREITGENOSSEN);
		return super.requestFocusInWindow();
	}

	@Override
	public AbstractAction getExtrahierenAktion(int beteiligtenTyp) throws IllegalArgumentException {
		if (Beteiligter.BEKLAGTE == beteiligtenTyp)
			return gesamtschuldnerschaftExtrahierenAction;
		else if (Beteiligter.DRITTWIDERBEKLAGTE == beteiligtenTyp)
			return gesamtschuldnerschaftDwbkExtrahierenAction;
		else
			throw new IllegalArgumentException(String.format("Unbekannter Beteiligtentyp: %d", beteiligtenTyp));
	}

}
// private JPopupMenu pmBeklagtenKontextmenu;
// private JPopupMenu pmKlaegerKontextmenu;
//private JPopupMenu pmDwbkKontextmenu;
// public static final String DRITTWIDERBEKLAGTE_POPUPLABEL =
// "Drittwiderbeklagte...";
// public static final String KLAEGER_POPUPLABEL = "Kläger...";
// public static final String BEKLAGTE_POPUPLABEL = "Beklagte...";
// GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.BOTH,
// false));
// GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH,
// false));
// GitterBeutelBeschraenkungen.getInstance(0, 3, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 4, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 4, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(2, 0, 2, 3, GridBagConstraints.BOTH,
// false));
// GitterBeutelBeschraenkungen.getInstance(2, 3, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(2, 4, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 5, 4, 1,
// GridBagConstraints.HORIZONTAL, false));

//	pmKlaegerKontextmenu = new JPopupMenu(KLAEGER_POPUPLABEL);
//	pmKlaegerKontextmenu.add(new JMenuItem(KLAEGER_POPUPLABEL));
//	pmKlaegerKontextmenu.addSeparator();
//	pmKlaegerKontextmenu.add(((BeschriebeneAktion) neuerKlaegerAction).toMenuItem());
//	pmKlaegerKontextmenu.add(((BeschriebeneAktion) klDwbkGesamtschAction).toMenuItem());
//	pmKlaegerKontextmenu.add(((BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction).toMenuItem());
//	pmKlaegerKontextmenu.add(((BeschriebeneAktion) eintragEntfernenAktion).toMenuItem());
//	tabKlaeger.addMouseListener(new KontextMouseListener(tabKlaeger, pmKlaegerKontextmenu));

//	pmBeklagtenKontextmenu = new JPopupMenu(BEKLAGTE_POPUPLABEL);
//	pmBeklagtenKontextmenu.add(new JMenuItem(BEKLAGTE_POPUPLABEL));
//	pmBeklagtenKontextmenu.addSeparator();
//	pmBeklagtenKontextmenu.add(((BeschriebeneAktion) neuerBeklagterAction).toMenuItem());
//	pmBeklagtenKontextmenu.add(((BeschriebeneAktion) bkGesamtschuldnerAction).toMenuItem());
//	pmBeklagtenKontextmenu.add(((BeschriebeneAktion) gesamtschuldnerschaftExtrahierenAction).toMenuItem());
//	pmBeklagtenKontextmenu.add(((BeschriebeneAktion) eintragEntfernenAktion).toMenuItem());
//	tabBeklagte.addMouseListener(new KontextMouseListener(tabBeklagte, pmBeklagtenKontextmenu));
//	pmDwbkKontextmenu = new JPopupMenu(DRITTWIDERBEKLAGTE_POPUPLABEL);
//	pmDwbkKontextmenu.add(new JMenuItem(DRITTWIDERBEKLAGTE_POPUPLABEL));
//	pmDwbkKontextmenu.addSeparator();
//	pmDwbkKontextmenu.add(((BeschriebeneAktion) neuerDrittwiderbeklagterAction).toMenuItem());
//	pmDwbkKontextmenu.add(((BeschriebeneAktion) klDwbkGesamtschAction).toMenuItem());
//	pmDwbkKontextmenu.add(((BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction).toMenuItem());
//	pmDwbkKontextmenu.add(((BeschriebeneAktion) eintragEntfernenAktion).toMenuItem());
//	tabDrittwiderbeklagte.addMouseListener(new KontextMouseListener(tabDrittwiderbeklagte, pmDwbkKontextmenu));
// new BeteiligtenRenderer(true);
// setTransferHandler(new XmlTransferHandler(this));
// "Kläger neu", "klaeger-neu", "<html><h1>Kläger neu</h1><p>Fügt einen neuen
// Kläger hinzu</p></p></html>",
// "Beklagter neu","beklagter-neu", "Fügt einen neuen (einzelnen) Beklagten
// hinzu.",
//"<html><h1>Beklagter neu</h1><p>"+ + "</p></p></html>"
// "Drittwiderbeklagter neu", "Fügt einen neuen Drittwiderbeklagten hinzu.",,
// "drittwiderbeklagter-neu"
//"<html><h1>Drittwiderbeklagter neu</h1><p>"
//+ "</p></p></html>"
//, "Gesamtschuldnerische Kl./Drwbk. neu",
//"Fügt eine gesamtschuldnerische Verurteilung auf Kläger-/Drittwiderbeklagtenseite hinzu.",
//"beklagte-gesamtschuldner"
// "<html><h1>Gesamtschuldnerische Kläger und Drittwiderbeklagte neu</h1><p>"
//+  + "<br>Vorher bitte die einzelnen Beteiligten hinzufügen.</p></html>"
//"Widerklagen aktivieren", "Aktiviert die Eingabeschnittstellen für Widerklagen", ,
//"widerklage-aktivieren"
//"<html><h1>Widerklage</h1><p>"+ "</p></html>"
//"Gesamtschuldnerische Beklagte neu",
//"Fügt eine gesamtschuldnerische Verurteilung zur Tabelle der Beklagten hinzu.", ,
//"gesamtschuldnerschaft-beklagte"
//"<html><h1>Gesamtschuldnerische Beklagten neu</h1><p>"+ "</p></html>"
//"Tenor ermitteln (Baumbach'sche Kostenformel)",
//"Erzeugt einen Kostentenor nach der Baumbach'schen Kostenformel.", , "baumbach-tenor"
// "Baumbach'schen Kostentenor berechnen",
//"<html><h1>Baumbach'schen Kostentenor berechnen</h1><p>"
//+ "</p></html>" 
//"Geht die Beteiligten einer Seite durch und guckt, ob sie z.T. gesamtschuldnerisch verurteilt wurden",
//"Gesamtschuldnerische Verurteilung ermitteln",, "extrahiere-gesamtschuld"
//"Gesamtschuldnerische Verurteilung (Widerbeklagte) ermitteln",, "extrahiere-widerklage-gesamtschuld"
//"Geht die Beteiligten einer Seite durch und guckt, ob sie z.T. gesamtschuldnerisch verurteilt wurden",
//"Beteiligte anlegen über Assistenten", "baumbach-assistent",
//"Der Assistent vereinfacht das Hinzufügen von Beteiligten durch eine geführte Abfrage der Daten.",
//"XJustiz importieren über Assistenten", "Liest einen XJustiz-Datensatz (XML) ein", "xjustiz",

// new JButton(neuerKlaegerAction);
// new JButton(neuerDrittwiderbeklagterAction);
// new JButton(neuerBeklagterAction);
// new JButton(klDwbkGesamtschAction);
// new JButton(widerKlageAktivierenAction);
// new JButton(bkGesamtschuldnerAction);
// new JButton(baumbachAction);
// new JMenuItem(neuerBeklagterAction));
// new JMenuItem(bkGesamtschuldnerAction));
// new JMenuItem(gesamtschuldnerschaftExtrahierenAction));
// new JMenuItem(eintragEntfernenAktion));
// new JMenuItem(neuerKlaegerAction));
// new JMenuItem(klDwbkGesamtschAction));
// new JMenuItem(gesamtschuldnerschaftDwbkExtrahierenAction));
// new JMenuItem(eintragEntfernenAktion));
// new JMenuItem(neuerDrittwiderbeklagterAction));
// new JMenuItem(klDwbkGesamtschAction));
// new JMenuItem(gesamtschuldnerschaftDwbkExtrahierenAction));
// new JMenuItem(eintragEntfernenAktion));
// table.setFillsViewportHeight(true);
// setWiderKlageAnzeigen(false);