/**
 * GebuehrenPanel.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.gebuehren;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.FocusEvent;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.TableModelEvent;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.AllzweckHorcher;
import eu.gronos.kostenrechner.controller.BeschriebeneAktion;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrNeuAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenAufstellungAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenBerechnungAufrufHorcher;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenBerechnungPruefer;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenSetzeRechnerTabelleAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenStreitwertHorcher;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenTabelleDruckenAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenTabelleKopierenAction;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.GebuehrenTableTextfieldHandler;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.WerteLesbar;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTableModel;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.gebuehren.StreitwertTableModel;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.GitterBeutelBeschraenkungen;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;
import eu.gronos.kostenrechner.view.KostenJPanel;
import eu.gronos.kostenrechner.view.menus.PopupMenuBuilder;

/**
 * Diese von {@link KostenJPanel} abgeleitete Klasse nimmt die
 * Oberflächenelemente für die Gebührenberechnung auf.
 *
 * @author Peter Schuster (setrok)
 * @date 13.06.2018
 *
 */
public class GebuehrenBerechnungPanel extends KostenJPanel
		implements WerteLesbar<GebuehrenBerechnung>, GebuehrenTableTextfieldHandler {
	private static final String SUMME_EUR = "Summe: %,.2f EUR";
	private static final long serialVersionUID = -8251865293845494552L;
	private static final GebuehrenBerechnungPanel instance = new GebuehrenBerechnungPanel();
	private Kostenrechner kostenrechner;
	private JTable tabStreitwertTabelle;
	private JFormattedTextField ftfStreitwert;
	public JTable tabGebuehrenTatbestaende;
	private JTextField tfGebuehrenSumme;
	public GebuehrenTableModel gttmRechner;

	private final AbstractAction neueGkgGebuehrAction = new GebuehrNeuAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(22021), TABLE_GEBUEHREN,
			GebuehrNeuAction.GKG_KEY);
			//GebuehrenGesetzesSammlung.getGkg().getVerzeichnis());
	private final AbstractAction neueRvgVerguetungAction = new GebuehrNeuAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(22031), TABLE_GEBUEHREN,
			GebuehrNeuAction.RVG_KEY);
			//GebuehrenGesetzesSammlung.getRvg().getVerzeichnis());
	private final AbstractAction gebuehrenAufstellungAction = new GebuehrenAufstellungAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(23031));
	public final AbstractAction normaleGebuehrenAction = new GebuehrenSetzeRechnerTabelleAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24221),
			GebuehrenSetzeRechnerTabelleAction.GEBUEHREN_STANDARD);
	public final AbstractAction saeumnisGebuehrenAction = new GebuehrenSetzeRechnerTabelleAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24211),
			GebuehrenSetzeRechnerTabelleAction.GEBUEHREN_SAEUMNIS);
	public final AbstractAction anerkenntnisGebuehrenAction = new GebuehrenSetzeRechnerTabelleAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24215),
			GebuehrenSetzeRechnerTabelleAction.GEBUEHREN_ANERKENNTNIS);
	private final AbstractAction mitInstanzGebuehrenAction = new GebuehrenSetzeRechnerTabelleAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24225),
			GebuehrenSetzeRechnerTabelleAction.GEBUEHREN_2INSTANZ);
	private final AbstractAction druckeSwTabelleAction = new GebuehrenTabelleDruckenAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24111));
	private final AbstractAction kopiereSwTabelleAction = new GebuehrenTabelleKopierenAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24101));
	private final AllzweckHorcher aktualisierungsHorcher = new GebuehrenStreitwertHorcher(this,
			((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(21001)).getTitle());

	public static final String FEHLER_DRUCKEN = "Fehler beim Drucken";
	public static final int FTF_STREITWERT = 5;
	public static final int TABLE_GEBUEHREN = 8;
	public static final int TAB_GEBUEHRENRECHNER = 3;
	// private static final int TAB_GEBUEHRENRECHNER = 3;
	private JLabel lblStreitwert;
	private ParsendUndBauend<GebuehrenBerechnung> pruefer = new GebuehrenBerechnungPruefer();
	private GebuehrenBerechnungAufrufHorcher aufrufHorcher;

	/**
	 * Der Konstruktor ruft nur die Oberklasse auf.
	 * 
	 */
	private GebuehrenBerechnungPanel() {
		super();
		//Kostenrechner.getLogger().info("Starte Panel.");
	}

	/**
	 * Das Panel ist als Singelton ausgestatltet. Diese Methode gibt die alleinige
	 * Instanz zurück.
	 * 
	 * @return gibt {@link #instance} als GebuehrenBerechnungPanel zurück.
	 */
	public static GebuehrenBerechnungPanel getInstance() {
		return instance;
	}

	/**
	 * Die Methode fuellt den GebuehrenRechnerPanel.
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#buildPanel()
	 */
	@Override
	public void buildPanel() {
		JPanel pnStreitwert = super.createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(21000),
				new GridBagLayout());
		lblStreitwert = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(21001));
		lblStreitwert.setHorizontalAlignment(SwingConstants.RIGHT);
		setFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT, (JFormattedTextField) beschrifter
						.beschrifte(new JFormattedTextField(KostenJPanel.FORMATTER), NameContainerSammlung.BESCHRIFTUNGEN.get(21005)));
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT).setColumns(10);
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT).setText("0");
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT)
				.addFocusListener(HinzufuegenDialog.BEI_FOCUS_MARKIEREN);
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT).setHorizontalAlignment(JTextField.RIGHT);
		lblStreitwert.setLabelFor(getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT));

		// JScrollpane mit Gebührentabelle, die sich mit dem ftf synchronisiert
		JLabel lblStreitwertTabelle = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(21010));
		//StreitwertTableModel swtm = new StreitwertTableModel();
		//JTable tbSw = new JTable();
		setTabStreitwertTabelle((JTable) beschrifter.beschrifte(new JTable(), NameContainerSammlung.BESCHRIFTUNGEN.get(21021)));//swtm
		setAufrufHorcher(new GebuehrenBerechnungAufrufHorcher());
		getTabStreitwertTabelle().setDefaultRenderer(Long.class, KostenJPanel.EURO_RENDERER);
		getTabStreitwertTabelle().setDefaultRenderer(Double.class, KostenJPanel.EURO_RENDERER);
		getTabStreitwertTabelle().getTableHeader().setDefaultRenderer(KostenJPanel.HEADER_RENDERER);
		getTabStreitwertTabelle().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		entferneTastaturFalle(getTabStreitwertTabelle());
		JScrollPane scpStreitwertTabelle = (JScrollPane) beschrifter
				.beschrifte(new JScrollPane(getTabStreitwertTabelle()), NameContainerSammlung.BESCHRIFTUNGEN.get(21020));
		lblStreitwertTabelle.setLabelFor(scpStreitwertTabelle);

		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT).addFocusListener(aktualisierungsHorcher);
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT).addKeyListener(aktualisierungsHorcher);
		getTabStreitwertTabelle().addFocusListener(aktualisierungsHorcher);
		getTabStreitwertTabelle().addMouseListener(aktualisierungsHorcher);
		getTabStreitwertTabelle().addKeyListener(aktualisierungsHorcher);
		getTabStreitwertTabelle().getSelectionModel().addListSelectionListener(aktualisierungsHorcher);

		// JScrollpane mit JTable mit gttm + JLabel
		JPanel pnGebuehren = super.createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(22000),
				new GridBagLayout());
		JLabel lblGebuehrenTatbestaende = (JLabel) beschrifter
				.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(22001));
		tabGebuehrenTatbestaende = (JTable) beschrifter.beschrifte(new JTable(), NameContainerSammlung.BESCHRIFTUNGEN.get(22011));
		leereGebuehrenTabellen();
		entferneTastaturFalle(tabGebuehrenTatbestaende);
		setzeGebuehrenTabelleAussehen(tabGebuehrenTatbestaende);
		JScrollPane scpGebuehrenTatbestaende = (JScrollPane) beschrifter
				.beschrifte(new JScrollPane(tabGebuehrenTatbestaende), NameContainerSammlung.BESCHRIFTUNGEN.get(22010));
		lblGebuehrenTatbestaende.setLabelFor(scpGebuehrenTatbestaende);

		// JButton Gebühr hinzufügen
		JButton btnNeueVerguetung = ((BeschriebeneAktion) neueRvgVerguetungAction).toButton();
		JButton btnNeueGebuehr = ((BeschriebeneAktion) neueGkgGebuehrAction).toButton();

		// JTextField Summe berechnen
		tfGebuehrenSumme = (JTextField) beschrifter.beschrifte(new JTextField(), NameContainerSammlung.BESCHRIFTUNGEN.get(23005));
		tfGebuehrenSumme.setEditable(false);
		tfGebuehrenSumme.setHorizontalAlignment(SwingConstants.RIGHT);
		JButton btnSummeBerechnen = ((BeschriebeneAktion) gebuehrenAufstellungAction).toButton();

		JPanel pnAufstellung = super.createPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(23000),
				new GridBagLayout());

		errechneGebuehrenSumme();

		// Der JPanel hinzufügen
		pnStreitwert.add(lblStreitwert,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnStreitwert.add(getFormattedTextField(FTF_STREITWERT),
				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnStreitwert.add(lblStreitwertTabelle,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnStreitwert.add(scpStreitwertTabelle,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, true));
		add(pnStreitwert, GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 3, GridBagConstraints.BOTH, true));

		pnGebuehren.add(lblGebuehrenTatbestaende,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnGebuehren.add(scpGebuehrenTatbestaende,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 2, 1, GridBagConstraints.BOTH, true));
		pnGebuehren.add(btnNeueGebuehr,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		pnGebuehren.add(btnNeueVerguetung,
				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		add(pnGebuehren, GitterBeutelBeschraenkungen.getInstance(0, 3, 2, 3, GridBagConstraints.BOTH, true));

		pnAufstellung.add(tfGebuehrenSumme,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
		pnAufstellung.add(btnSummeBerechnen,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 2, 1, GridBagConstraints.HORIZONTAL, false));
		add(pnAufstellung, GitterBeutelBeschraenkungen.getInstance(0, 6, 2, 2, GridBagConstraints.HORIZONTAL, false));

		// Kontextmenü für Streitwerttabelle
		PopupMenuBuilder builder = new PopupMenuBuilder();
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24100), getTabStreitwertTabelle(),
				new Integer[] { 1 }, (BeschriebeneAktion) druckeSwTabelleAction,
				(BeschriebeneAktion) kopiereSwTabelleAction);

		// Kontextmenü für gttm
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(24200), tabGebuehrenTatbestaende,
				new Integer[] { 1 }, (BeschriebeneAktion) neueGkgGebuehrAction,
				(BeschriebeneAktion) neueRvgVerguetungAction, (BeschriebeneAktion) saeumnisGebuehrenAction,
				(BeschriebeneAktion) anerkenntnisGebuehrenAction, (BeschriebeneAktion) normaleGebuehrenAction,
				(BeschriebeneAktion) mitInstanzGebuehrenAction, (BeschriebeneAktion) eintragEntfernenAktion);
		
		logDauer();
	}

	/**
	 * Die Methode leert alle Tabellen und sonstige Oberflächenelemente. Damit kann
	 * man dann die Registerkarte initialisieren oder auf den Anfangszustand
	 * zurücksetzen.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.WerteLesbar#leereWerte()
	 */
	@Override
	public void leereWerte() {
		leereGebuehrenTabellen();
	}

	/**
	 * Die Methode leert die GebuehrenRechnerTabelle, füllt sie mit den Gebühren aus
	 * {@link eu.gronos.kostenrechner.model.gebuehren.GebuehrenBerechnung}
	 * und setzt auch den Streitwert entsprechend.
	 * 
	 * @param container ein
	 *                  {@link eu.gronos.kostenrechner.model.gebuehren.GebuehrenBerechnung}
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#setzeWerte(Streitgenossen,
	 *      java.lang.Object)
	 */
	public void setzeWerte(final GebuehrenBerechnung container) {
		/* Erst den Streitwert eintragen und die Tabelle springen lassen */
		getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT)
				.setText(Long.toString((long) container.streitwert));
		aktualisierungsHorcher.focusLost(
				new FocusEvent(getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT), FocusEvent.FOCUS_LOST));
		/* Dann die Tatbestände in die Gebührentabelle eintragen */
		if (gttmRechner == null) {
			gttmRechner = new GebuehrenTableModel(container.gebuehren);
			gttmRechner.addTableModelListener(aktualisierungsHorcher);
			tabGebuehrenTatbestaende.setModel(gttmRechner);
		} else {
			gttmRechner.clear();
			gttmRechner.addAll(container.gebuehren);
		}
	}

	/**
	 * Die Methode dient dazu, eine Referenz auf das Hauptfenster zu übermitteln
	 * 
	 * @param kostenrechner
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#setKostenrechner(eu.gronos.kostenrechner.Kostenrechner)
	 */
	@Override
	public void setKostenrechner(Kostenrechner kostenrechner) {
		this.kostenrechner = kostenrechner;
	}

	/**
	 * Die Methode addiert die Gebühren der Tabelle.
	 * 
	 */
	public void errechneGebuehrenSumme() {
		String streitwert = "";
		double gebuehrenSumme = 0.0;
		if (getFormattedTextField(FTF_STREITWERT) != null && pruefeTabellen()) {
			try {
				streitwert = getFormattedTextField(FTF_STREITWERT).getText();
				gebuehrenSumme = gttmRechner.getValues().errechneGebuehrenSummeMitSteuer(
						longAusFormatter(getFormattedTextField(FTF_STREITWERT), getLabelText(FTF_STREITWERT)));
			} catch (NumberFormatException e) {
				FehlerHelper.zeigeFehler("Falsches Eingabeformat für den Streitwert: \"" + streitwert + "\"!\n ", e);
			}
		}
		String text = String.format(SUMME_EUR, gebuehrenSumme);
		if (tfGebuehrenSumme != null)
			tfGebuehrenSumme.setText(text);
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#gebuehrenBerechnung}
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#baueRueckgabewert()
	 */
	@Override
	public GebuehrenBerechnung baueRueckgabewert() throws NumberFormatException, NullPointerException {
		pruefer.leseComponents(getFormattedTextField(FTF_STREITWERT), tabGebuehrenTatbestaende).parseEingabe();
		return pruefer.getRueckgabewert();
	}

	/**
	 * Die Methode prüft, ob die Gebührentabelle Werte enthält.
	 * 
	 * @return true, wenn Gebührentatbestände in der Tabelle sind.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.PanelTableModelHandler#pruefeTabellen()
	 */
	@Override
	public boolean pruefeTabellen() {
		return getTableModel(TABLE_GEBUEHREN) != null && getTableModel(TABLE_GEBUEHREN).getRowCount() > 0;
	}

	/**
	 * @return gibt {@link #tabStreitwertTabelle} als JTable zurück.
	 */
	public JTable getTabStreitwertTabelle() {
		return tabStreitwertTabelle;
	}

	/**
	 * @param tabStreitwertTabelle d. {@link #tabStreitwertTabelle}, d. gesetzt
	 *                             werden soll als JTable
	 */
	public void setTabStreitwertTabelle(JTable tabStreitwertTabelle) {
		this.tabStreitwertTabelle = tabStreitwertTabelle;
	}

	/**
	 * Die Methode dient dazu, das passende {@link JFormattedTextField}
	 * zurückzuliefern
	 * 
	 * @param index ein Bezeichner für das {@link JFormattedTextField}, hier nur
	 *              {@link #TABLE_FTF_GEBUEHREN}
	 * @return gibt das gewählte {@link JFormattedTextField} zurück, hier nur
	 *         {@link #ftfStreitwert}.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.PanelTextFieldHandler#getFormattedTextField(int)
	 */
	@Override
	public JFormattedTextField getFormattedTextField(int index) {
		return ftfStreitwert;
	}

	/**
	 * Die Methode soll den Text des zum {@link JFormattedTextField} gehörenden
	 * {@link JLabel}s zurückgeben.
	 * 
	 * @param index ein Bezeichner für das {@link JFormattedTextField}, hier nur
	 *              {@link GebuehrenBerechnungPanel#FTF_STREITWERT}.
	 * @return {@link JLabel#getText()} als String
	 * @see eu.gronos.kostenrechner.interfaces.PanelTextFieldHandler#getLabelText(int)
	 */
	@Override
	public String getLabelText(int index) {
		if (lblStreitwert != null)
			return lblStreitwert.getText();
		else
			return "";
	}

	/**
	 * @return gibt {@link #kostenrechner} als Kostenrechner zurück.
	 */
	public Kostenrechner getKostenrechner() {
		return kostenrechner;
	}

	/**
	 * Die Methode dient dazu, das passende {@link GebuehrenTableModel}
	 * zurückzuliefern
	 * 
	 * @param index ein Bezeichner für das {@link GebuehrenTableModel}, hier nur
	 *              {@link #TABLE_FTF_GEBUEHREN}
	 * @return das gewählte {@link GebuehrenTableModel}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.PanelTableModelHandler#getTableModel(int)
	 */
	@Override
	public GebuehrenTableModel getTableModel(int index) {
		if (index != TABLE_GEBUEHREN && Kostenrechner.debug) {
			IllegalArgumentException ie = new IllegalArgumentException(
					String.format("Fehler: index war nicht TABLE_GEBUEHREN, sondern: '%d'!", index));
			FehlerHelper.zeigeFehler(ie.getLocalizedMessage(), ie);
		}
		return gttmRechner;
	}

	/**
	 * Die Methode dient dazu, dass eine {@link BeschriebeneAktion} den Focus auf
	 * die {@link JComponent} legen kann, die sie aufruft.
	 * 
	 * @return <code>true</code>, wenn der Aufruf wahrscheinlich Erfolg hat;
	 *         <code>false</code>, wenn er zum Scheitern verurteilt ist.
	 * 
	 * @see javax.swing.JComponent#requestFocusInWindow()
	 */
	@Override
	public boolean requestFocusInWindow() {
		getKostenrechner().getTabbedPane().setSelectedComponent(this);
		return super.requestFocusInWindow();
	}

	/**
	 * @return gibt {@link #aufrufHorcher} als {@link GebuehrenBerechnungAufrufHorcher} zurück.
	 */
	public GebuehrenBerechnungAufrufHorcher getAufrufHorcher() {
		return aufrufHorcher;
	}

	/**
	 * Die Methode schreibt das {@link StreitwertTableModel} in die {@link #getTabStreitwertTabelle()}
	 */
	public void initStreitwertTable() {
		getTabStreitwertTabelle().setModel(new StreitwertTableModel());
		getTabStreitwertTabelle().tableChanged(new TableModelEvent(getTabStreitwertTabelle().getModel()));
	}


	/**
	 * Die Methode leert die GebuehrenRechnerTabelle oder füllt sie mit
	 * üblicherweise anfallenden Gebühren, lässt aber die Listener in Ruhe.
	 */
	private void leereGebuehrenTabellen() {
		GebuehrenVerzeichnis values = new GebuehrenVerzeichnis();
		if (gttmRechner == null) {
			gttmRechner = new GebuehrenTableModel(values);
			gttmRechner.addTableModelListener(aktualisierungsHorcher);
			tabGebuehrenTatbestaende.setModel(gttmRechner);
		} else {
			gttmRechner.clear();
			gttmRechner.addAll(values);
		}

	}

	/**
	 * @param index         ein Bezeichner für das {@link JFormattedTextField}, hier
	 *                      nur {@link #TABLE_FTF_GEBUEHREN}
	 * @param ftfStreitwert d. {@link #ftfStreitwert}, d. gesetzt werden soll als
	 *                      JFormattedTextField
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.PanelTextFieldHandler#setFormattedTextField(int,
	 *      javax.swing.JFormattedTextField)
	 */
	private void setFormattedTextField(int index, JFormattedTextField ftfStreitwert) {
		this.ftfStreitwert = ftfStreitwert;
	}

	/**
	 * @param aufrufHorcher d. {@link #aufrufHorcher}, d. gesetzt werden soll als {@link GebuehrenBerechnungAufrufHorcher}.
	 */
	private void setAufrufHorcher(GebuehrenBerechnungAufrufHorcher aufrufHorcher) {
		this.aufrufHorcher = aufrufHorcher;
	}

}

/*
 * PanelTableModelHandler<GebuehrenTatbestandTableModel>, PanelTextFieldHandler
 */
// lblStreitwert.setDisplayedMnemonic(KeyEvent.VK_S);
// lblStreitwert.setDisplayedMnemonicIndex(0);// 20);
// lblStreitwertTabelle.setDisplayedMnemonic(KeyEvent.VK_W);
// lblGebuehrenTatbestaende.setDisplayedMnemonic(KeyEvent.VK_G);
// tfGebuehrenSumme.requestFocus();
// lblStreitwert.getText()Long.parseLong(streitwert)
// e.printStackTrace();
// private static final String LBL_STREITWERT_TEXT = "Bitte geben Sie den
// Streitwert ein:";
// private static final String STREITWERTE_POPUPLABEL = "Streitwerte...";
// private JPopupMenu pmStreitwert;
// private JPopupMenu pmGttmRechner;
// LBL_STREITWERT_TEXT);
// public static final String RECHNER_POPUPLABEL = "Gebührentatbestände...";
// new JLabel(LBL_STREITWERT_TEXT, SwingConstants.RIGHT);
// new JLabel("Oder wählen Sie einen Streitwert aus:");
// new JLabel("Bitte geben Sie die angefallenen Gebühren an:");
// setzeGebuehrenRechnerTabelle(0);
// lblGebuehrenTatbestaende.setDisplayedMnemonicIndex(33);
// new JButton(neueRvgVerguetungAction);
// new JButton(neueGkgGebuehrAction);
// new JButton(gebuehrenAufstellungAction);
// add(lblStreitwert, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// add(getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT),
// GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// add(tfGebuehrenSumme,
// GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(0, 4, 2, 1, GridBagConstraints.BOTH,
// true));
// GitterBeutelBeschraenkungen.getInstance(0, 5, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// GitterBeutelBeschraenkungen.getInstance(1, 5, 1, 1,
// GridBagConstraints.HORIZONTAL, false));
// add(btnSummeBerechnen,
// GitterBeutelBeschraenkungen.getInstance(0, 6, 2, 1,
// GridBagConstraints.HORIZONTAL, false));
// pmStreitwert = new JPopupMenu(STREITWERTE_POPUPLABEL);
// pmStreitwert.add(new JMenuItem(STREITWERTE_POPUPLABEL));
// pmStreitwert.addSeparator();
// pmStreitwert.add(new JMenuItem(druckeSwTabelleAction));
// pmStreitwert.add(new JMenuItem(kopiereSwTabelleAction));
// getTabStreitwertTabelle().addMouseListener(new
// KontextMouseListener(getTabStreitwertTabelle(), pmStreitwert));
// gttmRechner != null
// pmGttmRechner = new JPopupMenu(RECHNER_POPUPLABEL);
// pmGttmRechner.add(new JMenuItem(RECHNER_POPUPLABEL));
// pmGttmRechner.addSeparator();
// pmGttmRechner.add(new JMenuItem(neueGkgGebuehrAction));
// pmGttmRechner.add(new JMenuItem(neueRvgVerguetungAction));
// pmGttmRechner.add(new JMenuItem(saeumnisGebuehrenAction));
// pmGttmRechner.add(new JMenuItem(anerkenntnisGebuehrenAction));
// pmGttmRechner.add(new JMenuItem(normaleGebuehrenAction));
// pmGttmRechner.add(new JMenuItem(mitInstanzGebuehrenAction));
// pmGttmRechner.add(new JMenuItem(eintragEntfernenAktion));
// tabGebuehrenTatbestaende.addMouseListener(new
// KontextMouseListener(tabGebuehrenTatbestaende, pmGttmRechner));
// new GebuehrenRvgVerguetungAction
//"Neue Vergütung nach RVG", "<html><h1>Neue Gebühr</h1><p>Fügt einen neuen Vergütungstatbestand nach RVG in die Tabelle für die Gebühren zur freien Berechnung ein.</p></html>",, "gebuehr-rvg"/* GebuehrenNeueGebuehrAction */
/*
 * VerguetungsVerzeichnis.getAllGebuehrenTatbestaende() new
 * VerguetungsVerzeichnis()
 */
/* KostenVerzeichnis.getAllGebuehrenTatbestaende() new KostenVerzeichnis() */
//	 "Neue Gebühr nach GKG","gebuehr-gkg", 
//		"<html><h1>Neue Gebühr</h1><p>Fügt einen neuen Gebührentatbestand nach GKG in die Tabelle für die Gebühren zur freien Berechnung ein.</p></html>",
//			"Ausführliche Gebührenaufstellung erstellen", "gebuehren-aufstellung",
//			"<html><h1>Ausführliche Gebührenaufstellung erstellen</h1><p>Erstellt eine Aufstellung der Gebühren und ihrer Summe zu dem ausgewählten Streitwert.</p></html>",

//			"Gebühren im Normalfall","normal-gebuehren", 
//			"<html><h1>Gebühren im Normalfall</h1><p>Leert die Tabelle mit den Gebührentatbeständen des Rechners und <br>füllt sie mit der Konstellation eines normalen erstinstanzlichen Verfahrens</p></html>",
//			"Gebühren bei Säumnis","saeumnis-gebuehren", 
//			"<html><h1>Gebühren bei Säumnis</h1><p>Leert die Tabelle mit den Gebührentatbeständen des Rechners und <br>füllt sie mit der Konstellation Gebühren bei Säumnis</p></html>",
//			"Gebühren bei Anerkenntnis","anerkenntnis-gebuehren", 
//			"<html><h1>Gebühren bei Anerkenntnis</h1><p>Leert die Tabelle mit den Gebührentatbeständen des Rechners und <br>füllt sie mit der Konstellation Gebühren bei Anerkenntnis</p></html>",
//			"Gebühren einschließlich Berufung",
//			"<html><h1>Gebühren einschließlich Berufung</h1><p>Leert die Tabelle mit den Gebührentatbeständen des Rechners und <br>füllt sie mit der Konstellation eines normalen erst- und zweitinstanzlichen Verfahrens</p></html>",
//			"mit-instanz-gebuehren", 
//			"Tabelle drucken","print-table", 
//			"<html><h1>Tabelle drucken</h1><p>Druckt den Inhalt der Streitwerttabelle</p></html>",
//			"Streitwerttabelle kopieren","copy-table", 
//			"<html><h1>Streitwerttabelle kopieren</h1><p>Kopiert den Inhalt der Streitwerttabelle in die Zwischenablage</p></html>",

//		setzeGebuehrenRechnerTabelle(0);

// .setSelectedIndex(TAB_GEBUEHRENRECHNER);
//		List<GebuehrenTatbestand>
// ArrayList<GebuehrenTatbestand>
// ArrayList<GebuehrenTatbestand>();
// lblStreitwert.getText());Long.parseLong(getFtfStreitwert().getText());
// toArray();
// GebuehrenBerechnung container = new GebuehrenBerechnung();
// container.streitwert = 0L;
// container.gebuehren = new GebuehrenVerzeichnis();
// try {
// container.streitwert =
// longAusFormatter(getFormattedTextField(FTF_STREITWERT),
// getLabelText(FTF_STREITWERT));
// } catch (NumberFormatException e) {
// new FehlerDialog(FehlerDialog.FEHLER_BLANKO, e.getLocalizedMessage(),
// e).showDialog();
// }
// container.gebuehren = gttmRechner.getAllValues();
// return container;