/*
 * FontHelferForComponent.java
 * eu.gronos.kostenrechner.view.helpanderror (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.helpanderror;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.text.AttributeSet;
import javax.swing.text.StyleConstants;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.view.FontHelfer;

/**
 * Die Klasse erweitert {@link FontHelfer} um Abfragen aus einer
 * {@link JComponent}. Man kann Fonts suchen, Formatierungen setzen und die
 * Fenstergröße nach Schriftgröße zu setzen.
 *
 * @author Peter Schuster (setrok)
 * @date 19.07.2019
 *
 */
public class FontHelferForComponent extends FontHelfer {

	private final JComponent component;
	private final Font normal;
	private final Font fett;

	public FontHelferForComponent(JComponent component) {
		this.component = component;
		normal = component.getFont();
		fett = component.getFont().deriveFont(Font.BOLD);
		Kostenrechner.getLogger().info(String.format("Name: %s, Größe: %d", normal.getFontName(), normal.getSize()));
	}

	/**
	 * Zusehen, dass die {@link JComponent} nicht breiter als maxSpalten Zeichen und
	 * nicht höher als maxZeilen Zeilen wird.
	 * 
	 * @param attrib     ein {@link AttributeSet}
	 * @param maxSpalten Spaltenzahl für die Berechnung
	 * @param maxZeilen  Zeilenanzahl für die Berechnung
	 */
	public void setPreferredSizeFromAttributeSet(AttributeSet attrib, int maxSpalten, int maxZeilen) {
		Dimension maximumSize = component.getMaximumSize();
		Dimension preferredSize = component.getPreferredSize();
		calculatePreferredSizeFrom(maxSpalten, maxZeilen, getCharDimension(attrib), maximumSize, preferredSize);
		component.setPreferredSize(preferredSize);
	}

	/**
	 * @return {@link JComponent#getFontMetrics(Font)}
	 */
	public FontMetrics getFontMetrics(AttributeSet attrib) {
		return component.getFontMetrics(getFontForName(StyleConstants.getFontFamily(attrib)));
	}

	/**
	 * Die Methode dient dazu, die Breite eines Zeichens ('n') und die
	 * Standardzeilenhöhe zu ermitteln
	 * 
	 * @param attrib ein {@link AttributeSet}
	 * @return Zeichenbreite und Zeilenhöhe in Pixeln als {@link Dimension}
	 */
	private Dimension getCharDimension(AttributeSet attrib) {
		FontMetrics fontMetrics = component.getFontMetrics(getFontForName(StyleConstants.getFontFamily(attrib)));
		return getCharDimension(fontMetrics);
	}

	/*
	 * Die Methode definiert erstmal den grundlegenden Style ("base"), wobei diese
	 * direkt ins übergebene {@link StyledDocument} geschrieben werden. Zusätzlich
	 * wird der Style noch zurückgegeben.
	 * 
	 * @param doc ein {@link StyledDocument}
	 * @return ein {@link Style} mit den grundlegenden Formatierungen
	 * @see #createBaseStyle(StyledDocument)
	 */
	//public Style createBaseStyle(StyledDocument doc) {
		//final Font font = component.getFont();
		//return createBaseStyle(doc, font);
	//}

	/**
	 * Wenn man nicht nachhilft, ist ein {@link JLabel} schon einmal zu klein, im
	 * Sinne von nicht breit genug.
	 * 
	 * @param label         ein {@link JLabel}
	 * @param multiplikator die Breite, in Anzahl von Buchstaben gemessen.
	 */
	public void sizeLabel(JLabel label, int multiplikator) {
		Dimension dim = this.getCharDimension(component.getFontMetrics(normal));
		dim.setSize(dim.width * multiplikator, label.getPreferredSize().height);
		label.setPreferredSize(dim);
	}

	/**
	 * Die Methode setzt die Schriftart des {@link JLabel}s auf {@link Font#BOLD}.
	 * 
	 * @param label das zu verändernde {@link JLabel}
	 * @return den {@link Font} in {@link Font#BOLD}
	 */
	public Font boldLabel(JLabel label) {
		label.setFont(fett);
		return fett;
	}
}
