/*
 * BeschriftungApplyer.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.beschriftungen;

import java.awt.Component;
import java.awt.Container;
import java.awt.Window;
import java.util.List;

import javax.accessibility.AccessibleContext;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.border.TitledBorder;

import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.controller.BooleanConfigAction;
import eu.gronos.beschriftungen.interfaces.Bebildert;
import eu.gronos.beschriftungen.interfaces.Vorsorgend;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.BildOhneTitel;
import eu.gronos.beschriftungen.model.Gerichtet;
import eu.gronos.beschriftungen.model.GitterBeutelBeschraenkungen;
import eu.gronos.beschriftungen.model.IntervallUnBeschriftung;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.LangVorsorgeBeschriftung;
import eu.gronos.beschriftungen.model.NameContainer;
import eu.gronos.beschriftungen.model.UnBeschriftung;
import eu.gronos.beschriftungen.model.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;

/**
 * Neue Klasse, die Anwendungen für {@link NameContainer} bereithält. Mit
 * {@link #beschrifte(Component, NameContainer)} kann man einen
 * {@link NameContainer} auf eine {@link Component} anwenden. Mit
 * {@link #changeTo(Vorsorgend, int)} kann man bei einer
 * {@link Vorsorgend}-Klasse Title und ShortDescription wechseln.
 *
 * @author Peter Schuster (setrok)
 * @date 01.02.2020
 *
 */
public class ComponentBeschrifter {

	public static int MAX_ZEICHEN_PRO_ZEILE = 40;

	/**
	 * Neue Methode setzt in der Basisklasse {@link NameContainer} nur
	 * comp.setName(), bei einer {@link Beschriftung} weiter den Text/Title/Label
	 * und die shortDescription/toolTip.
	 * 
	 * holt sich bei {@link UnBeschriftung} aus {@link UnBeschriftung#getRef()} den
	 * Title und die shortDescription und setzt die auch.
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit der {@link NameContainer}
	 * @throws IllegalArgumentException wenn {@link JComponent} null ist
	 */
	public Component beschrifte(Component das, NameContainer damit) throws IllegalArgumentException {
		if (das == null)
			throw new IllegalArgumentException("Component darf nicht null sein!");
		das.setName(damit.getId());
		if (damit instanceof Beschriftung) {
			deployTitle(das, (Beschriftung) damit);
			deployShortDescription(das, (Beschriftung) damit, null);
			deployMnemonic(das, (Beschriftung) damit);
			if (damit instanceof LangBeschriftung) {
				deployActionCommandKey(das, (LangBeschriftung) damit);
				deployImage(das, (LangBeschriftung) damit);
				deployAcceleratorKey(das, (LangBeschriftung) damit);
				deployLongDescription(das, (LangBeschriftung) damit);
			}
		} else if (damit instanceof UnBeschriftung) {
			deployAccessibleName(das, createCompleteTitle((UnBeschriftung) damit));
			deployShortDescription(das, ((UnBeschriftung) damit).getRef(), (UnBeschriftung) damit);
			if (damit instanceof BildOhneTitel)
				deployImage(das, (Bebildert) damit);
			if (damit instanceof IntervallUnBeschriftung)
				deployIntervall(das, (IntervallUnBeschriftung) damit);
		}
		if (damit.richter != null) {
			deployAlignment(das, damit.richter);
		}
		return das;
	}

	/**
	 * Neue Methode setzt in der Basisklasse {@link NameContainer} nur
	 * comp.setName(), bei einer {@link Beschriftung} weiter den Text/Title/Label
	 * und die shortDescription/toolTip, holt sich bei {@link UnBeschriftung} aus
	 * {@link UnBeschriftung#getRef()} den Title und die shortDescription und setzt
	 * die auch und fügt das ganze dann noch in einen {@link Container} ein.
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit der {@link NameContainer}
	 * @param dahin der aufnehmende {@link Container}
	 * @throws IllegalArgumentException wenn {@link JComponent} oder
	 *                                  {@link Container} null ist
	 */
	public Component beschrifteVergittert(Component das, NameContainer damit, Container dahin) {
		if (dahin == null)
			throw new IllegalArgumentException("Container darf nicht null sein!");
		beschrifte(das, damit);
		add(das, damit, dahin);
		return das;
	}

	/**
	 * Die Methode beschriftet sowohl ein {@link JLabel} als auch eine weitere
	 * {@link Component}, fügt beide dem {@link Container} hinzu und markiert das
	 * {@link JLabel} als zur {@link Component} gehörig mit
	 * {@link JLabel#setLabelFor(Component)}.
	 * 
	 * @param dasLabel   ein {@link JLabel}, das
	 *                   {@link #beschrifte(Component, NameContainer)}t werden soll
	 * @param damitLabel die {@link Beschriftung} für das JLabel
	 * @param das        die {@link Component}, die die Methode beschriften soll
	 * @param damit      die {@link UnBeschriftung} für die {@link Component}, auf
	 *                   die sich das {@link JLabel} beziehen soll.
	 * @param dahin      der aufnehmende {@link Container}
	 * @return die {@link Component} - das JLabel wird nicht zurückgegeben.
	 */
	public Component beschrifteGemeinsam(JLabel dasLabel, Beschriftung damitLabel, Component das, UnBeschriftung damit,
			Container dahin) {
		beschrifteVergittert(dasLabel, damitLabel, dahin);
		beschrifteVergittert(das, damit, dahin);
//		if (damitLabel == damit.getRef())
//			System.out.printf("beschrifte zugehörig %s (= %s) zu %s.%n", damitLabel.getId(), damit.getRef().getId(),
//					damit.getId());
//		else
//			System.out.printf("Die Beschriftung von %s und %s stimmen nicht überein!%n", damitLabel.getId(),
//					damit.getRef().getId());
		dasLabel.setLabelFor(das);
		return das;
	}

	/**
	 * @param dasLabel ein {@link JLabel}, das
	 *                 {@link #beschrifte(Component, NameContainer)}t werden soll
	 * @param das      die {@link Component}, die die Methode beschriften soll
	 * @param damit    die {@link UnBeschriftung} für die {@link Component}, auf die
	 *                 sich das {@link JLabel} beziehen soll, und aus der auch die
	 *                 {@link Beschriftung} für das {@link JLabel} gezogen wird -
	 *                 über {@link UnBeschriftung#getRef()}.
	 * @param dahin    der aufnehmende {@link Container}
	 * @return die {@link Component} - das JLabel wird nicht zurückgegeben.
	 */
	public Component beschrifteGemeinsam(JLabel dasLabel, Component das, UnBeschriftung damit, Container dahin) {
		Beschriftung damitLabel = damit.getRef();
		return beschrifteGemeinsam(dasLabel, damitLabel, das, damit, dahin);
	}

	/**
	 * Die Methode beschriftet sowohl ein neu erstelltes {@link JLabel} als auch
	 * eine weitere {@link Component}, fügt beide dem {@link Container} hinzu und
	 * markiert das {@link JLabel} als zur {@link Component} gehörig mit
	 * {@link JLabel#setLabelFor(Component)}.
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit die {@link UnBeschriftung} für die {@link Component}, auf die
	 *              sich das neu erstellte {@link JLabel} beziehen soll, und aus der
	 *              auch die {@link Beschriftung} für das {@link JLabel} gezogen
	 *              wird - über {@link UnBeschriftung#getRef()}.
	 * @param dahin der aufnehmende {@link Container}
	 * @return die {@link Component} - das JLabel wird nicht zurückgegeben.
	 */
	public Component beschrifteGemeinsam(Component das, UnBeschriftung damit, Container dahin) {
		return beschrifteGemeinsam(new JLabel(), das, damit, dahin);
	}

	/**
	 * fügt das ganze dann noch in einen {@link Container} ein.
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit der {@link NameContainer}
	 * @param dahin der aufnehmende {@link Container}
	 */
	public Component add(Component das, NameContainer damit, Container dahin) {
		if (damit.gitterBeutel != null) {
			dahin.add(das, damit.gitterBeutel.get(0).toGitterBeutel());
		} else
			dahin.add(das);
		return das;
	}

	/**
	 * Zieht aus einer {@link BeschriebeneAktion} mit
	 * {@link BeschriebeneAktion#toButton()} und fügt sie in den {@link Container}
	 * ein.
	 * 
	 * @param das   die {@link BeschriebeneAktion}, aus der die Beschriftung und der
	 *              {@link GitterBeutelBeschraenkungen} gezogen werden
	 * @param dahin der aufnehmende {@link Container}
	 * @return die {@link Component}, die die Methode beschriften soll, also ein
	 *         {@link JButton} oder
	 */
	public Component add(BeschriebeneAktion das, Container dahin) {
		return add(das.toButton(), das.getBeschriftung(), dahin);
	}

	/**
	 * Wird eine {@link BooleanConfigAction} übergeben, wird über
	 * {@link BooleanConfigAction#toJToggleButton()} ein {@link JToggleButton}
	 * eingefügt.
	 * 
	 * @param das   die {@link BooleanConfigAction}, aus der die Beschriftung und
	 *              der {@link GitterBeutelBeschraenkungen} gezogen werden
	 * @param dahin der aufnehmende {@link Container}
	 * @return ein {@link JToggleButton}
	 */
	public Component add(BooleanConfigAction das, Container dahin) {
		return add(((BooleanConfigAction) das).toJToggleButton(), das.getBeschriftung(), dahin);
	}

	/**
	 * Die Methode setzt die {@link Action#getValue(String)}s der {@link Action}
	 * 
	 * @param action {@link Action}
	 * @param damit  die {@link Beschriftung}
	 */
	public Action beschrifte(Action action, LangBeschriftung damit) {
		deployTitle(action, damit);
		deployShortDescription(action, damit, null);
		deployLongDescription(action, damit);
		deployActionCommandKey(action, damit);
		deployImage(action, damit);
		deployMnemonic(action, damit);
		deployAcceleratorKey(action, damit);
		return action;
	}

	/**
	 * Die Methode setzt einen {@link Beschriftung#getTitle()} aus den
	 * {@link Vorsorgend#getAlternativeTitles()}, sofern er enthalten ist, und die
	 * {@link Beschriftung#getShortDescription()} aus den
	 * {@link Vorsorgend#getAlternativeDescriptions()}.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 * 
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 * 
	 * @return die {@link VorsorgendeBeschriftung} mit dem neuen
	 *         {@link Beschriftung#getTitle()}
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.Vorsorgend
	 */
	public Vorsorgend<?> changeTo(Vorsorgend<?> vorsorgend, int index) {
		changeTitle(vorsorgend, index);
		changeDescription(vorsorgend, index);
		return vorsorgend;
	}

	/**
	 * Die Methode setzt über {@link #changeTo(Vorsorgend, int)} den
	 * {@link Beschriftung#getTitle()} und den
	 * {@link Beschriftung#getShortDescription()}. Dann formatiert sie den
	 * {@link Beschriftung#getTitle()} mit Hilfe der <code>args</code> über
	 * {@link String#format(String, Object...)}, sofern
	 * {@link Vorsorgend#isFormatted()} gesetzt ist.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 * 
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 * @param args       die weiteren Argumente für
	 *                   {@link String#format(String, Object...)}
	 * @return die {@link VorsorgendeBeschriftung} mit dem neuen
	 *         {@link Beschriftung#getTitle()}
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.Vorsorgend
	 * @see java.lang.String#format(String, Object...)
	 */
	public Vorsorgend<?> changeTo(Vorsorgend<?> vorsorgend, int index, Object... args) {
		changeTo(vorsorgend, index);
		if (vorsorgend != null && vorsorgend.isFormatted()) {
			Beschriftung beschriftung = (Beschriftung) vorsorgend;
			String format = String.format(beschriftung.getTitle(), args);
			beschriftung.setTitle(format);
		} else {
			System.out.println("Ist nicht formatted!");
		}
		return vorsorgend;
	}

	/**
	 * Die Methode setzt einen {@link Beschriftung#getTitle()} aus den
	 * {@link Vorsorgend#getAlternativeTitles()}, sofern der Text aus
	 * <code>title</code> in einer von ihnen enthalten ist, und die
	 * {@link Beschriftung#getShortDescription()} aus den
	 * {@link Vorsorgend#getAlternativeDescriptions()}.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 *
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param title      den {@link Beschriftung#getTitle()}, der herausgesucht
	 *                   werden soll
	 * @return <code>true</code>, wenn er in {@link #getAlternativeTitles()}
	 *         gefunden wird
	 * @see eu.gronos.beschriftungen.interfaces.Vorsorgend
	 */
	public boolean changeTo(Vorsorgend<?> vorsorgend, String title) {
		int index = indexOf(vorsorgend.getAlternativeTitles(), title);
		if (index > -1) {
			changeTo(vorsorgend, index);
			return true;
		}
		Kostenrechner.getLogger().info(String.format("Beschriftung nicht gefunden: '%s'", title));
		return false;
	}

	/**
	 * Die Methode prüft, ob ein {@link String} in den <code>altenativen</code> ist.
	 * 
	 * @param alternativen die {@link List} aus {@link String}s mit
	 *                     {@link Vorsorgend#getAlternativeTitles()} oder
	 *                     {@link Vorsorgend#getAlternativeDescriptions()}
	 * @param neu          der zu prüfende Text als {@link String}
	 * @return den index von <code>neu</code> in <code>alternativen</code>
	 */
	public int indexOf(List<String> alternativen, String neu) {
		for (int index = 0; index < alternativen.size(); index++) {
			if (istGleicherText(alternativen.get(index), neu)) {
				return index;
			}
		}
		return -1;
	}

	/**
	 * Die Methode prüft, ob zwei {@link String}s unabhängig von
	 * Groß-/Kleinschreibung und führenden Leerzeichen gleich sind.
	 * 
	 * @param erste  {@link String}
	 * @param zweite {@link String}
	 * @return <code>true</code>, wenn beide gleich sind
	 */
	public boolean istGleicherText(String erste, String zweite) {
		return deWhiteSpace(erste).toLowerCase().equals(deWhiteSpace(zweite).toLowerCase());
	}

	/**
	 * Die Methode in dem übergebenen {@link String} nach einer bestimmten Anzahl
	 * von Zeichen ({@link #MAX_ZEICHEN_PRO_ZEILE}) einen Zeilenumbruch einzufügen,
	 * aber nicht mitten im Wort.
	 * 
	 * @param string der zu verarbeitende {@link String}
	 * @return der Ausgabe-{@link String} mit den eingefügten Zeilenumbrüchen.
	 */
	public String brecheZeilen(String string) {
		StringBuffer gesamt = new StringBuffer();
		StringBuffer zeile = new StringBuffer();
		String[] zeilen = string.trim().split("\\s+");
		for (String wort : zeilen) {
			if (zeile.length() + wort.length() > MAX_ZEICHEN_PRO_ZEILE) {
				gesamt.append(zeile);
				gesamt.append("\n");
				zeile = new StringBuffer();
			}
			zeile.append(wort);
			zeile.append(" ");

		}
		gesamt.append(zeile);
		gesamt.append("\n");
		return gesamt.toString();
	}

	/**
	 * Die Methode entfernt überflüssigen Whitespace aus einem String, der beim
	 * XML-Import entsteht.
	 * 
	 * @param string der {@link String}
	 * @return den um zuviel Whitespace bereinigten {@link String}
	 */
	public String deWhiteSpace(String string) {
		return string.replace('\t', ' ').replace('\n', ' ').replace('\r', ' ').replaceAll("\\s", " ")
				.replaceAll("\\s+", " ").trim();
	}

	/**
	 * @param dahin der aufnehmende {@link Container}
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @return die {@link Component}
	 */
//	public Component testeUnVergittert(Container dahin, Component das, GitterBeutelBeschraenkungen damit) {
//		System.out.println(das.getClass());
//		System.out.println("<... id=" + das.getName());
//
//		final AlignmentAdapter adapter = new AlignmentAdapter();
//
//		if (das instanceof JLabel) {
//			System.out.println(
//					"			<richter horizontal=\"" + adapter.marshal(((JLabel) das).getHorizontalAlignment())
//							+ "\" vertical=\"" + adapter.marshal(((JLabel) das).getVerticalAlignment()) + "\" />");
//		} else if (das instanceof JTextField) {
//			System.out.println("			<richter horizontal=\""
//					+ adapter.marshal(((JTextField) das).getHorizontalAlignment()) + "\" />");
//		} else if (das instanceof AbstractButton) {
//			System.out.println("			<richter horizontal=\""
//					+ adapter.marshal(((AbstractButton) das).getHorizontalAlignment()) + "\" vertical=\""
//					+ adapter.marshal(((AbstractButton) das).getVerticalAlignment()) + "\" />");
//		}
//
//		System.out.print("			");
//
//		damit.println(dahin.getName());
//
//		System.out.println("/ ...>");
//
//		dahin.add(das, damit);
//		return das;
//	}

	/**
	 * Die Methode setzt den {@link Beschriftung#getTitle()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung die {@link Beschriftung}
	 */
	private void deployTitle(Object das, Beschriftung beschriftung) {
		String title = beschriftung.getTitle();
		if (das instanceof Action) {
			((Action) das).putValue(Action.NAME, title);
		} else if (das instanceof JComponent) {
			if (das instanceof JLabel)
				((JLabel) das).setText(title);
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployTitle(((AbstractButton) das).getAction(), beschriftung);
				else
					((AbstractButton) das).setText(title);
			} else if ((das instanceof JPanel || das instanceof JScrollPane)
					&& (((JComponent) das).getBorder() instanceof TitledBorder))
				((TitledBorder) ((JComponent) das).getBorder()).setTitle(title);
		} else if (das instanceof Window) {
			if (das instanceof JDialog)
				((JDialog) das).setTitle(title);
			else if (das instanceof JFrame)
				((JFrame) das).setTitle(title);
		} else if (das instanceof Component)
			deployAccessibleName((Component) das, title);
	}

	/**
	 * Die Methode setzt den {@link AccessibleContext#getAccessibleName()} einer
	 * {@link Component}
	 * 
	 * @param das   {@link Component}
	 * @param title der String, der als
	 *              {@link AccessibleContext#getAccessibleName()} gesetzt werden
	 *              soll.
	 */
	private void deployAccessibleName(Component das, String title) {
		if (das instanceof JScrollPane && ((JScrollPane) das).getVerticalScrollBar() != null
				&& ((JScrollPane) das).getVerticalScrollBar().getAccessibleContext() != null) {
			((JScrollPane) das).getVerticalScrollBar().getAccessibleContext().setAccessibleName(title);
		} else if (das.getAccessibleContext() != null) {
			das.getAccessibleContext().setAccessibleName(title);
		} else
			System.err.println("AccessibleContext null bei " + das.getClass().toString());
	}

	/**
	 * Die Methode verknüpft {@link Beschriftung#getTitle()},
	 * {@link UnBeschriftung#getPrefix()} und {@link UnBeschriftung#getSuffix()}
	 * 
	 * @param un die {@link UnBeschriftung}
	 * 
	 * @return den String
	 */
	private String createCompleteTitle(UnBeschriftung un) {
		String title = un.getRef().getTitle();
		if (title == null || title.trim().isEmpty())
			return "";
		title = deWhiteSpace(title);
		final StringBuilder sb = new StringBuilder(title);
		if (un.getPrefix() != null && !un.getPrefix().isEmpty())
			sb.insert(0, un.getPrefix() + " ");
		if (un.getSuffix() != null && !un.getSuffix().isEmpty())
			sb.append(" " + un.getSuffix());
		return sb.toString();
	}

	/**
	 * Die Methode setzt die {@link Beschriftung#getShortDescription()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}. Sie setzt außerdem die
	 * {@link AccessibleContext#getAccessibleDescription()}. Im
	 * {@link Kostenrechner#debug}-Modus wird die {@link NameContainer#getId()} der
	 * {@link Beschriftung} und ggfs. auch der {@link UnBeschriftung} hinzugefügt
	 * 
	 * @param das            eine {@link Component} oder {@link Action}
	 * @param beschriftung   die {@link Beschriftung}
	 * @param unbeschriftung im {@link Kostenrechner#debug}-Modus wird die
	 *                       {@link NameContainer#getId()} hinzugefügt
	 */
	private void deployShortDescription(Object das, Beschriftung beschriftung, UnBeschriftung unbeschriftung) {
		String shortDescription = beschriftung.getShortDescription();
		if (Kostenrechner.debug) {
			String id = "(" + beschriftung.getId();
			if (unbeschriftung != null) {
				id += ":" + unbeschriftung.getId();
			}
			id += ") ";
			shortDescription = id + shortDescription;
		}
		if (shortDescription != null && !shortDescription.isEmpty()) {
			shortDescription = deWhiteSpace(shortDescription);
			shortDescription = brecheZeilen(shortDescription);
		}
		if (das instanceof Action)
			((Action) das).putValue(Action.SHORT_DESCRIPTION, shortDescription);
		else if (das instanceof AbstractButton && ((AbstractButton) das).getAction() != null)
			deployShortDescription(((AbstractButton) das).getAction(), beschriftung, unbeschriftung);
		else if (das instanceof JComponent)
			((JComponent) das).setToolTipText(shortDescription);
		if (das instanceof Component && ((Component) das).getAccessibleContext() != null)
			((Component) das).getAccessibleContext().setAccessibleDescription(shortDescription);
	}

	/**
	 * Die Methode setzt das {@link Beschriftung#getMnemonic()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung die {@link Beschriftung}
	 */
	private void deployMnemonic(Object das, Beschriftung beschriftung) {
		if (beschriftung.hasMnemonic()) {
			if (das instanceof Action)
				((Action) das).putValue(Action.MNEMONIC_KEY, Integer.valueOf(beschriftung.getMnemonic()));
			else if (das instanceof JComponent) {
				if (das instanceof JLabel)
					((JLabel) das).setDisplayedMnemonic(beschriftung.getMnemonic());
				else if (das instanceof AbstractButton) {
					if (((AbstractButton) das).getAction() != null)
						deployMnemonic(((AbstractButton) das).getAction(), beschriftung);
					else
						((AbstractButton) das).setMnemonic(beschriftung.getMnemonic());
				}
			}
		}
	}

	/**
	 * Wenn {@link Gerichtet#horizontal} oder {@link Gerichtet#vertical} gesetzt
	 * sind, wird bei einem {@link JLabel}, einem {@link AbstractButton} oder einem
	 * {@link JTextField} die horizontale beziehungsweise die vertikale Ausrichtung
	 * angepasst.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung das {@link Gerichtet} einer {@link Beschriftung} oder
	 *                     {@link UnBeschriftung}
	 */
	private void deployAlignment(Component das, Gerichtet damit) {
		// ggfs. bei bestimmten Components die horizontale Ausrichtung richten
		if (damit.horizontal != null) {
			if (das instanceof JLabel) {
				((JLabel) das).setHorizontalAlignment(damit.horizontal);
			} else if (das instanceof AbstractButton) {
				((AbstractButton) das).setHorizontalAlignment(damit.horizontal);
			} else if (das instanceof JTextField) {
				((JTextField) das).setHorizontalAlignment(damit.horizontal);
			}
		}
		// ggfs. bei bestimmten Components die vertikale Ausrichtung richten
		if (damit.vertical != null) {
			if (das instanceof JLabel) {
				((JLabel) das).setVerticalAlignment(damit.vertical);
			} else if (das instanceof AbstractButton) {
				((AbstractButton) das).setVerticalAlignment(damit.vertical);
			} else if (das instanceof JTextField) {
				// bei JTextField gibt's das nicht
			}
		}
	}
//	if (damit.gitterBeutel.get(0) != null && damit.gitterBeutel.get(0).horizontalAlignment != null
//			&& das instanceof JLabel) {
//		System.out.println("horizontalAlignment: " + damit.gitterBeutel.get(0).horizontalAlignment);
//		((JLabel) das).setHorizontalAlignment(damit.gitterBeutel.get(0).horizontalAlignment);
//	}

	/**
	 * Die Methode setzt den {@link LangBeschriftung#getActionCommandKey()} einer
	 * {@link LangBeschriftung} auf einen {@link AbstractButton} oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployActionCommandKey(Object das, LangBeschriftung beschriftung) {
		if (beschriftung.getActionCommandKey() != null) {
			if (das instanceof Action)
				((Action) das).putValue(Action.ACTION_COMMAND_KEY, beschriftung.getActionCommandKey());
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployActionCommandKey(((AbstractButton) das).getAction(), beschriftung);
				else
					((AbstractButton) das).setActionCommand(beschriftung.getActionCommandKey());
			}
		}
	}

	/**
	 * Die Methode setzt die {@link LangBeschriftung#getLongDescription()} einer
	 * {@link LangBeschriftung} auf einen {@link AbstractButton} oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployLongDescription(Object das, LangBeschriftung damit) {
		if (damit.getLongDescription() != null)
			if (das instanceof Action)
				((Action) das).putValue(Action.LONG_DESCRIPTION, damit.getLongDescription());
			else if (das instanceof AbstractButton && ((AbstractButton) das).getAction() != null)
				deployLongDescription(((AbstractButton) das).getAction(), damit);
	}

	/**
	 * Die Methode setzt den {@link LangBeschriftung#getAcceleratorKey()} einer
	 * {@link LangBeschriftung} eine {@link Action}. Auf einen
	 * {@link AbstractButton} nur, wenn er eine {@link AbstractButton#getAction()}
	 * vorhanden ist oder er ein {@link JMenuItem} ist.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployAcceleratorKey(Object das, LangBeschriftung damit) {
		if (damit.getAcceleratorKey() != null) {
			if (das instanceof Action)
				((Action) das).putValue(Action.ACCELERATOR_KEY, damit.getAcceleratorKey().toKeyStroke());
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployAcceleratorKey(((AbstractButton) das).getAction(), damit);
				else if (das instanceof JMenuItem)
					((JMenuItem) das).setAccelerator(damit.getAcceleratorKey().toKeyStroke());
			}
		}
	}

	/**
	 * Die Methode setzt das {@link Bebildert#getImage()} einer
	 * {@link LangBeschriftung} oder eines {@link BildOhneTitel}s auf ein
	 * {@link JLabel}, einen {@link AbstractButton} oder auf eine {@link Action}.
	 * 
	 * @param das   ein {@link JLabel}, einen {@link AbstractButton} oder auf eine
	 *              {@link Action}
	 * @param damit eine {@link LangBeschriftung} oder ein {@link BildOhneTitel}
	 */
	private void deployImage(Object das, Bebildert damit) {
		if (damit.getImage() != null) {
			ImageIcon ii = RechnerhammerIcons.loadIcon(damit.getImage());
			ii.setDescription(damit.getImage());
			if (das instanceof Action)
				((Action) das).putValue(Action.LARGE_ICON_KEY, ii);
			else if (das instanceof JLabel)
				((JLabel) das).setIcon(RechnerhammerIcons.loadIcon(damit.getImage()));
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployImage(((AbstractButton) das).getAction(), damit);
				else
					((AbstractButton) das).setIcon(RechnerhammerIcons.loadIcon(damit.getImage()));
			}
		}
	}

	/**
	 * Die Methode setzt die Werte für {@link JSlider#getMinimum()},
	 * {@link JSlider#getMaximum()}, {@link JSlider#getMinorTickSpacing()} und
	 * {@link JSlider#getMajorTickSpacing()} aus der {@link IntervallUnBeschriftung}
	 * 
	 * @param das   eine {@link Component}, derzeit nur {@link JSlider}
	 * @param damit eine {@link IntervallUnBeschriftung}
	 */
	private void deployIntervall(Component das, IntervallUnBeschriftung damit) {
		if (das instanceof JSlider) {
			JSlider slider = (JSlider) das;
			slider.setMinimum(damit.getMinimum());
			slider.setMaximum(damit.getMaximum());
			slider.setMinorTickSpacing(damit.getMinorTick());
			slider.setMajorTickSpacing(damit.getMajorTick());
			slider.setPaintTicks(true);
			slider.setPaintLabels(true);
		}
	}

	/**
	 * Die Methode setzt einen Title aus den vorhandenen
	 * {@link #getAlternativeTitles()}.
	 * 
	 * @param vorsorgend eine {@link VorsorgendeBeschriftung} oder
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 */
	private void changeTitle(Vorsorgend<?> vorsorgend, int index) {
		if (index > -1 && index < vorsorgend.getAlternativeTitles().size()) {
			((Beschriftung) vorsorgend).setTitle(deWhiteSpace(vorsorgend.getAlternativeTitles().get(index)));
		} else {
			Kostenrechner.getLogger().info(String.format("Titel Nr. %d aus Beschriftung %d nicht vorhanden: !%n", index,
					((NameContainer) vorsorgend).getKey()));
		}
	}

	/**
	 * Die Methode setzt eine {@link #getShortDescription()} aus den
	 * {@link #getAlternativeDescriptions()}.
	 * 
	 * @param vorsorgend eine {@link VorsorgendeBeschriftung} oder
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index der neuen {@link #getShortDescription()}
	 */
	private void changeDescription(Vorsorgend<?> vorsorgend, int index) {
		if (index > -1 && vorsorgend.getAlternativeDescriptions() != null
				&& index < vorsorgend.getAlternativeDescriptions().size()) {
			((Beschriftung) vorsorgend)
					.setShortDescription(deWhiteSpace(vorsorgend.getAlternativeDescriptions().get(index)));
		} else {
			Kostenrechner.getLogger()
					.info(String.format("Kurzbeschreibung Nr. %d aus Beschriftung %d nicht vorhanden!%n", index,
							((NameContainer) vorsorgend).getKey()));
		}
	}

}
