/**
 * ValuesAction.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.beschriftungen.controller;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.JMenuItem;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainer;

/**
 * Von {@link AbstractAction} abgeleitete Klasse, deren Konstruktor bereits
 * Felder aus einer {@link LangBeschriftung} setzt.
 *
 * @author Peter Schuster (setrok)
 * @date 31.08.2014
 *
 */
public abstract class BeschriebeneAktion extends AbstractAction {
	private static final long serialVersionUID = -2990321113189187333L;
	private final LangBeschriftung beschriftung;

	/**
	 * Ein Konstruktor, der bereits Felder setzen lässt.
	 * 
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link javax.swing.Action#MNEMONIC_KEY},
	 *                     {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 */
	public BeschriebeneAktion(LangBeschriftung beschriftung) {
		super();
		new ComponentBeschrifter().beschrifte(this, beschriftung);
		this.beschriftung = beschriftung;
	}

	/**
	 * @return gibt {@link #beschriftung} als {@link LangBeschriftung} zurück, eine
	 *         {@link LangBeschriftung}, der auch ({@link javax.swing.Action#NAME}),
	 *         ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *         ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen werden.
	 */
	public LangBeschriftung getBeschriftung() {
		return beschriftung;
	}

	/**
	 * Die Methode baut einen {@link JButton} mit der {@link BeschriebeneAktion} und
	 * setzt zusätzlich {@link JButton#getName()} auf die
	 * {@link NameContainer#getId()}.
	 * 
	 * @return den erstellten {@link JButton}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toButton()
	 */
	public JButton toButton() {
		JButton button = new JButton();
		return (JButton) toButton(button);
	}

	/**
	 * Die Methode versieht einen {@link JButton} oder ein abgeleitetes Objekt mit
	 * der {@link BeschriebeneAktion} und setzt zusätzlich {@link JButton#getName()}
	 * auf die {@link NameContainer#getId()}.
	 * 
	 * @param button einen {@link JButton}
	 * @return den präparierten {@link JButton}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toButton(javax.swing.AbstractButton)
	 */
	public AbstractButton toButton(AbstractButton button) {
		button.setAction(this);
		button.setName(beschriftung.getId());
		return button;
	}

	/**
	 * Die Methode baut ein {@link JMenuItem} mit der {@link BeschriebeneAktion} und
	 * setzt zusätzlich {@link JMenuItem#getName()} auf die
	 * {@link NameContainer#getId()}.
	 * 
	 * @return das erstellte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem()
	 */
	public JMenuItem toMenuItem() {
		JMenuItem item = new JMenuItem();
		return toMenuItem(item);
	}

	/**
	 * Die Methode versieht ein {@link JMenuItem} oder ein abgeleitetes Objekt mit
	 * der {@link BeschriebeneAktion} und setzt zusätzlich
	 * {@link JMenuItem#getName()} auf die {@link NameContainer#getId()}.
	 * 
	 * @param item ein {@link JMenuItem}
	 * @return das präparierte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem(javax.swing.JMenuItem)
	 */
	public JMenuItem toMenuItem(JMenuItem item) {
		item.setAction(this);
		item.setName(beschriftung.getId());
		return item;
	}

}
