package eu.gronos.beschriftungen.controller;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JToggleButton;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.interfaces.ConfigOption;
import eu.gronos.beschriftungen.model.BooleanConfigSchluessel;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.LangVorsorgeBeschriftung;
import eu.gronos.beschriftungen.model.NameContainer;
import eu.gronos.beschriftungen.model.SchluesselBeschriftung;
import eu.gronos.beschriftungen.util.BooleanConfigOption;

/**
 * Eine {@link BeschriebeneAktion} (und damit auch {@link ActionListener} für
 * {@link AbstractButton}-Elemente, vor {@link JToggleButton} und
 * {@link JCheckBoxMenuItem}.
 * 
 * Jede Instanz korrespondiert mit einem Boolean-Wert <code>schluessel</code> in
 * der Registry im Unterast <code>"/eu/gronos/kostentenor"</code> zum Java-Ast.
 * Diese wird zunächst eingelesen. Wenn der Schlüssel <code>schluessel</code>
 * noch nicht gesetzt ist, wird der als <code>standardWert</code> übergebene
 * Wahrheitswert genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 18 Feb 2019
 *
 */
public class BooleanConfigAction extends BeschriebeneAktion implements ConfigOption<Boolean> {

	private static final long serialVersionUID = 6129451338244952803L;
	private final BooleanConfigOption option;

	/**
	 * @param beschriftung eine {@link SchluesselBeschriftung}, der auch ein
	 *                     {@link BooleanConfigSchluessel},
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link javax.swing.Action#MNEMONIC_KEY},
	 *                     {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 */
	public BooleanConfigAction(SchluesselBeschriftung beschriftung) {
		super(beschriftung);
		option = new BooleanConfigOption((BooleanConfigSchluessel) beschriftung.getConfigSchluessel());
	}

	/**
	 * Die Methode wird aufgerufen, wenn sich der Status
	 * {@link AbstractButton#isSelected()} ändert.
	 * 
	 * Beim Klicken wird der Text des {@link AbstractButton} der Auswahl angepasst
	 * und mit {@link #inRegistrySchreiben(boolean)} der Wert auch gespeichert.
	 * 
	 * @param e ein {@link ActionEvent}
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();
		if (source instanceof AbstractButton) {
			setValueFromButton((AbstractButton) source);
		}
	}

	/**
	 * Die Methode leitet die Anfrage an die {@link #option} weiter
	 * 
	 * @return den {@link BooleanConfigOption#getStandardWert()} der {@link #option}
	 *         als {@link Boolean}
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#getStandardWert()
	 */
	@Override
	public Boolean getStandardWert() {
		return option.getStandardWert();
	}

	/**
	 * Die Methode leitet die Anfrage an die {@link #option} weiter
	 * 
	 * @return den {@link BooleanConfigOption#getSchluessel()} der {@link #option}
	 *         als {@link Boolean}
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#getSchluessel()
	 */
	@Override
	public String getSchluessel() {
		return option.getSchluessel();
	}

	/**
	 * Die Methode schreibt den Wert in die Registry/Preferences und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @param value
	 */
	public void setValue(Boolean value) {
		option.setValue(value);
		setValueToButton(value);//
	}

	/**
	 * 
	 * Die Methode liest den Wert aus der Registry (den Preferences) und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @return value als {@link Boolean}
	 */
	public Boolean getValue() {
		Boolean value = option.getValue();
		setValueToButton(value);
		return value;
	}

	/**
	 * Die Methode leitet den Aufruf an {@link #option} weiter
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#reset()
	 */
	@Override
	public void reset() {
		option.reset();
	}

	/**
	 * Die Methode baut einen {@link JToggleButton} mit dem {@link ItemListener} und
	 * setzt zusätzlich {@link JToggleButton#getName()} auf die
	 * {@link NameContainer#getId()}.
	 * 
	 * @return den erstellten {@link JToggleButton} als {@link AbstractButton}
	 * 
	 */
	public JToggleButton toJToggleButton() {
		final JToggleButton tb = (JToggleButton) toButton(new JToggleButton());
		tb.setName(getBeschriftung().getId());// sonst wird die nicht gesetzt?
		return tb;
	}

	/**
	 * Die Methode versieht einen {@link AbstractButton} oder ein abgeleitetes
	 * Objekt mit dem {@link ItemListener} und setzt zusätzlich
	 * {@link AbstractButton#getName()} auf die {@link NameContainer#getId()}.
	 * 
	 * @param button einen {@link AbstractButton}
	 * @return den präparierten {@link AbstractButton}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toButton(javax.swing.AbstractButton)
	 */
	@Override
	public AbstractButton toButton(AbstractButton button) {
		button.setAction(this);
		getValue();
		setButtonText();
		return button;
	}

	/**
	 * Die Methode baut ein {@link JMenuItem} mit dem {@link ItemListener} und setzt
	 * zusätzlich {@link JMenuItem#getName()} auf die {@link NameContainer#getId()}.
	 * 
	 * @return das erstellte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem()
	 */
	@Override
	public JMenuItem toMenuItem() {
		JCheckBoxMenuItem item = new JCheckBoxMenuItem();
		return toMenuItem(item);
	}

	/**
	 * Die Methode versieht ein {@link JMenuItem} oder ein abgeleitetes Objekt mit
	 * dem {@link ItemListener} und setzt zusätzlich {@link JMenuItem#getName()} auf
	 * die {@link NameContainer#getId()}.
	 * 
	 * @param item ein {@link JMenuItem}
	 * @return das präparierte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem(javax.swing.JMenuItem)
	 */
	@Override
	public JMenuItem toMenuItem(JMenuItem item) {
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.beschrifte(item, getBeschriftung());
		item.setAction(this);
		getValue();
		setButtonText();
		return item;
	}

	/**
	 * @return gibt {@link #beschriftung} als {@link LangBeschriftung} zurück, der
	 *         auch ({@link javax.swing.Action#NAME}),
	 *         ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *         {@link javax.swing.Action#MNEMONIC_KEY},
	 *         {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *         ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen werden.
	 */
	@Override
	public SchluesselBeschriftung getBeschriftung() {
		return (SchluesselBeschriftung) super.getBeschriftung();
	}


	/**
	 * Beim Klicken wird der Text des {@link AbstractButton} aus den
	 * {@link LangVorsorgeBeschriftung#getAlternativeTitles()} der Auswahl
	 * angepasst, bei {@link Boolean#TRUE} der index 1, bei {@link Boolean#FALSE}
	 * der index 0.
	 * 
	 * Bei formattierten Beschriftungen muss das die abgeleitete Klasse übernehmen.
	 * 
	 */
	protected void setButtonText() {
		int index = valueToIndex();
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.changeTo(getBeschriftung(), index);
		beschrifter.beschrifte(this, getBeschriftung());
	}

	/**
	 * Die Methode gibt statt des {@link Boolean} aus {@link #getValue()} einen
	 * <code>int</code> zurück, um einen index für die
	 * {@link LangVorsorgeBeschriftung} hat.
	 * 
	 * @return 1 bei <code>true</code>, 0 bei <code>false</code>
	 */
	private int valueToIndex() {
		return getValue().booleanValue() ? 1 : 0;
	}


	/**
	 * Die Methode liest den Wert vom {@link AbstractButton#isSelected()} und
	 * schreibt ihn in die {@link #option}
	 * 
	 * @param button den {@link AbstractButton#isSelected()}
	 */
	private void setValueFromButton(AbstractButton button) {
		boolean value = button.isSelected();
		option.inRegistrySchreiben(value);
		setButtonText();
	}

	/**
	 * Die Methode setzt das Oberflächenelement über
	 * {@link Action#SELECTED_KEY }entsprechend dem {@link #getValue()}.
	 * 
	 * @param value <code>true</code> if the button is selected, otherwise false
	 */
	private void setValueToButton(boolean value) {
		// Der value darf hier nicht aus #getValue gezogen werden, sonst Endlosschleife
		putValue(Action.SELECTED_KEY, value);
	}

}
