/*
 * NameContainer.java
 * eu.gronos.kostenrechner.model (Kostenrechner)
 */
package eu.gronos.beschriftungen.model;

import java.io.Serializable;
import java.util.List;
import java.util.Map.Entry;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlID;
import javax.xml.bind.annotation.XmlIDREF;
import javax.xml.bind.annotation.XmlTransient;

/**
 * Klasse, die alle Oberflächenelementbeschreibungen versammelt und das als XML
 * exportiert. Abstrakte Oberklasse „NameContainer“ mit String name/id
 * (XmlAttribute und XmlID),
 *
 * @author Peter Schuster (setrok)
 * @date 03.08.2019
 *
 */
public abstract class NameContainer implements Comparable<NameContainer>, Serializable, Entry<Integer, NameContainer> {
	private static final long serialVersionUID = -4652423723636448828L;
	private Integer id = null;

	protected NameContainer() {
		super();
	}

	/**
	 * Die {@link #id} soll den {@link NameContainer} und damit das beschriebene
	 * Oberflächenelement eindeutig identifizieren. Es wird intern als
	 * {@link Integer} / int gespeichert.
	 * 
	 * Sie wird aber als {@link String} zurückgegeben, da bei {@link JAXB} mit
	 * {@link XmlIDREF} nur auf einen {@link String} verwiesen werden kann.
	 * 
	 * @return gibt {@link #id} als {@link String} zurück, <code>null</code>, wenn
	 *         die #id noch nicht gesetzt wurde.
	 */
	@XmlAttribute
	@XmlID
	public String getId() {
		if (id != null && id > 0)
			return id.toString();
		else
			return null;
	}

	/**
	 * Die Methode setzt die {@link #getId()}
	 * 
	 * @param id d. {@link #getId}, d. gesetzt werden soll als {@link String}. Diese
	 *           {@link #getId()} muss sich in eine Ganzzahl umwandeln lassen.
	 * @throws NumberFormatException if the string cannot be parsed as an integer.
	 */
	public void setId(String id) throws NumberFormatException {
		this.id = Integer.valueOf(id);
	}

	/**
	 * Die Methode vergleicht die {@link NameContainer} anhand von {@link #getId()}
	 * 
	 * @param o the {@link NameContainer} to be compared.
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(NameContainer o) {
		return id.compareTo(o.id);
	}

	/**
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		if (id == null)
			return 0;
		return id.hashCode();
	}

	/**
	 * 
	 * @return den {@link Integer#valueOf(int)} von {@link #idIntValue()}
	 * 
	 * @see java.util.Map.Entry#getKey()
	 */
	@Override
	@XmlTransient
	public Integer getKey() {
		return id;
	}

	/**
	 * The GridBagConstraints class specifies constraints for components that are
	 * laid out using the GridBagLayout class. Creates a GridBagConstraint object
	 * with all of its fields set to their default value.
	 */
	@XmlElement
	public List<GitterBeutelBeschraenkungen> gitterBeutel;

	/**
	 * Horizontale und vertikale Ausrichtung
	 */
	@XmlElement
	public Gerichtet richter;

	/**
	 * Die Methode vergleicht anhand von {@link #getId()}
	 * 
	 * @param obj the reference object with which to compare.
	 * @return <code>true</code> if this object is the same as the obj argument;
	 *         <code>false</code> otherwise.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!(obj instanceof NameContainer)) {
			return false;
		}
		NameContainer other = (NameContainer) obj;
		if (id == null) {
			if (other.id != null) {
				return false;
			}
		} else if (getKey().equals(0)) {
			if (!other.getKey().equals(0)) {
				return false;
			}
		} else {
			return id.equals(other.id);
		}
		return true;
	}

	/**
	 * @return sich selbst als {@link NameContainer}
	 * 
	 * @see java.util.Map.Entry#getValue()
	 */
	@Override
	@XmlTransient
	public NameContainer getValue() {
		return this;
	}

	/**
	 * Die Methode gibt nur sich selbst zurück
	 * 
	 * @param value wird nicht verwandt.
	 * @return sich selbst als {@link NameContainer}
	 * 
	 * @see java.util.Map.Entry#setValue(java.lang.Object)
	 */
	@Override
	public NameContainer setValue(NameContainer value) {
		return this;
	}

}