/*
 * TastenKombination.java
 * eu.gronos.kostenrechner.model (Kostenrechner)
 */
package eu.gronos.beschriftungen.model;

import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import javax.swing.Action;
import javax.swing.KeyStroke;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import com.sun.glass.events.KeyEvent;

import eu.gronos.kostenrechner.Kostenrechner;

/**
 * Mit der Klasse soll eine Tastenkombination {@link KeyStroke} für ein
 * {@link Action#ACCELERATOR_KEY} in einer {@link LangBeschriftung} gespeichert
 * werden.
 *
 * @author Peter Schuster (setrok)
 * @date 21.08.2019
 *
 */
public class TastenKombination {

	private boolean altModifier = false;
	private boolean ctrlModifier = false;
	private boolean shiftModifier = false;
	private Integer keyCode = KeyEvent.VK_UNDEFINED;

	/**
	 * Konstruktor.
	 * 
	 */
	public TastenKombination() {
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("TastenKombination [");
		builder.append("keyCode=");
		builder.append(keyCode);

		// dahinter noch das Zeichen für bessere Lesbarkeit
//		builder.append("='");
//		builder.append(keyCode);//(getKeyChar());
//		builder.append("'");

		// Modifiers nur ausgeben, wenn gesetzt
		if (shiftModifier)
			builder.append(", shiftModifier");
		if (ctrlModifier)
			builder.append(", ctrlModifier");
		if (altModifier)
			builder.append(", altModifier");

		builder.append("]");
		return builder.toString();
	}

	/**
	 * Die Methode setzt nach dem als int übergebenen
	 * {@link KeyStroke#getModifiers()} die Felder {@link #isAltModifier()},
	 * {@link #isCtrlModifier()} und {@link #isShiftModifier()}.
	 * 
	 * @param modifiers die {@link KeyStroke#getModifiers()} alt <code>int</code>
	 */
	public void setAllModifiers(int modifiers) {
		if ((modifiers & InputEvent.SHIFT_DOWN_MASK) != 0) {
			setShiftModifier(true);
		} else {
			setShiftModifier(false);
		}
		if ((modifiers & InputEvent.CTRL_DOWN_MASK) != 0 || (modifiers & InputEvent.META_DOWN_MASK) != 0) {
			setCtrlModifier(true);
		} else {
			setCtrlModifier(false);
		}
		if ((modifiers & InputEvent.ALT_DOWN_MASK) != 0) {
			setAltModifier(true);
		} else {
			setAltModifier(false);
		}
	}

	/**
	 * Die Methode baut aus den Feldern {@link #isAltModifier()},
	 * {@link #isCtrlModifier()} und {@link #isShiftModifier()} einen
	 * {@link KeyStroke#getModifiers()} als int
	 * 
	 * @return {@link KeyStroke#getModifiers()} als int
	 */
	public int modifierIntValue() {
		int ctrlMask = Kostenrechner.ctrlMask;
		if (ctrlMask == ActionEvent.CTRL_MASK)
			ctrlMask = InputEvent.CTRL_DOWN_MASK;
		int modifiers = 0;
		if (isShiftModifier())
			modifiers += InputEvent.SHIFT_DOWN_MASK;
		if (isCtrlModifier())
			modifiers += ctrlMask;
		if (isAltModifier())
			modifiers += InputEvent.ALT_DOWN_MASK;
		return modifiers;
	}

	/**
	 * Die Methode wandelt die {@link TastenKombination} in einen {@link KeyStroke}
	 * um.
	 * 
	 * @return einen {@link KeyStroke}
	 */
	public KeyStroke toKeyStroke() {
		return KeyStroke.getKeyStroke(getKeyCode(), modifierIntValue());
	}

//	@XmlAttribute(name = "taste")
	// @XmlJavaTypeAdapter(VKAdapter.class) // TODO muss wohl eher an anderes Feld
//	public String getKeyChar() {
//		if (hasKeyCode()) {
//			return KeyEvent.getKeyText(keyCode);
//		} else
//			return null;
//	}

//	public void setKeyChar(String chars) {
//		if (chars == null) {
//			setKeyCode(0);
//			return;
//		}
//		if (chars.trim().length() == 1) {// !chars.isEmpty()
//			setKeyCode(KeyEvent.getExtendedKeyCodeForChar(chars.charAt(0)));
//		} else if (chars.length() > 1) {
//			setKeyCode(KeyStroke.getKeyStroke(chars).getKeyCode());
//		} else
//			setKeyCode(0);
//	}

	/**
	 * @return ob ein {@link #getKeyChar()} vorliegt.
	 */
	public boolean hasKeyCode() {
		return getKeyCode() != KeyEvent.VK_UNDEFINED;// > 0;
	}

	/**
	 * Die Methode gibt die Taste aus. Das kann 'A', 'B' oder auch 'F1' sein.
	 * 
	 * @return einen {@link Integer}
	 * 
	 * @see javax.swing.KeyStroke#getKeyStroke(String)
	 * @see java.awt.event.KeyEvent#getExtendedKeyCodeForChar(int)
	 */
	@XmlAttribute(name = "taste")
	@XmlJavaTypeAdapter(VKAdapter.class)
	public Integer getKeyCode() {
		return keyCode;
	}

	/**
	 * Die Methode baut aus einem char (eigentlich einem {@link String}) einen
	 * KeyCode.
	 * 
	 * @param chars ein String mit einem einzigen Zeichen, wie in
	 *              {@link KeyEvent#getExtendedKeyCodeForChar(int)} beschrieben,
	 *              oder mit der Bezeichnung einer Spezialtaste (DELETE), wie in
	 *              {@link KeyStroke#getKeyStroke(String)} beschrieben.
	 * 
	 * @see javax.swing.KeyStroke#getKeyStroke(String)
	 * @see java.awt.event.KeyEvent#getExtendedKeyCodeForChar(int)
	 */
	public void setKeyCode(Integer keyCode) {
		this.keyCode = keyCode;
	}

	/**
	 * @return gibt {@link #shiftModifier} als {@link boolean} zurück, also ob die
	 *         Umschalttaste gedrückt ist, {@link InputEvent#SHIFT_DOWN_MASK}.
	 */
	@XmlAttribute(name = "shift")
	public boolean isShiftModifier() {
		return shiftModifier;
	}

	/**
	 * @param shiftModifier d. {@link #shiftModifier}, d. gesetzt werden soll als
	 *                      {@link boolean}.
	 */
	public void setShiftModifier(boolean shiftModifier) {
		this.shiftModifier = shiftModifier;
	}

	/**
	 * @return gibt {@link #ctrlModifier} als {@link boolean} zurück, also ob
	 *         Ctrl/Strg {@link InputEvent#CTRL_DOWN_MASK} gedrückt ist bzw. auf
	 *         Macs {@link InputEvent#META_DOWN_MASK}
	 *
	 */
	@XmlAttribute(name = "ctrl")
	public boolean isCtrlModifier() {
		return ctrlModifier;
	}

	/**
	 * @param ctrlModifier d. {@link #ctrlModifier}, d. gesetzt werden soll als
	 *                     {@link boolean}.
	 */
	public void setCtrlModifier(boolean ctrlModifier) {
		this.ctrlModifier = ctrlModifier;
	}

	/**
	 * @return gibt {@link #altModifier} als {@link boolean} zurück, also ob die
	 *         Alt-Taste gedrückt ist, {@link InputEvent#ALT_DOWN_MASK}.
	 */
	@XmlAttribute(name = "alt")
	public boolean isAltModifier() {
		return altModifier;
	}

	/**
	 * @param altModifier d. {@link #altModifier}, d. gesetzt werden soll als
	 *                    {@link boolean}.
	 */
	public void setAltModifier(boolean altModifier) {
		this.altModifier = altModifier;
	}
}
