/*
 * IntegerConfigOption.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.beschriftungen.util;

import eu.gronos.beschriftungen.model.IntegerConfigSchluessel;

/**
 * Eine Klasse für den Zugriff auf Registry oder Benutzereinstellungen mit
 * {@link Integer}-Werten. Sie speichert den Ast /eu/gronos/kostentenor und den
 * Schlüssel. Jede Instanz korrespondiert mit einem Boolean-Wert
 * <code>schluessel</code> in der Registry im Unterast
 * <code>"/eu/gronos/kostentenor"</code> zum Java-Ast. Diese wird zunächst
 * eingelesen. Wenn der Schlüssel <code>schluessel</code> noch nicht gesetzt
 * ist, wird der als <code>standardWert</code> übergebene Wahrheitswert
 * genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 16.03.2020
 *
 */
public class IntegerConfigOption extends PreferencesConfigOption<Integer> {

	/**
	 * @param parameter ein {@link IntegerConfigSchluessel}
	 */
	public IntegerConfigOption(IntegerConfigSchluessel parameter) {
		super(parameter);
		setValue(ausRegistryLesen());
	}

	/**
	 * Schreibt den übergebenen {@link Integer}-Wert in den Java-Zweig der Registry,
	 * und zwar in den Schlüssel, der dem Konstruktor übergeben wurde. Wird von
	 * {@link #setValue(Object)} aufgerufen.
	 * 
	 * @param wert der {@link Integer}-Wert, der in die Registry soll
	 * 
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 * 
	 * @see eu.gronos.beschriftungen.util.PreferencesConfigOption#inRegistrySchreiben(java.lang.Object)
	 */
	@Override
	protected void inRegistrySchreiben(Integer wert) {
		getUserprefs().putInt(getSchluessel(), wert);
		logInfo(String.format("%s", wert));
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde. Wird vom Konstruktor aufgerufen.
	 * 
	 * @return den {@link Integer}-Wert aus der Registry oder den
	 *         {@link #getStandardWert()}, der dem Konstruktor mitgegeben wurde
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 * 
	 * @see eu.gronos.beschriftungen.util.PreferencesConfigOption#ausRegistryLesen()
	 */
	@Override
	protected Integer ausRegistryLesen() {
		return getUserprefs().getInt(getSchluessel(), getStandardWert());
	}

}
