/*
 * StringConfigOption2.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.beschriftungen.util;

import eu.gronos.beschriftungen.model.StringConfigSchluessel;

/**
 * Eine Klasse für den Zugriff auf Registry oder Benutzereinstellungen mit
 * {@link String}-Werten. Sie speichert den Ast
 * <code>/eu/gronos/kostentenor</code> und den Schlüssel. Jede Instanz
 * korrespondiert mit einem Boolean-Wert <code>schluessel</code> in der Registry
 * im Unterast <code>"/eu/gronos/kostentenor"</code> zum Java-Ast. Diese wird
 * zunächst eingelesen. Wenn der Schlüssel <code>schluessel</code> noch nicht
 * gesetzt ist, wird die als <code>standardWert</code> übergebene Zeichenkette
 * genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 14 Jun 2019
 *
 */
public class StringConfigOption extends PreferencesConfigOption<String> {

	/**
	 * Der Konstruktor erfordert den Wert für den finalen {@link String}
	 * {@link #getSchluessel()}
	 * 
	 * @param parameter ein {@link StringConfigSchluessel}
	 * 
	 */
	public StringConfigOption(StringConfigSchluessel parameter) {
		super(parameter);
		setValue(ausRegistryLesen());
	}

	/**
	 * Schreibt den übergebenen String in den Java-Zweig der Registry, und zwar in
	 * den Schlüssel, der dem Konstruktor übergeben wurde. Wird von
	 * {@link #setValue(Object)} aufgerufen.
	 * 
	 * @param wert ein {@link String} mit dem neuen Wert
	 * 
	 * @see eu.gronos.beschriftungen.util.PreferencesConfigOption#inRegistrySchreiben(java.lang.Object)
	 */
	@Override
	protected void inRegistrySchreiben(String wert) {
		getUserprefs().put(getSchluessel(), wert);
		logInfo(wert);
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde. Wird vom Konstruktor aufgerufen.
	 * 
	 * @return einen {@link String}
	 * 
	 * @see eu.gronos.beschriftungen.util.PreferencesConfigOption#ausRegistryLesen()
	 */
	@Override
	protected String ausRegistryLesen() {
		return getUserprefs().get(getSchluessel(), getStandardWert());
	}

}
