package eu.gronos.beschriftungen.view;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Window;
import java.awt.event.FocusAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.border.EmptyBorder;
import javax.swing.text.JTextComponent;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.controller.BeiFocusMarkierenHorcher;
import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.controller.DialogFensterSchliessenHorcher;
import eu.gronos.beschriftungen.controller.HinzufuegenAbbrechenAction;
import eu.gronos.beschriftungen.interfaces.Vorsorgend;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.GitterBeutelBeschraenkungen;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.LangVorsorgeBeschriftung;
import eu.gronos.beschriftungen.model.NameContainer;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.model.UnBeschriftung;

/**
 * Abstrakte Oberklasse zum HinzufügenDialog und für Fenster, die keine Werte
 * zurückgeben sollen. Hieß mal KostenJDialog.
 *
 * @author Peter Schuster (setrok)
 * @date 24.01.2020
 *
 */
public abstract class BeschrifteterJDialog extends JDialog {

	private static final long serialVersionUID = -520050986950014909L;

	private final BeschriebeneAktion cancelAction = new HinzufuegenAbbrechenAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(76061));
	protected final WindowAdapter fensterSchliesser = new DialogFensterSchliessenHorcher(this);
	public static final FocusAdapter BEI_FOCUS_MARKIEREN = new BeiFocusMarkierenHorcher();
	protected static final EmptyBorder BORDER = new EmptyBorder(5, 5, 5, 5);
	protected final FontHelferForWindow fufhelfer = new FontHelferForWindow(this);
	/**
	 * Der JPanel mit dem Namen contentPanel ist das JPanel für den eigentlichen
	 * Inhalt
	 */
	protected final JPanel contentPanel = new JPanel();
	private JPanel buttonPane;
	private final Beschriftung beschriftung;
	protected final ComponentBeschrifter beschrifter = new ComponentBeschrifter();

	/**
	 * Erstellt einen modalen {@link JDialog} mit einer buttonPane (nur Abbrechen
	 * bzw. Schließen) und einem {@link #contentPanel}.
	 * 
	 * Dieser muss von der abgeleiteten Klasse in der Methode
	 * {@link #fuelleContentPane()} mit {@link JComponent}s gefüllt werden, die von
	 * {@link #showDialogAndAsk()} aufgerufen wird.
	 * 
	 * @param owner        das {@link Window} ({@link JFrame} oder {@link JDialog})
	 *                     von dem aus der {@link BeschrifteterJDialog} angezeigt
	 *                     wird
	 * @param beschriftung eine {@link Beschriftung}, mit dem der anzuzeigende Titel
	 *                     {@link JDialog#getTitle()} gesetzt wird
	 * @throws HeadlessException wenn in einer Umgebung aufgerufen, die keinen
	 *                           Bildschirm, Tastatur oder Maus unterstützt
	 */
	public BeschrifteterJDialog(Window owner, Beschriftung beschriftung) {
		super(owner);
		this.beschriftung = beschriftung;
		// Kostenrechner.getLogger().info("Starte Dialog " +
		// this.getClass().toString());
		beschrifter.beschrifte(this, beschriftung);
		setContentPanel();
		setButtonPane();
		super.setModal(true);
	}

	/**
	 * Die Methode lässt die Position vom Betriebssystem zuweisen und setzt die
	 * Größe auf die bevorzugte Größe. Wird von {@link #showDialogAndAsk()}
	 * aufgerufen.
	 * 
	 * @see java.awt.Window#setBounds(int,int,int,int)
	 * @see java.awt.Window#setLocationByPlatform(boolean)
	 * @see java.awt.Window#setSize(Dimension)
	 * @see java.awt.Window#pack()
	 */
	@Override
	public void pack() {
		setLocationByPlatform(true);
		setSize(fufhelfer.calculatePreferredSizeWith(BORDER.getBorderInsets()));//
		setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		addWindowListener(fensterSchliesser);
		super.pack();
	}

	/**
	 * Die Methode dient dazu, den Dialog aufzurufen, den {@link #contentPanel} über
	 * die Methode {@link #fuelleContentPane()} der abgeleiteten Klasse mit
	 * Komponenten zu füllen und mit {@link JDialog#setVisible(boolean)} den Dialog
	 * anzuzeigen.
	 */
	public void showDialog() {
		fuelleContentPane();
		pack();
		setVisible(true);
	}

	/**
	 * Die Methode schließt den Dialog; kann in einer {@link Action} benutzt werden
	 * 
	 */
	public void fensterSchliessenBeiAbbruch() {
		setVisible(false);
		dispose();
	}

	/**
	 * @return gibt {@link #beschriftung} als {@link Beschriftung} zurück.
	 */
	public Beschriftung getBeschriftung() {
		return beschriftung;
	}

	/**
	 * @return gibt {@link #cancelAction} als {@link AbstractAction} zurück.
	 */
	public BeschriebeneAktion getCancelAction() {
		return cancelAction;
	}

	/**
	 * @return gibt den buttonPane als JPanel zurück.
	 */
	public JPanel getButtonPane() {
		return buttonPane;
	}

	/**
	 * Neue Methode setzt in der Basisklasse {@link NameContainer} nur
	 * comp.setName(), bei einer {@link Beschriftung} weiter den Text/Title/Label
	 * und die shortDescription/toolTip, holt sich bei {@link UnBeschriftung} aus
	 * {@link UnBeschriftung#getRef()} den Title und die shortDescription und setzt
	 * die auch und fügt das ganze dann noch in {@link #getContentPane()} ein
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit der {@link NameContainer}
	 * @return
	 */
	protected Component toContentPanel(Component das, NameContainer damit) {
		return beschrifter.beschrifteVergittert(das, damit, contentPanel);
//		return beschrifter.beschrifteVergittert(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(35021),
	}

	/**
	 * @param das
	 * @param damit
	 * @return
	 */
	protected Component togetherToContentPanel(Component das, UnBeschriftung damit) {
		return beschrifter.beschrifteGemeinsam(das, damit, contentPanel);
	}

	/**
	 * @param dasLabel
	 * @param das
	 * @param damit
	 * @return
	 */
	protected Component toContentPanel(JLabel dasLabel, Component das, UnBeschriftung damit) {
		return beschrifter.beschrifteGemeinsam(dasLabel, das, damit, contentPanel);
	}

	/**
	 * Zieht aus einer {@link BeschriebeneAktion} mit
	 * {@link BeschriebeneAktion#toButton()} einen {@link JButton} und fügt diesen
	 * in die {@link #getButtonPane()} und über {@link #addToRootInputMap(Action)}
	 * in die {@link JRootPane#getActionMap()} ein.
	 * 
	 * @param das die {@link BeschriebeneAktion}, aus der die Beschriftung und ggfs.
	 *            der {@link GitterBeutelBeschraenkungen} gezogen werden
	 * @return den erzeugten {@link JButton} als {@link Component}
	 */
	protected Component toButtonPanel(BeschriebeneAktion das) {
		addToRootInputMap(das);
		return beschrifter.add(das, buttonPane);
	}

	/**
	 * Die Methode setzt die Abbrechen-Schaltfläche in einen eigenen JPanel des
	 * Dialogs
	 * 
	 */
	protected void setButtonPane() {
		buttonPane = new JPanel();
		buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT));
		getContentPane().add(buttonPane, BorderLayout.SOUTH);

		JButton cancelButton = ((BeschriebeneAktion) getCancelAction()).toButton();
		// Im Grundfall muss die Schaltfläche nur "schließen" lauten
		changeCancelBeschriftung(1);

		buttonPane.add(cancelButton);
		addToRootInputMap(getCancelAction());
		getRootPane().setDefaultButton(cancelButton);
	}

	/**
	 * Mit dieser Methode können die von {@link BeschrifteterJDialog} abgeleiteten
	 * Klassen bei der {@link #getCancelAction()} zwischen der Beschriftung
	 * Schließen bzw. Abbrechen wechseln.
	 * 
	 * @param index siehe {@link Vorsorgend#changeTo(ComponentBeschrifter, int)}
	 */
	protected void changeCancelBeschriftung(int index) {
		LangVorsorgeBeschriftung beschriftung = (LangVorsorgeBeschriftung) getCancelAction().getBeschriftung();
		// ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.changeTo(beschriftung, index);
		// beschriftung.applyTo(getCancelAction(), beschrifter);
		beschrifter.beschrifte(getCancelAction(), beschriftung);
	}

	/**
	 * Die Methode füllt den Inhaltsbereich des Dialogs mit JComponents zur Eingabe.
	 * Wird von {@link #showDialogAndAsk()} aufgerufen.
	 * 
	 */
	protected abstract void fuelleContentPane();

	/**
	 * Die Methode initialisiert die allgemeinen JComponents für den Inhaltsbereich
	 * des Dialogs und wird vom Konstruktor selbsttätig aufgerufen.
	 * 
	 */
	protected void setContentPanel() {
		GridBagLayout gbl_contentPanel = new GridBagLayout();
		contentPanel.setBorder(BORDER);
		contentPanel.setLayout(gbl_contentPanel);
		getContentPane().add(contentPanel, BorderLayout.CENTER);
	}

	/**
	 * Die Methode fügt eine Action der ActionMap und der InputMap der JRootPane des
	 * HinzufuegenDialogs hinzu.
	 * 
	 * @param action eine Action, bei der die Values
	 *               {@link Action#ACTION_COMMAND_KEY} und
	 *               {@link Action#ACCELERATOR_KEY} gesetzt sein müssen.
	 */
	protected void addToRootInputMap(Action action) {
		JRootPane rp = getRootPane();
		/* Zunächst die Action der ActionMap hinzufügen unter ihrem Namen */
		rp.getActionMap().put(action.getValue(Action.ACTION_COMMAND_KEY), action);
		/* Dann die Action der InputMap unter ihrer Tastenkombination hinzufügen */
		rp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put((KeyStroke) action.getValue(Action.ACCELERATOR_KEY),
				action.getValue(Action.ACTION_COMMAND_KEY));
	}

	/**
	 * Die Methode entfernt eine Action aus der ActionMap und der InputMap der
	 * JRootPane des HinzufuegenDialogs.
	 * 
	 * @param action eine Action, bei der die Values
	 *               {@link Action#ACTION_COMMAND_KEY} und
	 *               {@link Action#ACCELERATOR_KEY} gesetzt sein müssen.
	 */
	protected void removeFromInputMap(Action action) {
		JRootPane rp = getRootPane();
		/* Zuerst die Tastenkombination entfernen */
		rp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).remove((KeyStroke) action.getValue(Action.ACCELERATOR_KEY));
		/* Dann die Action entfernen */
		rp.getActionMap().remove(action.getValue(Action.ACTION_COMMAND_KEY));
	}

	/**
	 * Die Methode macht den {@link JTextComponent#getCaret()} der {@link #textPane}
	 * sichtbar, im Dienste der Barrierefreiheit
	 * 
	 * @param textComponent die {@link JTextComponent}
	 */
	protected void caretVisible(JTextComponent textComponent) {
		textComponent.getCaret().setVisible(true);
	}

	/**
	 * Setzt für ein {@link JFormattedTextField} die {@link AbstractAction} in die
	 * ActionMap und InputMap, die bei Enter ausgeführt werden soll. Zudem fügt die
	 * Methode das Verhalten hinzu, dass bei FocusErlangung der Inhalt markiert
	 * wird.
	 * 
	 * @param field  das {@link JTextField}, dessen Entertaste belegt werden soll
	 * @param action die {@link AbstractAction}, die bei Entertaste ausgeführt
	 *               werden soll
	 */
	protected void setEnterAction(JTextField field, AbstractAction action) {
		if (action != null) {
			field.getInputMap(JComponent.WHEN_FOCUSED).put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
					action.getValue(Action.ACTION_COMMAND_KEY));
			field.getActionMap().put(action.getValue(Action.ACTION_COMMAND_KEY), action);
		}
		field.setHorizontalAlignment(SwingConstants.RIGHT);
		field.addFocusListener(BEI_FOCUS_MARKIEREN);
	}

}