/*
 * FontHelfer.java
 * eu.gronos.kostenrechner.view.result (Kostenrechner)
 */
package eu.gronos.beschriftungen.view;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GraphicsEnvironment;
import java.awt.Insets;

import javax.swing.JComponent;
import javax.swing.JDialog;

/**
 * Die Klasse dient dazu, Fonts zu suchen, Formatierungen zu setzen und die
 * Fenstergröße nach Schriftgröße zu setzen.
 *
 * @author Peter Schuster (setrok)
 * @date 19 Jun 2019
 *
 */
public class FontHelfer {

	/**
	 * maximale Anzahl der Zeilen für die Größe des Dialogfensters
	 */
	public static final int MAX_ZEILEN = 40;
	/**
	 * maximale Anzal der Spalten für die Größe des Dialogfensters
	 */
	public static final int MAX_SPALTEN = 150;
	public static double zoom = 1.33;

	/**
	 * Konstruktor.
	 * 
	 */
	public FontHelfer() {
		super();
	}

	/**
	 * Die Methode dient dazu, die Breite eines Zeichens ('n') und die
	 * Standardzeilenhöhe zu ermitteln
	 * 
	 * @param fontMetrics
	 * @return Zeichenbreite und Zeilenhöhe in Pixeln als {@link Dimension}, also
	 *         eine {@link Dimension} aus {@link FontMetrics#charWidth(int)} und
	 *         {@link FontMetrics#getHeight()}
	 */
	public Dimension getCharDimension(FontMetrics fontMetrics) {
		int charWidth = fontMetrics.charWidth('n');// 'm'
		int lineHeight = fontMetrics.getHeight();
//		Kostenrechner.getLogger()
//				.info("Breite eines 'n':" + charWidth + "; Höhe des Fonts: " + fontMetrics.getHeight());
		return new Dimension(charWidth, lineHeight);
	}

	/**
	 * Die Methode errechnet für eine {@link JComponent} die Maße, dass sie nicht
	 * breiter als maxSpalten Zeichen und nicht höher als maxZeilen Zeilen wird.
	 * @param maxSpalten    Spaltenzahl für die Berechnung
	 * @param maxZeilen     Zeilenanzahl für die Berechnung
	 * @param charDimension Breite und Höhe eines Buchstaben als {@link Dimension},
	 *                      {@link #getCharDimension(FontMetrics)}
	 * @param maximumSize   {@link JComponent#getMaximumSize()}
	 * @param preferredSize {@link JComponent#getPreferredSize()}
	 */
	public Dimension calculatePreferredSizeFrom(int maxSpalten, int maxZeilen, Dimension charDimension,
			Dimension maximumSize, Dimension preferredSize) {
		/* Breite */
		if (charDimension.width > 0 && (charDimension.width * maxSpalten) < maximumSize.width) {
			preferredSize.width = charDimension.width * maxSpalten;
		} else {
			preferredSize.width = maximumSize.width;
		}
		/* Höhe */
		if (charDimension.height > 0 && (charDimension.height * maxZeilen) < maximumSize.height) {
			preferredSize.height = charDimension.height * maxZeilen;
		}
//		Kostenrechner.getLogger()
//				.info("maximumSize:" + maximumSize.toString() + "; preferredSize: " + preferredSize.toString());
		return preferredSize;
	}

	/**
	 * Die Methode ermittelt die bevorzugte Größe des {@link JDialog}s, indem sie
	 * die Größe der Ränder hinzurechnet.
	 * 
	 * @param insets    {@link Insets} des {@link JDialog}s
	 * @param minimum   {@link JComponent#getMinimumSize()}
	 * @param preferred {@link JComponent#getPreferredSize()}
	 * @return eine {@link java.awt.Dimension} mit der vergrößerten bevorzugten
	 *         Größe
	 */
	public Dimension calculatePreferredSizeWith(Insets insets, final Dimension minimum, final Dimension preferred) {
		// Insets insets = border.getBorderInsets();
		double rand = (double) (insets.left + insets.right + insets.top + insets.bottom);
		// Liefert getMinimumSize() oder getPreferredSize() den größeren Wert
		// für Breite und Höhe?
		// Nimm die größere Breite
		double width = minimum.getWidth() > preferred.getWidth() ? minimum.getWidth() : preferred.getWidth();
		// Nimm die größere Höhe
		double height = minimum.getHeight() > preferred.getHeight() ? minimum.getHeight() : preferred.getHeight();
		// Rand hinzurechnen
		width += rand;
		height += rand;
		return new Dimension((int) width, (int) height);
	}

	/**
	 * Die Methode getFontForName dient dazu, herauszufinden, ob ein Font mit dem
	 * als <code>name</code> übergebenen Namen im System existiert.
	 * 
	 * @param name Suchbegriff Name des Fonts
	 * @return den Font des übergebenen Namens, falls gefunden.
	 */
	public Font getFontForName(String name) {
		if (name == null || "".equals(name))
			return null;
		Font f = null;
		for (String fonts : GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames())
			if (name.equals(fonts))
				f = new Font(fonts, Font.PLAIN, 12);
		return f;
	}
}
