
package eu.gronos.kostenrechner;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.awt.event.WindowListener;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Logger;

import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.border.EmptyBorder;
import javax.xml.bind.JAXB;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.NameContainerMap;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.view.FontHelferForWindow;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.controller.system.LizenzNoergler;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTabelle;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.system.ArgumentParser;
import eu.gronos.kostenrechner.view.KostenJPanel;
import eu.gronos.kostenrechner.view.OsLookAndFeelChanger;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;
import eu.gronos.kostenrechner.view.baumbach.StreitgenossenPanel;
import eu.gronos.kostenrechner.view.forderungen.ForderungsStaffelungPanel;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenBerechnungPanel;
import eu.gronos.kostenrechner.view.gebuehren.TeilklageruecknahmePanel;
import eu.gronos.kostenrechner.view.helpanderror.FehlerDialog;
import eu.gronos.kostenrechner.view.helpanderror.HilfePanel;
import eu.gronos.kostenrechner.view.menus.BearbeitenJMenu;
import eu.gronos.kostenrechner.view.menus.DateiJMenu;
import eu.gronos.kostenrechner.view.menus.ExtrasJMenu;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Mit dem Tool k&ouml;nnen Hauptsache- und Kostentenor in einfachen F&auml;llen
 * einer Teilklager&uuml;cknahme, einem streitgenossenschaftlichem
 * Teilunterliegen und einer Staffelung von Hauptklage und Hilfsaufrechnung
 * erzeugt werden.
 * 
 * @author Peter Schuster (setrok)
 * @version 0.7
 * 
 * @since 0.7 (2025-06-03) Euro, Fraction, VerlusteBank, enums statt Konstanten
 *        in Beteiligter, höhere Berufungsgrenze.
 * 
 * @since 0.6.4 (2021-01-06) Tab-Verhalten in Tabellen. Anpassung unter Mac bei
 *        Umgebungen mit JRE > 8; Gebühren (hoehe) sind jetzt double, Änderung
 *        durch KostBRÄG 2025 ab 01.06.2025 beachtet.
 *
 * @since 0.6.3 (2020-12-13) XML-String-Verkleinerung für E-Mails. Cursor in
 *        großen Textfeldern (Tenor, Hilfe) sichtbar für Barrierefreiheit.
 *        Kostenrechtsänderungsgesetz 2021 vom 21.12.2020 eingebunden.
 * 
 * @since 0.6.2 (2020-04-18) Paket-Struktur weiter aufgeteilt und um die Äste
 *        *.data.* und *.util.* erweitert. Externes JAXB eingebunden.
 *        Vorkehrungen für Kostenrechtsänderung 2021.
 * 
 * @since 0.6.1 (2020-03-16) Auswahl des Schwellenwerts für § 92 II ZPO. Bei
 *        Gebührenaufstellung ist sofort die Begründung aktiv.
 * 
 * @since 0.6 (2019-10-12) Umstieg auf Java 1.8. Aktionen-Menü weiter aufgeteilt
 *        in Bearbeiten und Extras. RTF-Export aufgemotzt.
 * 
 * @since 0.5.9 (2019-06-03) die Laden-/Speicher-Dialoge merken sich den Pfad
 *        und schreiben ihn in die Registry. Aktives JPanel wird im VDC
 *        gespeichert. Überflüssiges Nebeneinander zwischen Ok- und
 *        Weiter-Schaltfläche im BaumbachAssistentDialog beendet.
 *        Gebühr-Hinzufügen-Dialog aufgehübscht und mit Suchfeld versehen.
 *        Gesamtschuldner-Dialog aufgehübscht und mit
 *        Gesamtschuldnerschaft-Extraktion versehen. Umstieg auf
 *        {@link Beschriftung} für Accessibility.
 * 
 * @since 0.5.8 (2018-12-07) Fehler aus 0.5.7 beseitigt der die Widerklage im
 *        Baumbach-Panel nicht mehr sichtbar machte. Fehler beim Drag&Drop bei
 *        Teilklagerücknahme beseitigt. Die Auswahl, ob §92iiZPO berücksichtigt
 *        werden soll, wird in der Registry rsp. den Preferences gespeichert.
 *        Bei der Teilklagerücknahme lässt eine Schaltfläche die Quotenmethode
 *        oder Mehrkostenmethode vorauswählen und die Schaltfläche Tenor
 *        ermitteln arbeitet dann immer nach der ausgewählten Methode.
 *        Datenschutzmitteilung vor dem Versand von E-Mails. TenorXml kann
 *        mitgespeichert werden.
 * 
 * @since 0.5.7 (2017-12-28) Fehler in Kostenrechner behoben, durch den
 *        formatierte Zahlen bei Teilklagerücknahme (zB 10,000) wegen des Kommas
 *        nicht geparst werden konnten. Auf mehrere Packages aufgeteilt. JPanels
 *        zu einzelnen Klassen mit Oberklasse KostenJPanel gemacht. Dateimenü
 *        (neu), Aktionenmenü und RechnerhammerIcons zu eigenen Klassen gemacht.
 *        Die berechnenden Klassen bekommen jetzt einheitlich einen Container
 *        als Parameterobjekt. {@link TenorDialog} setzt Größe jetzt über
 *        PreferredSize und kann mit ESC verlassen werden. Neue Klasse
 *        {@link BegruendungsZahlenTabelle}, die die Daten hält, die ein Tabulierend
 *        braucht, als Vorbereitung für den XML-Export der Tenordaten über
 *        {@link TenorDatenContainer}. Erste JUnit-Tests. Parsende Methoden aus
 *        der Hauptklasse in ParserHilfe und die zum Verarbeiten des VDC in
 *        VerfahrensDatenContainer2GUI extrahiert. Ausgaben jetzt über Logger.
 *        Die Gebührenaufstellung liefert auch Gründe. Fehler bei der
 *        Vollstreckbarkeits-Tenorierung behoben. Sprechendere Labels im Panel
 *        {@link TeilklageruecknahmePanel}.
 * 
 * @since 0.5.6 (2017-05-01) Fehler in TenorDialog behoben, der ein
 *        Speichern/Kopieren verhinderte (NullPointer), wenn eine Begründung
 *        hinzugefügt war. Fehler in GebuehrenTabelle.errechneStreitwertListe
 *        behoben, der dafür sorgte, dass die Tabelle bei 0 EUR statt dem ersten
 *        Sprung begann.
 * 
 * @since 0.5.5 (2016-03-28) E-Mail-Schaltfläche im TenorDialog, die auch den
 *        Tenor und die XML-Daten mitschickt; E-Mail-Menüpunkt im Hauptmenü;
 *        E-Mail-Versand bei Fehlermeldungen.
 * 
 * @since 0.5.2 und 0.5.3 Fehlerbeseitigung (0.5.4 übersprungen)
 * 
 * @since 0.5.1 Fehlerbeseitigung. {@link GebuehrenTabelle} fit für {@link JAXB}
 *        gemacht.
 * 
 * @since 0.5 Auch bei Teilklagerücknahme wird auf Wunsch § 92 II ZPO beachtet.
 * 
 * @since 0.4.8 VerfahrensDatenContainer durch einen neuen ersetzt. Beim Laden
 *        von Xjustiz wird auch GenusNumerus der Beteiligten-Auswahllisten auf
 *        den anderen Registerkarten richtig gesetzt. SKKTX-Dateien können als
 *        Kommandozeilen-Parameter übergeben werden. Der
 *        {@link XmlTransferHandler} nimmt jetzt zum Drag'n'Drop auch
 *        skktx-Dateien und von allen Registerkarten.
 * 
 * @since 0.4.7 Laden und speichern möglich (skktx - xml).
 * 
 * @since 0.4.6 Breite des TenorDialogs auf 120 Spalten begrenzt. Links sind
 *        klickbar.
 * 
 * @since 0.4.5 Vorläufige Vollstreckbarkeit auch bei der Baumbach'schen
 *        Kostenformel
 * 
 * @since 0.4.2 Neuer "Keine-Umsatzsteuer"-Tatbestand. In der Gebührentabelle
 *        können Einträge verschoben werden. Langfassung der GPL kann angezeigt
 *        werden.
 * 
 * @since 0.4.1 Bugfix: Drag'n'Drop funktionierte nicht mehr mit
 *        Auslagetatbeständen. AngriffsVerteidigungsmittelStaffelung und
 *        StreitwertReduktion haben Vollstreckbarkeitstenor.
 * 
 * @since 0.4 Neue Klassen VollstreckbarkeitsObjekt und VollstreckbarkeitsListe.
 *        Freigabe unter GPL 3.0. Annahme der GPL zum Programmstart n&ouml;tig.
 * 
 * @since 0.3.1 Bugfix beim Setzen des Streitwerts durch BaumbachAssistenten.
 *        Durch FocusListener wird jede Änderung des niedrigeren Streitwerts
 *        nach Letztlich Obsiegt zurück geschrieben. Bessere Wortstellung im
 *        Hauptsachetenor.
 * 
 * @since 0.3.0 GebührenTatbestand als Basisklasse war in StreitwertReduktion
 *        und Kindklassen noch nicht angekommen; jetzt aber schon. :) Neues
 *        Interface Tabulierend und neue Klasse TabulierendTableModel, von
 *        AngriffsVerteidigungsmittelStaffelung, BaumbachFormel und
 *        StreitwertReduktion unterstützt. Der tabellarische Teil der Begründung
 *        des Tenors wird jetzt auch als Tabelle eingefügt.
 * 
 * @since 0.2.9 (2014-09-19) Eingabefelder für Zahlen jetzt rechtsbündig. Bei
 *        Xjustiz-Import (und bei sonstiger Verwendung des Assistenten) werden
 *        auch die Streitwert-/Obsiegen-Felder bei Teilklagerücknahme und bei
 *        der freien Gebührenberechnung gefüllt.
 * 
 * @since 0.2.8 (2014-08-21) Tenorierend aufgeteilt in simpleres Ober-Interface
 *        Tenorierend und abgeleitetes Interface UrteilsTenorierend mit dem
 *        gewohnten Umfang. Umsatzsteuer auf Rechtsanwaltsverg&uuml;tung wird
 *        unterst&uuml;tzt. Neue Funktionalit&auml;t Geb&uuml;hrenaufstellung.
 *        Neue Klasse GebuehrenErhoehungsTatbestand. Neue Klasse FehlerDialog.
 *        Die OK-Schaltfläche kann automatisch den Focus bekommen. Mehr
 *        GebührenTatbestände. Neue Klasse BeschriebeneAktion, um bei
 *        AbstractActions bereits im Konstruktor values zu setzen. Bei
 *        UrteilsTenorierend kann man eine Begründung implementieren, was
 *        AngriffsVerteidigungsmittelStaffelung, BaumbachFormel und
 *        StreitwertReduktion auch machen. In TenorDialog kann man übers
 *        Kontextmenü die Begründung einfügen. Reset in
 *        AngriffsVerteidigungsmittelStaffelung eingebaut, damit die Berechnung
 *        auch noch ein zweites Mal (nach Änderung der Werte) klappt.
 * 
 * @since 0.2.7 (2014-08-17) Freien Gebührenrechner hinzugefügt. Die
 *        GebührenRechner geben eine Liste der Gebührensprünge als ArrayList
 *        heraus. Tabelle kann gedruckt und in die Zwischenablage kopiert
 *        werden.
 * 
 * @since 0.2.6 (2014-08-14) Konstanten extrahiert. Test-Unit erweitert.
 *        GebuehrenTabelle aufgeteilt in eine erweiterte
 *        ArrayList&lt;GebuehrenZeile&gt; namens GebuehrenTabelle und einen
 *        GebuehrenRechner als Oberklasse mit Berechnungsmethoden. Rundung beim
 *        Vergleich der identischen Quoten. GitterBeutelBeschränkungen zu einem
 *        Quasi-Singleton gemacht und leicht angepasst.
 * 
 * @since 0.2.5 (2014-08-11) Tooltips für Spaltenüberschriften.
 * 
 * @since 0.2.2 (2014-08-08) Tabellen weitergehend editierbar: Renderer und
 *        Editor für Widerklage (Boolean), DefaultEditor für Beteiligte. Mehr
 *        Plausis. Die Gesamtschuldnerschaft selbst kann entfernt werden.
 * 
 * @since 0.2.1 (2014-08-05) Bugfix im GttmNachher-Kontextmenü und
 *        setViewportFill bei allen Tabellen. ACTION_COMMAND_KEY in Actions.
 *        Kommentiert. JFormattedTextFields im Hauptfenster werden jetzt auch
 *        auf Focus markiert. Streitgenossen in Tabellen sind jetzt auch
 *        nachträglich editierbar.
 * 
 * @since 0.2.0 (2014-08-04) Verschieben von Gebühren in andere Tabelle über
 *        Kontextmenü und über Drag'n'Drop. Xjustiz-Import per Drag'n'Drop.
 * 
 * @since 0.1.9 (2014-08-02) BeteiligtenRenderer lässt lfdNr in Ruhe, wenn schon
 *        gesetzt. Registerkarte 3 (Hilfsaufrechnung) aktiviert und gefüllt.
 *        Staffelung mit Klage und Hilfsaufrechnung kann jetzt berechnet werden.
 *        Bezeichner in ComboBox doch wieder mit Artikel. Tabelle für
 *        Forderungen auch editierbar.
 * 
 * @since 0.1.8 (2014-07-29) GebührensatzRenderer. Beim Zurücksetzen wird das
 *        Anzeigen von Widerklagen jetzt auch richtig zurückgesetzt.
 *        Design-Fehler beim Skalieren im BaumbachAssistentDialog behoben. Mehr
 *        Plausis im Xjustiz-Import über BaumbachAssistenten. Neue Klassen
 *        Forderung, KlageForderung und AufrechnungsForderung. Neue Interfaces
 *        RowList und RowHandler. BeteiligtenRenderer und
 *        SimpleBeteiligtenRenderer zusammengeführt.
 * 
 * @since 0.1.7 (2014-07-24) Neuer Hilfetext, "darüber hinaus" auch mit
 *        geschachtelten Gesamtschuldnerschaften. Kontextmenü und "Eintrag
 *        entfernen" auch in den Gebühren-Tabellen. Komfortablere Eingabe der
 *        Beteiligtenzahlen im BaumbachAssistenten durch JSpinner. Beim Aufruf
 *        ist die Übergabe des Dateinamens einer XJustiz-Datei möglich.
 * 
 * @since 0.1.6 (2014-07-20) EuroRenderer auch im Hauptfenster, TenorDialog
 *        erweitert HinzufügenDialog (schönere Buttons, Fenstergrößere besser
 *        angepasst), Abfrage vor Beenden. BaumbachFormel sortiert die
 *        Klageabweisung (und "im Übrigen") besser in den Hauptsachetenor.
 *        "darüber hinaus" jetzt im Hauptsachetenor und im Kostentenor, aber nur
 *        für einfache BaumbachBeteiligten. Gleichlautende Kostenverteilung wird
 *        in der Baumbachformel im selben Satz ausgegeben. Kontextmenüs für die
 *        Beteiligten-JTables. Alles zurücksetzen setzt auch Widerklagen wieder
 *        zurück.
 * 
 * @since 0.1.5 (2014-07-17) BaumbachAssistent mit XJustiz-Import
 * 
 * @since 0.1.4 (2014-07-08) XJustiz-Import
 * 
 * @since 0.1.3 (2014-07-04) Gesamtschuldnerschaft extrahieren
 * 
 * @since 0.1.2 (2014-07-01) Tastenkürzel für Tabs, Tabellen in Tab-Reihenfolge
 * 
 * @since 0.1.1 (2014-06-29) Schönere Dialoge durch neue Oberklasse
 *        HinzufuegenDialog&lt;E&gt;. Bei Vorliegen einer Widerklage wird bei
 *        Streitwertbeschluss auch eine Summe gebildet, sofern Klage und
 *        Widerklage nicht wirtschaftlich identisch sind. Mehr Tastenkürzel und
 *        Kurzbeschreibungen.
 */
public class Kostenrechner extends JFrame {

	public static final String KOSTENRECHNER_VERSION = " v0.7";// -b007 -b022 -b056
	private static final long serialVersionUID = -8308234314555759118L;

	/**
	 * Der boolesche Wert gibt an, ob der {@link FehlerDialog} auch den
	 * {@link Throwable#getStackTrace() StackTrace} anzeigen soll.
	 */
	public static boolean debug = false;
	/**
	 * Der int mit dem Namen ctrlMask ermöglicht, dass für Windows u.a. die
	 * Strg-Taste bei Tastenkombinationen auf dem Mac durch Cmd ersetzt wird.
	 */
	public static int ctrlMask = ActionEvent.CTRL_MASK;// TODO: InputEvent.CTRL_DOWN_MASK?
	private JPanel contentPane;
	private JTabbedPane tabbedPane;
	public static final Clock systemDefaultZone = Clock.systemDefaultZone();
	private static Kostenrechner instance = null;
	private static final Logger logger = Logger.getLogger("eu.gronos.kostenrechner");

	private static String logFileName = "(kein)";
	private static Path logFile;

	/**
	 * Launch the application.
	 * 
	 */
	public static void main(final String[] args) {
		OsLookAndFeelChanger.pruefeOS();
		// Der Logger muss vor dem ArgumentParser gesetzt werden, weil der Parser schon
		// loggen will.
		addFileLogger();
		// Der ArgumentParser muss nach -debug suchen, bevor die GUI aufgebaut wird
		ArgumentParser argumeterParser = new ArgumentParser(args);
		debug = argumeterParser.parseDebug();
		if (debug) {
			getLogger().info("Debug aktiviert!");
		}
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				try {
					setInstance(new Kostenrechner());
					instance.setVisible(true);
					getLogger().info((String.format("Einlesen der Beschriftungen dauerte %d ms.",
							(NameContainerMap.end - NameContainerMap.start))));
					/*
					 * XJustiz-Datei kann als Parameter übergeben werden. Nach Aufbau des JFrame
					 * wird dann der XML-Parser und dann der Assistent gestartet.
					 */
					Path xml = null;
					if (args != null && args.length > 0) {
						xml = argumeterParser.parseDateiname();
						new XmlTransferHandler().nimmXmlDatei(xml);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		});
	}

	/**
	 * Constructs a new frame that is initially invisible. This constructor sets the
	 * component's locale property to the value returned by
	 * JComponent.getDefaultLocale.
	 */
	public Kostenrechner() {
		super();
		setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
//		setSize(849, 617);
		setLocationByPlatform(true);
		contentPane = new JPanel();
		contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
		setContentPane(contentPane);
		contentPane.setLayout(new BorderLayout(0, 0));

		setTabbedPane(new JTabbedPane(JTabbedPane.TOP));
		contentPane.add(getTabbedPane());

		ArrayList<Dimension> dimensionen = new ArrayList<>();

		dimensionen.addAll(Arrays.asList(fuelleStreitwertReduktionPanel()));
		dimensionen.addAll(Arrays.asList(fuelleBaumbachPanel()));
		dimensionen.addAll(Arrays.asList(fuelleHilfsaufrechnungPanel()));
		dimensionen.addAll(Arrays.asList(fuelleGebuehrenRechnerPanel()));
		dimensionen.add(getTabbedPane().getPreferredSize());

		dimensionen.add(fuelleHilfePanel());

		setSize(FontHelferForWindow.averageDimension(dimensionen));

		System.out.println("Größe des Fensters: " + getSize());

		baueMenuBar();

		definiereIcons();

		setTransferHandler(new XmlTransferHandler());

		// Jetzt zum DateiJMenu bewegt: addWindowListener(FENSTER_HORCHER);

		new LizenzNoergler(this).noergle();

		Beschriftung beschriftung = (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(100);
		beschriftung.setTitle(beschriftung.getTitle() + KOSTENRECHNER_VERSION);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.beschrifte(this, beschriftung);

		beschrifter.beschrifte(tabbedPane, NameContainerSammlung.BESCHRIFTUNGEN.get(111));
	}

	// private List<Dimension> tabDimensions

//	/**
//	 * @param size
//	 * @param preferredSize
//	 * @return
//	 */
//	private Dimension adjustSize(Dimension size, Dimension preferredSize) { //
//		return averageSize(size, new Dimension(//
//				Integer.max(size.width, preferredSize.width), //
//				Integer.max(size.height, preferredSize.height)));
//	}
//
//	private Dimension averageSize(Dimension a, Dimension b) {
//		return new Dimension(//
//				(a.width + b.width) / 2, //
//				(a.height + b.height) / 2//
//		);
//	}

	/**
	 * @return gibt {@link #instance} als Kostenrechner zurück.
	 */
	public static Kostenrechner getInstance() {
		return instance;
	}

	/**
	 * @param instance d. {@link #instance}, d. gesetzt werden soll als
	 *                 {@link Kostenrechner}.
	 */
	public static void setInstance(Kostenrechner instance) {
		Kostenrechner.instance = instance;
	}

	/**
	 * @return gibt {@link #tabbedPane} als JTabbedPane zurück.
	 */
	public JTabbedPane getTabbedPane() {
		return tabbedPane;
	}

	/**
	 * @return gibt {@link #logger} als Logger zurück.
	 * @url {@link http://openbook.rheinwerk-verlag.de/java7/1507_20_001.html#dodtpf59fb46c-8f8b-4f81-af07-6ef529d72da9}
	 */
	public static Logger getLogger() {
		return logger;
	}

	/**
	 * Die Methode liefert den Namen der Log-Datei
	 * 
	 * @return einen {@link String} mit
	 *         {@link Files#createTempFile(String, String, java.nio.file.attribute.FileAttribute...)}{@link Path#toAbsolutePath()}{@link Path#toString()}
	 */
	public static String getLogFileName() {
		return logFileName;
	}

	/**
	 * @return gibt {@link #logFile} als {@link Path} zurück.
	 */
	public static Path getLogFile() {
		return logFile;
	}

	/**
	 * @param logFile d. {@link #logFile}, d. gesetzt werden soll als {@link Path}.
	 */
	public static void setLogFile(Path logFile) {
		Kostenrechner.logFile = logFile;
	}

	/**
	 * Definiert die Programm-Icons
	 * 
	 */
	void definiereIcons() {
		RechnerhammerIcons alleIcons = RechnerhammerIcons.getInstance();
		alleIcons.loadIcons();
		setIconImages(alleIcons);
	}

	/**
	 * Die Methode fügt einen {@link FileHandler} an den {@link #getLogger()}
	 * ({@link Logger}) an.
	 * 
	 * @url {@link http://openbook.rheinwerk-verlag.de/java7/1507_20_001.html#dodtpf59fb46c-8f8b-4f81-af07-6ef529d72da9}
	 */
	public static void addFileLogger() {
		try {
			logFile = Files.createTempFile(null, ".kostentenor.log");
			String fileName = logFile.toAbsolutePath().toString();
			Handler handler = new FileHandler(fileName, true);
			getLogger().addHandler(handler);
			getLogger().info("Logge in: " + fileName);
			logFileName = fileName;
		} catch (IOException e) {
			FehlerHelper.zeigeFehler("Konnte Logger nicht bauen.\nLogger konnte Datei nicht öffnen", e);
		}
	}

	/**
	 * Die Methode baut eine Menüzeile
	 */
	private void baueMenuBar() {
		JMenuBar menuBar = new JMenuBar();
		ExtrasJMenu aktionenMenu = ExtrasJMenu.getInstance();
		DateiJMenu dateiMenu = DateiJMenu.getInstance();
		BearbeitenJMenu bearbeitenMenu = BearbeitenJMenu.getInstance();
		menuBar.add(dateiMenu);
		menuBar.add(bearbeitenMenu);
		menuBar.add(aktionenMenu);

		dateiMenu.setKostenrechner(this);
		dateiMenu.buildMenu();

		bearbeitenMenu.setKostenrechner(this);
		bearbeitenMenu.buildMenu();

		aktionenMenu.setKostenrechner(this);
		aktionenMenu.buildMenu();
		setJMenuBar(menuBar);
		// Muss jetzt hierhin
		addWindowListener((WindowListener) dateiMenu.getBeendenAction());
	}

	/**
	 * Die Methode setzt die Oberflächenelemente für die Registerkarte 1
	 * Teilklagerücknahme
	 * 
	 * @return die {@link JPanel#getPreferredSize()}
	 * 
	 */
	private Dimension[] fuelleStreitwertReduktionPanel() {
		KostenJPanel pnStreitwertreduktion = TeilklageruecknahmePanel.getInstance();
		pnStreitwertreduktion.intoTabbedPane(getTabbedPane(),
				(Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(10000));
		pnStreitwertreduktion.setKostenrechner(this);
		pnStreitwertreduktion.buildPanel();
		return new Dimension[] { pnStreitwertreduktion.getMinimumSize(), pnStreitwertreduktion.getPreferredSize() };
	}

	/**
	 * Die Methode setzt die Oberflächenelemente für die Registerkarte 2
	 * Streitgenossen
	 * 
	 * @return die {@link JPanel#getPreferredSize()}
	 * 
	 */
	private Dimension[] fuelleBaumbachPanel() {
		KostenJPanel pnBaumbach = StreitgenossenPanel.getInstance();
		pnBaumbach.intoTabbedPane(getTabbedPane(), (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(30000));
		pnBaumbach.setKostenrechner(this);
		pnBaumbach.buildPanel();
		return new Dimension[] { pnBaumbach.getMinimumSize(), pnBaumbach.getPreferredSize() };
	}

	/**
	 * Die Methode setzt die Oberflächenelemente für die Registerkarte 3
	 * Hilfsaufrechnung.
	 * 
	 * @return die {@link JPanel#getPreferredSize()}
	 * 
	 */
	private Dimension[] fuelleHilfsaufrechnungPanel() {
		ForderungsStaffelungPanel pnHilfsaufrechnung = ForderungsStaffelungPanel.getInstance();
		pnHilfsaufrechnung.intoTabbedPane(getTabbedPane(),
				(Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(40000));
		pnHilfsaufrechnung.setKostenrechner(this);
		pnHilfsaufrechnung.buildPanel();
		return new Dimension[] { pnHilfsaufrechnung.getMaximumSize(), pnHilfsaufrechnung.getPreferredSize() };
	}

	/**
	 * Die Methode fuellt den GebuehrenRechnerPanel.
	 * 
	 * @return die {@link JPanel#getPreferredSize()}
	 * 
	 */
	private Dimension[] fuelleGebuehrenRechnerPanel() {
		KostenJPanel pnGebuehrenRechner = GebuehrenBerechnungPanel.getInstance();
		pnGebuehrenRechner.intoTabbedPane(getTabbedPane(),
				(Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(20000));
		pnGebuehrenRechner.setKostenrechner(this);
		pnGebuehrenRechner.buildPanel();
		getTabbedPane().addChangeListener(((GebuehrenBerechnungPanel) pnGebuehrenRechner).getAufrufHorcher());
		return new Dimension[] { pnGebuehrenRechner.getMaximumSize(), pnGebuehrenRechner.getPreferredSize() };
	}

	/**
	 * Die Methode setzt die Registerkarte 5 Hilfe mit einem einzigen JScrollPane *
	 * 
	 * @return die {@link JPanel#getPreferredSize()}
	 */
	private Dimension fuelleHilfePanel() {
		HilfePanel pnHilfePanel = HilfePanel.getInstance();
		pnHilfePanel.intoTabbedPane(getTabbedPane(), (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(50000));
		pnHilfePanel.setKostenrechner(this);
		pnHilfePanel.buildPanel();
		return pnHilfePanel.getPreferredSize();
	}

	/**
	 * @param tabbedPane d. {@link #tabbedPane}, d. gesetzt werden soll als
	 *                   JTabbedPane
	 */
	private void setTabbedPane(JTabbedPane tabbedPane) {
		this.tabbedPane = tabbedPane;
	}

}