/**
 * AssistentWeiterAction.java
 * eu.gronos.kostenrechner.view.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.event.ActionEvent;

import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.view.baumbach.BaumbachAssistentDialog;

/**
 * Alle Eingaben parsen (und Daten übergeben).
 * 
 * Wenn der jeweils letzte Zustand erreicht ist, Verhalten wie okButton.
 * 
 * Sonst dann den Zustand erhöhen.
 * 
 * Bestimmte Zustände (Anzahl der Parteien) müssen übersprungen werden, wenn
 * eine XJustiz-Datei importiert wird.
 *
 * @author Peter Schuster (setrok)
 * @date 21 Feb 2019
 *
 */
public class AssistentWeiterAction extends BeschriebeneAktion {
	private final BaumbachAssistentDialog dialog;
	private StreitgenossenAssistentParser pruefer;
	private static final long serialVersionUID = -1432886554037727631L;

	public AssistentWeiterAction(BaumbachAssistentDialog dialog, LangBeschriftung beschriftung,
			ParsendUndBauend<Streitgenossen> pruefer) {
		super(beschriftung);
		this.dialog = dialog;
		this.pruefer = (StreitgenossenAssistentParser) pruefer;
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		// Alle Eingaben parsen (und Daten übergeben)
		if (!parseEingabe(pruefer.getZustand()))
			return;
		if (pruefer.getZustand() >= pruefer.getHoechsterZustand() - 1) {
			/*
			 * Wenn der jeweils letzte Zustand erreicht ist, Verhalten wie okButton
			 */
			dialog.performOkAction(e);
		} else {
			/* Sonst dann den Zustand erhöhen */
			pruefer.setZustand(pruefer.getZustand() + 1);
			Kostenrechner.getLogger().info(String.format("Zustand: %d, überspringbar: %s!%n", pruefer.getZustand(),
					pruefer.isUeberspringbar(pruefer.getZustand())));
			while (pruefer.isUeberspringbar(pruefer.getZustand())) {
				pruefer.setZustand(pruefer.getZustand() + 1);
			}
			// Das jeweils nächste Feld mit Werten bestücken.
			pruefer.setzeWerte(pruefer.getZustand());
			dialog.zustandAnzeigen(pruefer.getZustand());
			// Ein Element muss den Focus haben (zustandAnzeigen gibt dem Weiter/Ok-Button
			// den Focus)
			if (pruefer.focusComponent != null) {
				pruefer.focusComponent.requestFocusInWindow();
				pruefer.focusComponent.requestFocus();
			}
		}
	}

	/**
	 * Die Variante von parseEingabe, die nach Import von Xjustiz genutzt werden
	 * soll. Überprüft die Eingabe und übergibt sie an das nächste Tab, füllt dieses
	 * aber, wenn möglich mit Daten aus Xjustiz.
	 * 
	 * Der Konstruktor für vdc überprüft schon, ob genug Kläger und Beklagte da sind
	 * und setzt die Anzahl dieser Beteiligten.
	 * 
	 * Um das eigentliche Überspringen muss sich der Weiter-Button kümmern.
	 * 
	 * Bei den zu überspringenden Tabs werden die zweidimensionalen Object-Arrays
	 * gefüllt, die sbtm-Modelle angelegt, setModel bei den JTables ausgeführt und
	 * die jeweiligen Streitwert-Textfelder gefüllt.
	 * 
	 * @param tab           die Nummer des Tabulators
	 * @param ueberspringen <code>true</code> = es sollen die bei Xjustiz-Import
	 *                      unnötigen Tabs übersprungen werden
	 * @return true, wenn zum nächsten, nicht zu überspringenden Tab gewechselt
	 *         werden kann
	 */
	public boolean parseEingabe(int tab) {
		boolean boole = false;
		pruefer.leseComponents(dialog.getComponents());
		try {
			boole = pruefer.parseEingabe(tab);
		} catch (IllegalArgumentException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		} catch (NullPointerException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		}
		return boole;
	}
}