/*
 * WiderklageCheckboxTextfieldHorcher.java
 * eu.gronos.kostenrechner.view.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFormattedTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.model.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus.Numerus;

/**
 * Klasse dient dazu, adjustWiderklageCheckBox aufzurufen, wenn ein anderes
 * Geschlecht für den Beteiligten ausgewählt wird. So kann der Text der Checkbox
 * an den Genus der Auswahl angepasst werden.
 *
 * Guckt bei Klägern, ob sie an der Widwrklage beteiligt sind und gibt
 * dementsprechend die Textfelder frei.
 *
 * @author Peter Schuster (setrok)
 * @date 11 Apr 2019
 *
 */
public class WiderklageCheckboxTextfieldHorcher implements ChangeListener, ItemListener {
	private final JCheckBox cbAnWiderklageBeteiligt;
	private final JComboBox<Beteiligter> cmbTypGenusNumerus;
	private final JFormattedTextField ftfStreitwert;
	private final JFormattedTextField ftfUnterliegen;
	private final VorsorgendeBeschriftung beschriftung;

	/**
	 * Konstruktor
	 */
	public WiderklageCheckboxTextfieldHorcher(JComboBox<Beteiligter> cmbTypGenusNumerus,
			JCheckBox cbAnWiderklageBeteiligt, JFormattedTextField ftfStreitwert, JFormattedTextField ftfUnterliegen,
			VorsorgendeBeschriftung beschriftung) {
		this.cmbTypGenusNumerus = cmbTypGenusNumerus;
		this.cbAnWiderklageBeteiligt = cbAnWiderklageBeteiligt;
		this.ftfStreitwert = ftfStreitwert;
		this.ftfUnterliegen = ftfUnterliegen;
		this.beschriftung = beschriftung;
	}

	@Override
	public void stateChanged(ChangeEvent e) {
		adjustTextfieldEditability();
	}

	/**
	 * Die Methode itemStateChanged dient dazu, adjustWiderklageCheckBox aufzurufen,
	 * wenn ein anderes Geschlecht für den Beteiligten ausgewählt wird. So kann der
	 * Text der Checkbox an den Genus der Auswahl angepasst werden.
	 * 
	 * The listener interface for receiving item events. The class that is
	 * interested in processing an item event implements this interface. The object
	 * created with that class is then registered with a component using the
	 * component's addItemListener method. When an item-selection event occurs, the
	 * listener object's itemStateChanged method is invoked.
	 * 
	 * @param ie
	 * 
	 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
	 */
	@Override
	public void itemStateChanged(ItemEvent ie) {
		// TODO: angeblich sind ActionListener besser für JComboBox
		adjustWiderklageCheckBox();
	}

	/**
	 * Die Methode adjustWiderklageCheckBox dient dazu, den Text der JCheckBox
	 * chckbxAnWiderklageBeteiligt an den BeteiligtenTyp anzupassen ("und
	 * Widerkläger/Widerbeklagter")
	 */
	private void adjustWiderklageCheckBox() {
		// int //0; männlich = 0; weiblich = 1
		GenusNumerus genusNumerus = GenusNumerus.MAENNLICH_SINGULAR;
		// int typ = -1;
		BeteiligtenTyp typ = null;
		if (cmbTypGenusNumerus != null) {
			final Beteiligter beteiligter = cmbTypGenusNumerus.getModel()
					.getElementAt(cmbTypGenusNumerus.getSelectedIndex());
			genusNumerus = GenusNumerus.of(beteiligter.getGenusNumerus().getGenus(), Numerus.SINGULAR);
			// .genusNumerusAsInt() % Beteiligter.PLURAL;
			typ = beteiligter.getTyp();
		}
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		switch (typ) {
		case DRITTWIDERBEKLAGTE:// 1
			cbAnWiderklageBeteiligt.setSelected(true);
			cbAnWiderklageBeteiligt.setEnabled(false);
			// jetzt kommt ein fall-through
		case KLAEGER:// 0
			beschrifter.changeTo(beschriftung, 1 + genusNumerus.ordinal());
			beschrifter.beschrifte(cbAnWiderklageBeteiligt, beschriftung);
			break;
		case BEKLAGTE:// 2
			beschrifter.changeTo(beschriftung, 3 + genusNumerus.ordinal());
			beschrifter.beschrifte(cbAnWiderklageBeteiligt, beschriftung);
			break;
		default:
			beschrifter.changeTo(beschriftung, 0);
			beschrifter.beschrifte(cbAnWiderklageBeteiligt, beschriftung);
		}
	}

	/**
	 * Die Methode sperrt die Eingabe-Textfelder bei Klägern, sofern die Widerklage
	 * nicht aktiviert ist
	 */
	private void adjustTextfieldEditability() {
		if (cbAnWiderklageBeteiligt == null || !cbAnWiderklageBeteiligt.isVisible() || ftfStreitwert == null
				|| ftfUnterliegen == null) {
			return;
		}
		// int = -1;
		BeteiligtenTyp typ = null;
		if (cmbTypGenusNumerus != null) {
			final Beteiligter beteiligter = cmbTypGenusNumerus.getModel()
					.getElementAt(cmbTypGenusNumerus.getSelectedIndex());
			typ = beteiligter.getTyp();
		}
		if (typ == BeteiligtenTyp.KLAEGER) {
			if (cbAnWiderklageBeteiligt.isSelected()) {
				ftfStreitwert.setEnabled(true);
				ftfUnterliegen.setEnabled(true);
			} else {
				ftfStreitwert.setEnabled(false);
				ftfStreitwert.setText("0");
				ftfUnterliegen.setEnabled(false);
				ftfUnterliegen.setText("0");
			}
		}
	}
}