/**
 * XjustizTransferHandler.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.awt.Image;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.GregorianCalendar;
import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.TransferHandler;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenXJustizLesenAction;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.data.tenordaten.Allgemein;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.util.files.BerechnungXmlDatei;
import eu.gronos.kostenrechner.util.files.TenorXmlDatei;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;
import eu.gronos.kostenrechner.view.baumbach.StreitgenossenPanel;
import eu.gronos.kostenrechner.view.forderungen.ForderungsStaffelungPanel;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenBerechnungPanel;
import eu.gronos.kostenrechner.view.gebuehren.TeilklageruecknahmePanel;
import eu.gronos.kostenrechner.view.menus.ExtrasJMenu;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Die Klasse stellt Methoden bereit, um die Werte einem
 * {@link VerfahrensDatenContainer} in die einzelnen JPanels zu schreiben, um
 * umgekehrt die Werte aus den JPanels in einen {@link VerfahrensDatenContainer}
 * zu schreiben und auch, um diese Werte aus einer XML-Datei zu nehmen.
 * 
 * Hält auch Methoden und Konstanten zum Lesen, Schreiben und Verwalten von
 * VerfahrensDatenContainern bereit.
 * 
 * Außerdem TransferHandler zum Drag-and-Drop-Support für Berechnungs-XML- und
 * Xjustiz-XML-Dateien
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 05.08.2014
 */
public class XmlTransferHandler extends TransferHandler {

	private static final long serialVersionUID = 8733776081066774904L;
	private static final String FEHLER_KEINE_XML_DATEI = "Keine XML-Datei!";
	private static final String FEHLER_MEHRERE_DATEIEN = "Mehr als eine oder keine Datei im Zugriff!";

	/**
	 * Erzeugt einen TransferHandler
	 * 
	 */
	public XmlTransferHandler() {
		super();
	}

	/**
	 * This method is called repeatedly during a drag gesture and returns true if
	 * the area below the cursor can accept the transfer, or false if the transfer
	 * will be rejected. For example, if a user drags a color over a component that
	 * accepts only text, the canImport method for that component's TransferHandler
	 * should return false.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the import can happen, false otherwise
	 * 
	 * @see javax.swing.TransferHandler#canImport(javax.swing.TransferHandler.TransferSupport)
	 */
	@Override
	public boolean canImport(TransferSupport support) {
		if (support.isDrop())
			support.setShowDropLocation(false);
		if (!support.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
			Kostenrechner.getLogger().info("Flavor not supported: javaFileListFlavor!");
			return false;
		}
		boolean copySupported = (COPY & support.getSourceDropActions()) == COPY;
		if (!copySupported) {
			Kostenrechner.getLogger().info("COPY not supported!");
			return false;
		}
		support.setDropAction(COPY);
		return true;
	}

	/**
	 * This method is called on a successful drop (or paste) and initiates the
	 * transfer of data to the target component. This method returns true if the
	 * import was successful and false otherwise.
	 * 
	 * Causes a transfer to occur from a clipboard or a drag and drop operation. The
	 * Transferable to be imported and the component to transfer to are contained
	 * within the TransferSupport. While the drag and drop implementation calls
	 * canImport to determine the suitability of a transfer before calling this
	 * method, the implementation of paste does not. As such, it cannot be assumed
	 * that the transfer is acceptable upon a call to this method for paste. It is
	 * recommended that canImport be explicitly called to cover this case.
	 * 
	 * Note: The TransferSupport object passed to this method is only valid for the
	 * duration of the method call. It is undefined what values it may contain after
	 * this method returns.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the data was inserted into the component, false otherwise
	 * @throws NullPointerException if <code>support</code> is null
	 * 
	 * @see javax.swing.TransferHandler#importData(javax.swing.TransferHandler.TransferSupport)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean importData(TransferSupport support) throws NullPointerException {
		if (!canImport(support)) {
			return false;
		}
		Transferable t = support.getTransferable();
		try {
			Object o = (List<File>) t.getTransferData(DataFlavor.javaFileListFlavor);
			List<File> list;
			if (o instanceof List<?>) {
				list = (List<File>) o;
			} else {
				return false;
			}
			if (list == null || list.size() != 1) {
				FehlerHelper.zeigeFehler(FEHLER_MEHRERE_DATEIEN, new IOException(FEHLER_MEHRERE_DATEIEN));
				return false;
			}
			final Path xml = list.get(0).toPath();
			if (isXml(xml) || isBerechnung(xml)) {
				Kostenrechner.getLogger().info("(isXml(xml) || isBerechnung(xml))");
				SwingUtilities.invokeLater(() -> nimmXmlDatei(xml));
				Kostenrechner.getLogger().info("return true;");
				return true;
			} else {
				FehlerHelper.zeigeFehler(FEHLER_KEINE_XML_DATEI, new IOException(FEHLER_KEINE_XML_DATEI));
				return false;
			}
		} catch (UnsupportedFlavorException e) {
			FehlerHelper.zeigeFehler("Flavor not supported. " + e.getLocalizedMessage(), e);
			return false;
		} catch (IOException e) {
			FehlerHelper.zeigeFehler("Data is no longer available in the requested flavor. " + e.getLocalizedMessage(),
					e);
			return false;
		}
	}

	@Override
	public Image getDragImage() {
		return RechnerhammerIcons.getInstance().get(4);
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return einen {@link VerfahrensDatenContainer}
	 */
	public VerfahrensDatenContainer liesAlleWertefuerContainer() {
		VerfahrensDatenContainer container = new VerfahrensDatenContainer();

		/*
		 * Allgemeine Programmeinstellungen, die nicht vom Container selbst gesetzt
		 * werden
		 */
		container.allgemein = ExtrasJMenu.getInstance().baueRueckgabewert();

		/* Erste Registerkarte Teilklagerücknahme */
		container.teilklageruecknahme = TeilklageruecknahmePanel.getInstance().baueRueckgabewert();

		/* Zweite Registerkarte Streitgenossen */
		container.streitgenossen = StreitgenossenPanel.getInstance().baueRueckgabewert();

		/* Dritte Registerkarte Forderungsstaffelung */
		container.staffelung = ForderungsStaffelungPanel.getInstance().baueRueckgabewert();

		/* Vierte Registerkarte Freie Gebührenberechnung */
		container.gebuehrenBerechnung = GebuehrenBerechnungPanel.getInstance().baueRueckgabewert();

		/* Höchster Wert im Verfahren */
		container.setzeStreitwertInAllgemein();

		/* und weg damit */
		return container;
	}

	/**
	 * Die Methode dient zum Xjustiz-Import bzw Laden von Xml-Berechnungen aus
	 * {@link Kostenrechner#main(String[]) main} und aus anderen Klassen.
	 * 
	 * Wenn <code>null</code> übergeben wird, macht sie einfach nix.
	 * 
	 * @param xml ein {@link File}
	 */
	public void nimmXmlDatei(Path xml) {
		VerfahrensDatenContainer container = null;
		if (xml != null) {
			Kostenrechner.getLogger().info("nimmXmlDatei");
			if (isXml(xml)) {
				container = new StreitgenossenXJustizLesenAction(StreitgenossenPanel.getInstance(),
						Kostenrechner.getInstance(), (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39211))
								.xJustizLesen(xml);
				// Beim XJustiz-Import dürfen nur der allgemeine Teil und die Streitgenossen
				// eingelesen werden
				if (container != null) {
					if (container.allgemein != null)
						ExtrasJMenu.getInstance().setzeWerte(container.allgemein);
					if (container.streitgenossen != null)
						StreitgenossenPanel.getInstance().setzeWerte(container.streitgenossen);
				}
			} else if (isBerechnung(xml)) {
				container = TenorXmlDatei.createInstanceFor(xml).ladeXml(xml);
				// Beim Import einer Berechnung alle Werte setzen!
				if (container != null) {
					setzeAlleWerteAusContainer(container);
					// Wenn ein TenorDatenContainer (.rtf.skktx) geladen wird, erwartet der Nutzer
					// wohl, dass sich auch der TenorDialog öffnet.
					if (container instanceof TenorDatenContainer) {
						new TenorDialog(Kostenrechner.getInstance(), (TenorDatenContainer) container).showDialog();
					}
				}
			}
		}
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände dem
	 * {@link VerfahrensDatenContainer} anzupassen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 */
	public void setzeAlleWerteAusContainer(VerfahrensDatenContainer container) {
		Kostenrechner.getLogger().info(container.createFileInfo());

		container.allgemein.selectedPanel = getSelectedPanel(container);

		/* Allgemeine Einstellungen */
		Kostenrechner.getLogger().info("container.allgemein");
		ExtrasJMenu.getInstance().setzeWerte(container.allgemein);

		/* Erste Registerkarte Teilklagerücknahme */
		Kostenrechner.getLogger().info("container.teilklageruecknahme");
		TeilklageruecknahmePanel.getInstance().setzeWerte(container.teilklageruecknahme);

		/* Zweite Registerkarte Streitgenossen */
		Kostenrechner.getLogger().info("container.streitgenossen");
		StreitgenossenPanel.getInstance().setzeWerte(container.streitgenossen);

		/* Dritte Registerkarte Forderungsstaffelung */
		Kostenrechner.getLogger().info("container.staffelung");
		ForderungsStaffelungPanel.getInstance().setzeWerte(container.staffelung);

		/* Vierte Registerkarte Freie Gebührenberechnung */
		Kostenrechner.getLogger().info("container.gebuehrenBerechnung");
		GebuehrenBerechnungPanel.getInstance().setzeWerte(container.gebuehrenBerechnung);

	}

	/**
	 * Die Methode dient dazu, das aktuelle Systemdatum zum Speichern zu ermitteln
	 * 
	 * @return einen {@link XMLGregorianCalendar} mit dem aktuellen Systemdatum
	 * @throws DatatypeConfigurationException
	 */
	public XMLGregorianCalendar speicherDatum() {
		try {
			DatatypeFactory factory = DatatypeFactory.newInstance();
			return factory.newXMLGregorianCalendar(new GregorianCalendar());
		} catch (DatatypeConfigurationException dce) {
			FehlerHelper.zeigeFehler("Schwerer Fehler beim Konfigurieren der Zeit!", dce);
			return null;
		}
	}

	/**
	 * Die Methode findet das gesetzte {@link Allgemein#selectedPanel} heraus. Wenn
	 * der Wert nicht in der Datei gesetzt ist, dann über
	 * {@link UnterContainerKlasse#isEmpty()}
	 * 
	 * @param container der {@link VerfahrensDatenContainer}
	 * @return einen <code>int</code>: entweder -1 oder
	 *         {@link Teilklageruecknahme#TAB_ID}, {@link Streitgenossen#TAB_ID},
	 *         {@link ForderungsStaffelung#TAB_ID} oder
	 *         {@link GebuehrenBerechnung#TAB_ID}
	 */
	public int getSelectedPanel(VerfahrensDatenContainer container) {
		// In den meisten Fällen ist wohl container.allgemein.selectedPanel gesetzt
		if (container.allgemein.selectedPanel > -1) {
			return container.allgemein.selectedPanel;
		}
		// Sonst die einzelnen Panels durchgehen
		if (!container.teilklageruecknahme.isEmpty())
			return Teilklageruecknahme.TAB_ID;
		if (!container.streitgenossen.isEmpty())
			return Streitgenossen.TAB_ID;
		if (!container.staffelung.isEmpty())
			return ForderungsStaffelung.TAB_ID;
		if (!container.gebuehrenBerechnung.isEmpty())
			return GebuehrenBerechnung.TAB_ID;
		// Wenn alle leer sind, den zwischengespeicherten aktuellen Panel zurückliefern
		return container.allgemein.selectedPanel;// selectedPanel;
	}

	/**
	 * Die Methode dient dazu, anhand der Dateiendung zu prüfen, ob es sich um eine
	 * gespeicherte Berechnung im skktx-XML-Format handelt
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn der Dateiname auf
	 *         {@link BerechnungXmlDatei#DATEIENDUNG} endet und deswegen von
	 *         {@link BerechnungXmlDatei#DATEI_FILTER} akzeptiert wird.
	 */
	private boolean isBerechnung(final Path datei) {
		return BerechnungXmlDatei.DATEI_FILTER.acceptExtension(datei);
	}

	/**
	 * Die Methode dient dazu, anhand der Dateiendung zu prüfen, ob es sich um eine
	 * Xjustiz-XML-Datei handelt
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn der Dateiname vom
	 *         {@link XjustizContentHandler#XJUSTIZ_FILTER} akzeptiert wird, also
	 *         auf .xml endet
	 */
	private boolean isXml(final Path datei) {
		return XjustizContentHandler.XJUSTIZ_FILTER.acceptExtension(datei);
	}

}