/*
 * ForderungPruefer.java
 * eu.gronos.kostenrechner.controller.forderungen (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.forderungen;

import java.awt.Component;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.forderungen.Forderung;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.util.forderungen.ForderungPruefer;
import eu.gronos.kostenrechner.view.forderungen.ForderungHinzufuegen;

/**
 * Der Prüfer kontrolliert alle Werte zum Bau einer {@link Forderung}.
 *
 * @author Peter Schuster (setrok)
 * @date 15.07.2019
 *
 */
public class ForderungParser implements ParsendUndBauend<Forderung> {

	private JComboBox<Forderung> cmbForderungsart;
	private JFormattedTextField ftfStreitwert;
	private JFormattedTextField ftfUnterliegen;
	private Forderung rueckgabewert;

	/**
	 * 
	 * @param components eine {@link JComboBox} und zwei
	 *                   {@link JFormattedTextField}s
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<Forderung> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("45015".equals(component.getName())) {
					cmbForderungsart = (JComboBox<Forderung>) component;
				} else if ("45025".equals(component.getName())) {
					ftfStreitwert = (JFormattedTextField) component;
				} else if ("45035".equals(component.getName())) {
					ftfUnterliegen = (JFormattedTextField) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode liest mit {@link #leseFelder()} die Felder aus und überprüft dann
	 * den {@link #getRueckgabewert()}.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			return parseEingabe(rueckgabewert);
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode wird von okAction aufgerufen, um zu prüfen, ob alle JComponents
	 * zur Eingabe mit gültigen Werten gefüllt sind und diese zueinander nicht im
	 * Widerspruch stehen.
	 * 
	 * @param forderung den {@link #getRueckgabewert()}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @throws IllegalArgumentException Wenn erfolg höher ist als der Wert der
	 *                                  Forderung
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(Forderung forderung)
			throws IllegalArgumentException, NumberFormatException, NullPointerException {
		this.rueckgabewert = forderung;
		Pruefend<Forderung> pruefer = new ForderungPruefer();
		return pruefer.pruefeEingabe(forderung);
	}

	/**
	 * Die Methode soll aus den Eingabefeldern des Dialogs eine Forderung
	 * zusammenzubauen, der dann von {@link #showDialogAndAsk()} an
	 * {@link ForderungHinzufuegen} zurückgegeben werden kann.
	 * 
	 * @return eine {@link Forderung} aus allen Werten
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public Forderung getRueckgabewert() {
		return rueckgabewert;
	}

	/**
	 * 
	 * @return eine {@link Forderung}
	 * @throws NumberFormatException
	 * @throws NullPointerException  TODO (kommentieren)
	 */
	private Forderung leseFelder() throws NumberFormatException, NullPointerException {
		rueckgabewert = (Forderung) cmbForderungsart.getSelectedItem();
		if (rueckgabewert == null) {
			throw new NullPointerException("Keine Forderungsart ausgewählt. Wählen Sie bitte eine Forderungsart aus!");
		}
		final Double wert = new NumberExtractor(ftfStreitwert, ftfStreitwert.getAccessibleContext().getAccessibleName())
				.extractDouble();
		final Double erfolg = new NumberExtractor(ftfUnterliegen,
				ftfUnterliegen.getAccessibleContext().getAccessibleName()).extractDouble();
		rueckgabewert.setAntrag(Euro.ofEuros(wert));
		rueckgabewert.setErfolg(Euro.ofEuros(erfolg));
		return rueckgabewert;
	}

}
