/*
 * ForderungsStaffelungsPruefer.java
 * eu.gronos.kostenrechner.view.forderungen (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.forderungen;

import java.awt.Component;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JTable;

import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.interfaces.WerteLesbar;
import eu.gronos.kostenrechner.model.forderungen.ForderungenTableModel;
import eu.gronos.kostenrechner.util.forderungen.ForderungsStaffelungPruefer;
import eu.gronos.kostenrechner.view.forderungen.ForderungsStaffelungPanel;

/**
 * Das {@link ParsendUndBauend} kontrolliert alle Bestandteile des
 * {@link ForderungsStaffelungPanel}s
 *
 * @author Peter Schuster (setrok)
 * @date 30.08.2019
 *
 */
public class ForderungsStaffelungParser implements ParsendUndBauend<ForderungsStaffelung> {

	private ForderungsStaffelung rueckgabewert;
	private JComboBox<Beteiligter> cbAngriffsKlaeger;
	private JComboBox<Beteiligter> cbVerteidigungsBeklagter;
	private JTable tbForderungen;
	/**
	 * wenn <code>true</code>, dann prüft der {@link ParsendUndBauend} noch weitere
	 * Plausibilitäten
	 */
	private final boolean streng;

	/**
	 * Konstruktor, der nur die Standard-Plausibilitäten prüft.
	 * 
	 */
	public ForderungsStaffelungParser() {
		this(false);
	}

	/**
	 * Konstruktor.
	 * 
	 * @param streng boolean wenn <code>true</code>, dann prüft der
	 *               {@link ParsendUndBauend} noch weitere Plausibilitäten
	 */
	public ForderungsStaffelungParser(boolean streng) {
		super();
		this.streng = streng;
	}

	/**
	 * @param components zwei {@link JComboBox}es und eine {@link JTable}
	 * @return diesen {@link ForderungsStaffelungParser}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<ForderungsStaffelung> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("41005".equals(component.getName())) {
					cbAngriffsKlaeger = (JComboBox<Beteiligter>) component;
				} else if ("41015".equals(component.getName())) {
					cbVerteidigungsBeklagter = (JComboBox<Beteiligter>) component;
				} else if ("42011".equals(component.getName())) {
					tbForderungen = (JTable) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode wird von {@link WerteLesbar#baueRueckgabewert()} aufgerufen, um
	 * mit {@link #leseFelder()} die Felder auszulesen und zu prüfen, ob der einen
	 * Felder wirft.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
			if (streng)
				return parseEingabe(rueckgabewert);
			else
				return true;
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode und überprüft den {@link #getRueckgabewert()}. ob alle Felder der
	 * {@link UnterContainerKlasse} mit gültigen Werten gefüllt sind und diese
	 * zueinander nicht im Widerspruch stehen.
	 * 
	 * @param forderung den {@link #getRueckgabewert()}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @throws IllegalArgumentException Wenn forderungen null oder leer ist; oder
	 *                                  wenn die Reihenfolge (erst Klageforderungen,
	 *                                  dann Aufrechnungsforderungen) nicht
	 *                                  eingehalten ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(ForderungsStaffelung rueckgabewert)
			throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = rueckgabewert;
		Pruefend<ForderungsStaffelung> pruefer = new ForderungsStaffelungPruefer();
		return pruefer.pruefeEingabe(rueckgabewert);
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return den Untercontainer von {@link VerfahrensDatenContainer#staffelung}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public ForderungsStaffelung getRueckgabewert() {
		return rueckgabewert;
	}

	private ForderungsStaffelung leseFelder() throws NullPointerException {
		rueckgabewert = new ForderungsStaffelung();
		rueckgabewert.parteien.add((Beteiligter) cbAngriffsKlaeger.getSelectedItem());
		rueckgabewert.parteien.add((Beteiligter) cbVerteidigungsBeklagter.getSelectedItem());
		ForderungenTableModel ftm = (ForderungenTableModel) tbForderungen.getModel();
		rueckgabewert.forderungen = ftm.getAllValues();
		return rueckgabewert;
	}
}
