/*
 * AuslagenHoeheVisibilityHorcher.java
 * eu.gronos.kostenrechner.view.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.model.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.data.gebuehren.AuslagenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRahmenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahl;
import eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear;

/**
 * Der Horcher hängt sich an eine {@link JList} (als
 * {@link ListSelectionListener}) sowie an ein {@link ListModel} (als
 * {@link ListDataListener}) und passt entsprechend die Sichtbarkeit des
 * {@link JTextField}s an.
 *
 * @author Peter Schuster (setrok)
 * @date 3 May 2019
 *
 */
public class AuslagenHoeheVisibilityHorcher implements ListSelectionListener, ListDataListener {
	private final JFormattedTextField ftf;
	private final JLabel label;
	private int selectedIndex = -1;
	private final VorsorgendeBeschriftung beschriftung;

	/**
	 * Der Konstruktor braucht einen Verweis auf das anzupassende {@link JLabel} und
	 * das {@link JFormattedTextField}. Ein Verweis auf die {@link JList} ist
	 * hingegen nicht nötig, da die Quelle aus dem {@link ListSelectionEvent}
	 * gezogen wird.
	 * 
	 * @param ftf          das {@link JFormattedTextField}
	 * @param label        das anzupassende {@link JLabel}
	 * @param beschriftung eine {@link VorsorgendeBeschriftung}
	 */
	public AuslagenHoeheVisibilityHorcher(JFormattedTextField ftf, JLabel label, VorsorgendeBeschriftung beschriftung) {
		this.ftf = ftf;
		this.label = label;
		this.beschriftung = beschriftung;
	}

	/**
	 * Die Methode passt die Beschreibbarkeit des Eingabefelds für die Auslagenhöhe
	 * an die Art des GebührenTatbestands an.
	 * 
	 * @param e {@link ListSelectionEvent}
	 * 
	 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
	 */
	@Override
	public void valueChanged(ListSelectionEvent e) {
		if (!(e.getSource() instanceof JList<?>))
			return;
		final GebuehrenTatbestand gt = getSelectedItem((JList<?>) e.getSource());
		changeVisibility(gt);
	}

	/**
	 * Die Methode wird aufgerufen, wenn sich die Liste in einer komplexen Weise
	 * geändert hat, etwa Bestandteile ausgetauscht wurden. Sie passt die
	 * Beschreibbarkeit des Eingabefelds für die Auslagenhöhe an die Art des
	 * GebührenTatbestands an.
	 * 
	 * @param e ein {@link ListDataEvent} mit den nötigen Informationen
	 * 
	 * @see javax.swing.event.ListDataListener#contentsChanged(javax.swing.event.ListDataEvent)
	 */
	@Override
	public void contentsChanged(ListDataEvent e) {
		if (!(e.getSource() instanceof ListModel<?>))
			return;
		final GebuehrenTatbestand gt = getSelectedItem((ListModel<?>) e.getSource());
		changeVisibility(gt);
	}

	/**
	 * Die Methode wird aufgerufen, wenn ein Abschnitt zur Liste hinzugefügt wurde
	 * 
	 * @param e ein {@link ListDataEvent} mit den nötigen Informationen
	 * 
	 * @see javax.swing.event.ListDataListener#intervalAdded(javax.swing.event.ListDataEvent)
	 */
	@Override
	public void intervalAdded(ListDataEvent e) {
	}

	/**
	 * Die Methode wird aufgerufen, wenn ein Abschnitt von der Liste entfernt wurde
	 * 
	 * @param e ein {@link ListDataEvent} mit den nötigen Informationen
	 * 
	 * @see javax.swing.event.ListDataListener#intervalRemoved(javax.swing.event.ListDataEvent)
	 */
	@Override
	public void intervalRemoved(ListDataEvent e) {
	}

	/**
	 * Die Methode entscheidet, ob das Feld aktiv werden soll. Sie passt die
	 * Beschreibbarkeit des Eingabefelds für die Auslagenhöhe an die Art des
	 * GebührenTatbestands an.
	 * 
	 * @param gt ein {@link GebuehrenTatbestand}, nach dem entschieden werden soll
	 */
	private void changeVisibility(final GebuehrenTatbestand gt) {
		if (gt != null && gt instanceof AuslagenTatbestand && (((AuslagenTatbestand) gt).getBetrag() == null
				|| Euro.ofCents(0L).compareTo(((AuslagenTatbestand) gt).getBetrag()) >= 0)) {
			aktiviereFtfAuslagenHoehe("0", 1, gt);
		} else if (gt != null && gt instanceof GebuehrenMitAnzahlLinear && ((GebuehrenMitAnzahl) gt).getAnzahl() < 0) {
			aktiviereFtfAuslagenHoehe("0", 2, gt);
		} else if (gt != null && gt instanceof GebuehrenAnrechnungsTatbestand) {
			aktiviereFtfAuslagenHoehe(String.format("%,.2f", ((GebuehrenAnrechnungsTatbestand) gt).getMittel()), 3, gt);
		} else if (gt != null && gt instanceof GebuehrenRahmenTatbestand) {
			aktiviereFtfAuslagenHoehe(String.format("%,.1f", ((GebuehrenRahmenTatbestand) gt).getMittel()), 3, gt);
		} else {
			this.ftf.setEnabled(false);
			ComponentBeschrifter beschrifter = new ComponentBeschrifter();
			beschrifter.changeTo(beschriftung, 0);
			beschrifter.beschrifte(label, beschriftung);
		}
	}

	/**
	 * Die Methode dient dazu, das Eingabefeld zu aktivieren
	 * 
	 * @param feldText  der Standardwert für das {@link JFormattedTextField}
	 * @param gt        ein {@link GebuehrenTatbestand}, nach dem entschieden werden
	 *                  soll
	 * @param labelText der Wert für {@link #label}
	 * 
	 */
	private void aktiviereFtfAuslagenHoehe(String feldText, int index, GebuehrenTatbestand gt) {
		ftf.setVisible(true);
		ftf.setEnabled(true);
		ftf.setText(feldText);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.changeTo(beschriftung, index);
		if (gt instanceof GebuehrenMitAnzahlLinear && beschriftung.getTitle().contains("%"))
			beschriftung.setTitle(String.format(beschriftung.getTitle(), ((GebuehrenMitAnzahl) gt).getEinheit()));
		beschrifter.beschrifte(label, beschriftung);
	}

	/**
	 * Die Methode ermittelt {@link JList#getSelectedIndex()} (und speichert ihn in
	 * {@link #selectedIndex}). Damit grabscht er sich das {@link ListModel} und
	 * dann mittels {@link #getSelectedItem(ListModel)} den markierten
	 * {@link GebuehrenTatbestand}.
	 * 
	 * @param liste die {@link JList}
	 * @return den derzeit markierten {@link GebuehrenTatbestand} in der
	 *         {@link JList}
	 */
	private GebuehrenTatbestand getSelectedItem(JList<?> liste) {
		selectedIndex = liste.getSelectedIndex();
		if (liste == null || selectedIndex < 0)
			return null;
		final ListModel<?> model = liste.getModel();
		return getSelectedItem(model);
	}

	/**
	 * Die Methode gibt ausgehend vom Feld {@link #selectedIndex} den markierten
	 * {@link GebuehrenTatbestand} aus dem übergebenen {@link ListModel} zurück. Der
	 * {@link #selectedIndex} muss gesetzt sein.
	 * 
	 * So kann der markierte {@link GebuehrenTatbestand} auch ermittelt werden, wenn
	 * der Listener als {@link ListDataListener} am {@link ListModel} hängt.
	 * 
	 * @param model ein {@link ListModel}&lt;{@link GebuehrenTatbestand}&gt;
	 * @return den derzeit markierten {@link GebuehrenTatbestand} in der
	 *         {@link JList}
	 */
	private GebuehrenTatbestand getSelectedItem(final ListModel<?> model) {
		final Object element = model.getElementAt(selectedIndex);
		if (element instanceof GebuehrenTatbestand)
			return (GebuehrenTatbestand) element;
		else
			return null;
	}

}