/**
 * TeilklageGebuehrVorherAction.java
 * eu.gronos.kostenrechner.view.panel (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.model.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.data.gebuehren.AnwaltsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenGesetzesSammlung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.data.gebuehren.GerichtsGebuehrenTabelle;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;
import eu.gronos.kostenrechner.interfaces.PanelTableModelHandler;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTableModel;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrHinzufuegenDialog;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenBerechnungPanel;
import eu.gronos.kostenrechner.view.gebuehren.TeilklageruecknahmePanel;

/**
 * Die {@link BeschriebeneAktion} fügt einen neuen Gebührentatbestand in die
 * Tabelle der vor der Streitwertreduktion entstandenen, in die Tabelle für die
 * nach Gebühren ein, die nach der Streitwertreduktion entstanden sind oder in
 * die Tabelle für die Gebühren zur freien Berechnung ein.
 * 
 * {@link KeyEvent#VK_N} und {@link KeyEvent#VK_R}
 *
 */
public class GebuehrNeuAction extends BeschriebeneAktion implements AbfrageLieferant<GebuehrHinzufuegenDialog> {
	private final PanelTableModelHandler<GebuehrenTableModel> panel;
	private static final long serialVersionUID = 7170038539836176410L;
	private final int index;
	private GebuehrenVerzeichnis alleTatbestaende;
	private GebuehrHinzufuegenDialog dialog = null;
	private final int gebuehrenKey;
	public static final int RVG_KEY = 32;
	public static final int GKG_KEY = 64;
	public static final int ALLE_KEY = RVG_KEY + GKG_KEY;

	/**
	 * Der Standardkonstruktor fordert auch ein {@link GebuehrenAuflistung} aller
	 * {@link GebuehrenTatbestand}e
	 * 
	 * @param panel        ein {@link PanelTableModelHandler}
	 * @param index        {@link TeilklageruecknahmePanel#TABLE_FTF_VORHER} oder
	 *                     {@link TeilklageruecknahmePanel#TABLE_FTF_NACHHER} oder
	 *                     {@link GebuehrenBerechnungPanel#TABLE_FTF_GEBUEHREN}
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 * @param mnemonic     ({@link javax.swing.Action#MNEMONIC_KEY}) The key used
	 *                     for storing an Integer that corresponds to one of the
	 *                     KeyEvent key codes. The value is commonly used to specify
	 *                     a mnemonic. KeyEvent.getExtendedKeyCodeForChar('\u0444'))
	 *                     sets the mnemonic of myAction to Cyrillic letter "Ef".
	 * @param gebuehrenKey
	 * 
	 */
	public GebuehrNeuAction(PanelTableModelHandler<GebuehrenTableModel> panel, LangBeschriftung beschriftung, int index,
			int gebuehrenKey) {
		super(beschriftung);
		this.panel = panel;
		this.index = index;
		this.gebuehrenKey = gebuehrenKey;
	}

	/**
	 * Der weitere Konstruktor nimmt einfach alle Gebührentatbestände
	 * 
	 * @param panel        ein {@link PanelTableModelHandler}
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),{@link javax.swing.Action#MNEMONIC_KEY},
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 * @param index        {@link TeilklageruecknahmePanel#TABLE_FTF_VORHER} oder
	 *                     {@link TeilklageruecknahmePanel#TABLE_FTF_NACHHER} oder
	 *                     {@link GebuehrenBerechnungPanel#TABLE_FTF_GEBUEHREN}
	 */
	public GebuehrNeuAction(PanelTableModelHandler<GebuehrenTableModel> panel, LangBeschriftung beschriftung,
			int index) {
		this(panel, beschriftung, index, ALLE_KEY);
	}

	@Override
	public void actionPerformed(ActionEvent event) {
		final String name = (String) getValue(Action.NAME);
		VorsorgendeBeschriftung vorsorgend = (VorsorgendeBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(16000);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.changeTo(vorsorgend, 0);
		beschrifter.changeTo(vorsorgend, name);
		GebuehrenTatbestand gt = setDialog(
				new GebuehrHinzufuegenDialog(Kostenrechner.getInstance(), vorsorgend, getAlleTatbestaende()))
						.showDialogAndAsk();
		if (gt != null)
			panel.getTableModel(index).addRow(gt);
		setDialog(null);
	}

	/**
	 * 
	 * @return
	 * @return gibt den {@link GebuehrHinzufuegenDialog} zurück, den die Aktion
	 *         erstellt.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	@Override
	public GebuehrHinzufuegenDialog getDialog() {
		return dialog;
	}

	/**
	 * @return gibt {@link #alleTatbestaende} als {@link GebuehrenAuflistung}
	 *         zurück.
	 */
	private GebuehrenVerzeichnis getAlleTatbestaende() {
		if (alleTatbestaende == null) {
			if (RVG_KEY == gebuehrenKey)
				alleTatbestaende = GebuehrenGesetzesSammlung.getRvg().getVerzeichnis();
			else if (GKG_KEY == gebuehrenKey)
				alleTatbestaende = GebuehrenGesetzesSammlung.getGkg().getVerzeichnis();
			else
				alleTatbestaende = createAlleGebuehrenTatbestaende();
		}
		return alleTatbestaende;
	}

	private GebuehrHinzufuegenDialog setDialog(GebuehrHinzufuegenDialog dialog) {
		this.dialog = dialog;
		return dialog;
	}

	/**
	 * Die Methode dient dazu, alle {@link GebuehrenTatbestand}e für
	 * {@link GebuehrHinzufuegenDialog} zusammenzustellen, falls keine dem
	 * Konstruktor übergeben wurden.
	 * 
	 * @return die {@link GebuehrenTatbestand}e der {@link GerichtsGebuehrenTabelle}
	 *         und der {@link AnwaltsGebuehrenTabelle} als Array
	 */
	private static GebuehrenVerzeichnis createAlleGebuehrenTatbestaende() {
		GebuehrenVerzeichnis verzeichnis = new GebuehrenVerzeichnis();
		verzeichnis.putAll(GebuehrenGesetzesSammlung.getGkg().getVerzeichnis());
		verzeichnis.putAll(GebuehrenGesetzesSammlung.getRvg().getVerzeichnis());
		return verzeichnis;
	}
}