/**
 * GebuehrenAufstellungAction.java
 * eu.gronos.kostenrechner.view.panel (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.event.KeyEvent;

import javax.swing.Action;

import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.kostenrechner.controller.TenorAction;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.GebuehrenTableTextfieldHandler;
import eu.gronos.kostenrechner.logic.gebuehren.GebuehrenBerechnungsAufstellung;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenBerechnungPanel;

/**
 * Die {@link Action} erstellt eine Aufstellung der Gebühren und ihrer Summe zu
 * dem ausgewählten Streitwert. {@link KeyEvent#VK_E}
 *
 * @author Peter Schuster (setrok)
 * @date 04.11.2018
 *
 */
public class GebuehrenAufstellungAction extends TenorAction {
	private final GebuehrenTableTextfieldHandler handler;
	private static final long serialVersionUID = 4793754967839528087L;

	/**
	 * Konstruktor.
	 * 
	 * @param handler      Verweis auf {@link GebuehrenBerechnungPanel}
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link Action#MNEMONIC_KEY},
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 */
	public GebuehrenAufstellungAction(GebuehrenTableTextfieldHandler handler, LangBeschriftung beschriftung) {
		super(beschriftung);
		this.handler = handler;
	}

	/**
	 * Die Methode prüft, ob alle nötigen Oberflächenelemente non-null sind
	 * 
	 * @return <code>true</code>, wenn {@link #alleWerte()} gefahrlos laufen kann.
	 * @throws IllegalArgumentException wird ansonsten geworfen (<code>false</code>
	 *                                  gibt es nicht), damit die {@link Exception}
	 *                                  direkt behandelt werden kann.
	 * 
	 * @see eu.gronos.kostenrechner.controller.TenorAction#allesOk()
	 */
	@Override
	protected boolean allesOk() throws IllegalArgumentException {
		handler.requestFocusInWindow();
		if (handler.getFormattedTextField(GebuehrenBerechnungPanel.FTF_STREITWERT) != null
				&& handler.getTableModel(GebuehrenBerechnungPanel.TABLE_GEBUEHREN) != null)
			return true;
		throw new IllegalArgumentException("Streitwert oder Gebühren nicht vorhanden!");
	}

	/**
	 * Die Methode ruft {@link GebuehrenBerechnungsAufstellung} auf.
	 * 
	 * @param container einen {@link VerfahrensDatenContainer}, den
	 *                  {@link TenorAction} über
	 *                  {@link XmlTransferHandler#liesAlleWertefuerContainer()}
	 *                  zusammenstellt
	 * @return den {@link TenorDatenContainer}
	 * @throws IllegalArgumentException wenn was schief läuft
	 * @throws NullPointerException     wenn was <code>null</code> ist.
	 * 
	 * @see eu.gronos.kostenrechner.controller.TenorAction#tenorDaten(eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer)
	 */
	@Override
	protected TenorDatenContainer tenorDaten(VerfahrensDatenContainer container)
			throws NullPointerException, IllegalArgumentException {
		final TenorDatenContainer tenorDaten = new GebuehrenBerechnungsAufstellung(container).erzeugeContainer();
		return tenorDaten;
	}
}