/*
 * GebuehrHinzuPruefer.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JList;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.gebuehren.AuslagenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenErhoehungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRahmenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.MehrfachPauschalTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.PauschalTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.util.gebuehren.GebuehrenPruefer;

/**
 * Der Prüfer kontrolliert alle Werte zum Bau eines {@link GebuehrenTatbestand}s
 *
 * @author Peter Schuster (setrok)
 * @date 26.06.2019
 *
 */
public class GebuehrenParser implements ParsendUndBauend<GebuehrenTatbestand> {
	private JFormattedTextField ftfAuslagenHoehe;
	private JLabel lblAuslagenHoehe;
	private JList<GebuehrenTatbestand> liGebuehrenListe;
	private GebuehrenTatbestand rueckgabewert;

	/**
	 * @param components braucht ein {@link JFormattedTextField}, das dazu gehörende
	 *                   {@link JLabel} und eine {@link JList}.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<GebuehrenTatbestand> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("16035".equals(component.getName())) {
					ftfAuslagenHoehe = (JFormattedTextField) component;
				} else if ("16021".equals(component.getName())) {
					liGebuehrenListe = (JList<GebuehrenTatbestand>) component;
				} else if ("16031".equals(component.getName())) {
					lblAuslagenHoehe = (JLabel) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, nachdem er Felder
	 * ausgelesen hat.
	 * 
	 * @return true, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			parseEingabe(rueckgabewert);
		} catch (NumberFormatException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
		return true;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, ohne Felder
	 * auszulesen, indem er ihr übergeben wird.
	 * 
	 * @param rueckgabewert der zu prüfende {@link GebuehrenTatbestand}
	 * @return true, wenn die Plausibilitäten zufrieden sind
	 * @throws NumberFormatException wenn der {@link NumberExtractor} bei einem
	 *                               {@link JFormattedTextField} eine solche
	 *                               Exception wirft oder Plausibilitäten nicht
	 *                               eingehalten sind.
	 * @throws NullPointerException  wenn kein {@link GebuehrenTatbestand} in der
	 *                               {@link JList} ausgewählt ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(GebuehrenTatbestand gt) throws NumberFormatException, NullPointerException {
		this.rueckgabewert = gt;
		return new GebuehrenPruefer().pruefeEingabe(rueckgabewert);
	}

	/**
	 * @return einen {@link GebuehrenTatbestand}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public GebuehrenTatbestand getRueckgabewert() {
		return rueckgabewert;
	}

	/**
	 * Die Methode gibt das ausgewählte Element der {@link JList} zurück.
	 * 
	 * @return {@link GebuehrenTatbestand}
	 */
	public GebuehrenTatbestand getSelectedItem() {
		if (liGebuehrenListe.getSelectedIndex() < 0)
			return null;
		else
			return (GebuehrenTatbestand) liGebuehrenListe.getModel().getElementAt(liGebuehrenListe.getSelectedIndex());
	}

	/**
	 * Übergibt den ausgewählten {@link GebuehrenTatbestand} bzw. baut aus dem
	 * ausgewählten {@link GebuehrenTatbestand} und den übrigen Werten einen
	 * {@link GebuehrenTatbestand}.
	 * 
	 * Bei {@link AuslagenTatbestand}, {@link GebuehrenRahmenTatbestand},
	 * {@link GebuehrenAnrechnungsTatbestand} und
	 * {@link GebuehrenErhoehungsTatbestand} wird geklont, um nicht die Werte im
	 * Original zu ändern.
	 * 
	 * @return einen {@link GebuehrenTatbestand}
	 * @throws NumberFormatException wenn der {@link NumberExtractor} bei einem
	 *                               {@link JFormattedTextField} eine solche
	 *                               Exception wirft.
	 * @throws NullPointerException  wenn kein {@link GebuehrenTatbestand} in der
	 *                               {@link JList} ausgewählt ist.
	 */
	private GebuehrenTatbestand leseFelder() throws NumberFormatException, NullPointerException {
		rueckgabewert = getSelectedItem();
		final NumberExtractor extractor = new NumberExtractor(ftfAuslagenHoehe, lblAuslagenHoehe.getText());
		if (rueckgabewert == null) {
			throw new NullPointerException(
					"Kein Gebührentatbestand ausgewählt. Wählen Sie bitte einen Gebührentatbestand aus!");
		} else if (rueckgabewert instanceof MehrfachPauschalTatbestand
				&& ((MehrfachPauschalTatbestand) rueckgabewert).getAnzahl() < 0) {
			int anzahl = extractor.extractInteger();
			rueckgabewert = ((MehrfachPauschalTatbestand) rueckgabewert).clone();
			((MehrfachPauschalTatbestand) rueckgabewert).setAnzahl(anzahl);
		} else if (rueckgabewert instanceof AuslagenTatbestand
				&& ((((AuslagenTatbestand) rueckgabewert).getBetrag() == null)
				|| ((AuslagenTatbestand) rueckgabewert).getBetrag().compareTo(Euro.ofCents(0L)) < 0)) {
			double auslagen = extractor.extractDouble();
			System.out.println("Auslagen: " + auslagen);
			rueckgabewert = ((AuslagenTatbestand) rueckgabewert).clone();
			((AuslagenTatbestand) rueckgabewert).setBetrag(Euro.ofEuros(auslagen));
			System.out.println("AuslagenBetrag: " + ((AuslagenTatbestand) rueckgabewert).getBetrag());
		} else if (rueckgabewert instanceof PauschalTatbestand) {
			rueckgabewert = (PauschalTatbestand) rueckgabewert;
		} else if (rueckgabewert instanceof GebuehrenErhoehungsTatbestand
				&& ((GebuehrenErhoehungsTatbestand) rueckgabewert).getAnzahl() < 0) {
			int anzahl = extractor.extractInteger();
			rueckgabewert = ((GebuehrenErhoehungsTatbestand) rueckgabewert).clone();
			((GebuehrenErhoehungsTatbestand) rueckgabewert).setAnzahl(anzahl);
		} else if (rueckgabewert instanceof GebuehrenAnrechnungsTatbestand) {
			double satz = extractor.extractDouble();
			rueckgabewert = ((GebuehrenAnrechnungsTatbestand) rueckgabewert).clone();
			((GebuehrenAnrechnungsTatbestand) rueckgabewert).setSatz(satz);
		} else if (rueckgabewert instanceof GebuehrenRahmenTatbestand) {
			double satz = extractor.extractDouble();
			rueckgabewert = ((GebuehrenRahmenTatbestand) rueckgabewert).clone();
			((GebuehrenRahmenTatbestand) rueckgabewert).setSatz(satz);
		}
		return rueckgabewert;
	}

}
