/**
 * GebuehrenTatbestandHandler.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.datatransfer.Transferable;
import java.awt.event.InputEvent;

import javax.swing.DropMode;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.TransferHandler;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTableModel;

/**
 * TransferHandler zum Weitergeben von GebührenTatbeständen
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 04.08.2014
 */
public class GebuehrenTransferHandler extends TransferHandler {

	private static final long serialVersionUID = -35313196581225380L;

	/**
	 * Konstruktor: Erzeugt einen TransferHandler zum Weitergeben von
	 * GebührenTatbeständen und bereitet die JTable schonend darauf vor.
	 * 
	 * @param table die JTable, für die der TransferHandler tätig werden soll. Diese
	 *              wird nicht gespeichert, sondern nur Drag'n'Drop-bereit
	 *              geschaltet.
	 * 
	 */
	public GebuehrenTransferHandler(JTable table) {
		super();
		table.setDragEnabled(true);
		table.setDropMode(DropMode.ON_OR_INSERT_ROWS);
		table.setFillsViewportHeight(true);
		table.addMouseMotionListener(new GebuehrenDragListener());
	}

	/**
	 * Causes the Swing drag support to be initiated. This is called by the various
	 * UI implementations in the javax.swing.plaf.basic package if the dragEnabled
	 * property is set on the component. This can be called by custom UI
	 * implementations to use the Swing drag support. This method can also be called
	 * by a Swing extension written as a subclass of JComponent to take advantage of
	 * the Swing drag support. The transfer will not necessarily have been completed
	 * at the return of this call (i.e. the call does not block waiting for the
	 * drop). The transfer will take place through the Swing implementation of the
	 * java.awt.dnd mechanism, requiring no further effort from the developer. The
	 * exportDone method will be called when the transfer has completed.
	 * 
	 * @param comp   the component holding the data to be transferred; provided to
	 *               enable sharing of TransferHandlers
	 * @param ie     the event that triggered the transfer
	 * @param action the transfer action initially requested; either COPY, MOVE or
	 *               LINK; the DnD system may change the action used during the
	 *               course of the drag operation
	 * 
	 * @see javax.swing.TransferHandler#exportAsDrag(javax.swing.JComponent,
	 *      java.awt.event.InputEvent, int)
	 */
	@Override
	public void exportAsDrag(JComponent comp, InputEvent ie, int action) {
		super.exportAsDrag(comp, ie, action);
	}

	/**
	 * Returns the type of transfer actions supported by the source; any bitwise-OR
	 * combination of COPY, MOVE and LINK. Some models are not mutable, so a
	 * transfer operation of MOVE should not be advertised in that case. Returning
	 * NONE disables transfers from the component.
	 * 
	 * Gibt TransferHandler.MOVE zurück, wenn es sich um eine JTable mit einem
	 * {@link GebuehrenTableModel} als Model handelt. Sonst NONE.
	 * 
	 * @param comp the component holding the data to be transferred; provided to
	 *             enable sharing of TransferHandlers
	 * @return
	 * 
	 * @see javax.swing.TransferHandler#getSourceActions(javax.swing.JComponent)
	 */
	@Override
	public int getSourceActions(JComponent comp) {
		return MOVE;
	}

	/**
	 * Creates a Transferable to use as the source for a data transfer. Returns the
	 * representation of the data to be transferred, or null if the component's
	 * property is null
	 * 
	 * @param c the component holding the data to be transferred; provided to enable
	 *          sharing of TransferHandlers
	 * @return
	 * 
	 * @see javax.swing.TransferHandler#createTransferable(javax.swing.JComponent)
	 */
	@Override
	public Transferable createTransferable(JComponent c) {
		if (isJTableValidSource(c)) {
			JTable t = (JTable) c;
			GebuehrenTableModel gttm = (GebuehrenTableModel) t.getModel();
			return new GebuehrenTransferable(gttm.getRow(t.getSelectedRow()));
		} else
			return null;
	}

	/**
	 * Invoked after data has been exported. This method should remove the data that
	 * was transferred if the action was MOVE. This method is implemented to do
	 * nothing since MOVE is not a supported action of this implementation
	 * (getSourceActions does not include MOVE).
	 * 
	 * 
	 * @param source the component that was the source of the data
	 * @param data   The data that was transferred or possibly null if the action is
	 *               NONE.
	 * @param action the actual action that was performed
	 * 
	 * @see javax.swing.TransferHandler#exportDone(javax.swing.JComponent,
	 *      java.awt.datatransfer.Transferable, int)
	 */
	@Override
	public void exportDone(JComponent source, Transferable data, int action) {
		if ((isJTableValidSource(source) && action == MOVE)) {
			JTable t = (JTable) source;
			GebuehrenTableModel model = (GebuehrenTableModel) t.getModel();
			model.removeRow(t.getSelectedRow());
		}
	}

	/**
	 * This method is called repeatedly during a drag gesture and returns true if
	 * the area below the cursor can accept the transfer, or false if the transfer
	 * will be rejected. For example, if a user drags a color over a component that
	 * accepts only text, the canImport method for that component's TransferHandler
	 * should return false.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the import can happen, false otherwise
	 * 
	 * @see javax.swing.TransferHandler#canImport(javax.swing.TransferHandler.TransferSupport)
	 */
	@Override
	public boolean canImport(TransferSupport support) {
		if (support.isDrop())
			support.setShowDropLocation(true);
		if (!support.isDataFlavorSupported(GebuehrenTransferable.GEBUEHREN_FLAVOR)) {
			Kostenrechner.getLogger().info(
					String.format("Flavor not supported: %s!", GebuehrenTransferable.GEBUEHREN_FLAVOR.toString()));
			return false;
		}
		return true;
	}

	/**
	 * This method is called on a successful drop (or paste) and initiates the
	 * transfer of data to the target component. This method returns true if the
	 * import was successful and false otherwise.
	 * 
	 * Causes a transfer to occur from a clipboard or a drag and drop operation. The
	 * Transferable to be imported and the component to transfer to are contained
	 * within the TransferSupport. While the drag and drop implementation calls
	 * canImport to determine the suitability of a transfer before calling this
	 * method, the implementation of paste does not. As such, it cannot be assumed
	 * that the transfer is acceptable upon a call to this method for paste. It is
	 * recommended that canImport be explicitly called to cover this case.
	 * 
	 * Note: The TransferSupport object passed to this method is only valid for the
	 * duration of the method call. It is undefined what values it may contain after
	 * this method returns.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the data was inserted into the component, false otherwise
	 * @throws NullPointerException if <code>support</code> is null
	 * 
	 * @see javax.swing.TransferHandler#importData(javax.swing.TransferHandler.TransferSupport)
	 */
	@Override
	public boolean importData(TransferSupport support) throws NullPointerException {
		if (support == null)
			throw new NullPointerException("TransferSupport ist null!");
		if (!isJTableValid((JComponent) support.getComponent())) {
			return false;
		}
		if (!canImport(support)) {
			return false;
		}
		String woNull = "(?)";
		try {
			GebuehrenTatbestand gt = (GebuehrenTatbestand) support.getTransferable()
					.getTransferData(GebuehrenTransferable.GEBUEHREN_FLAVOR);
			JTable table = (JTable) support.getComponent();
			if (table == null)
				woNull = "table";
			GebuehrenTableModel gttm = (GebuehrenTableModel) table.getModel();
			if (gttm == null)
				woNull = "gttm";
			/*
			 * Herausfinden, wo fallen gelassen wurde. Vorher prüfen, ob isDrop, denn sonst
			 * kann getDropLocation einen Fehler werfen. Ansonsten oder wenn die
			 * DropLocation außerhalb des Zeilenbereichs der Table ist, auf -1 setzen.
			 */
			int dropPoint = -1;
			if (support.isDrop()) {
				javax.swing.JTable.DropLocation dropLocation = (JTable.DropLocation) support.getDropLocation();
				dropPoint = dropLocation.getRow();
			}
			/*
			 * Wenn innerhalb des gültigen Zeilenbereichs fallen gelassen: Element an
			 * bestimmter Stelle einfügen.
			 */
			if (dropPoint > -1 && dropPoint < gttm.getRowCount()) {
				gttm.addRowAt(dropPoint, gt);
				/*
				 * Falls das bisherige Element in derselben Tabelle war (selection > -1), muss
				 * sichergestellt werden, dass das richtige Element markiert wird. Denn die
				 * markierte Zeile wird gelöscht.
				 */
			} else {
				/*
				 * Wenn das Element außerhalb des gültigen Zeilenbereichs fallen gelassen wurde,
				 * wird es ans Ende gesetzt. Dann muss auf andere Weise sichergestellt werden,
				 * dass das richtige Element markiert bleibt (zum Verschieben muss es vorher
				 * markiert worden sein).
				 */
				gttm.addRow(gt);
			}
			return true;
		} catch (Exception e) {
			FehlerHelper.zeigeFehler(String.format(
					"Fehler beim Gebührenkopieren: Fehler (%s) beim Kopieren eines Gebührentatbestands (%s): %s",
					e.getClass().toString(), woNull, e.getLocalizedMessage()), e);
			return false;
		}
	}

	/**
	 * Prüft, ob die JComponent eine taugliche Quelle von GebührenTatbeständen ist.
	 * 
	 * @param comp the component holding the data to be transferred; provided to
	 *             enable sharing of TransferHandlers
	 * @return true, wenn {@link GebuehrenTransferHandler#isJTableValid(JComponent)
	 *         isJTableValid} und zusätzlich eine Zeile markiert ist.
	 */
	private boolean isJTableValidSource(JComponent comp) {
		return isJTableValid(comp) && ((JTable) comp).getSelectedRow() >= 0;
	}

	/**
	 * Prüft, ob die JComponent fähig ist, GebührenTatbestände abzugeben oder
	 * aufzunehmen
	 * 
	 * @param comp the component; provided to enable sharing of TransferHandlers
	 * @return true, comp eine JTable != null und das TableModel ein
	 *         GebuehrenTatbestandTableModel ist.
	 */
	private boolean isJTableValid(JComponent comp) {
		return comp != null && comp instanceof JTable && ((JTable) comp).getModel() instanceof GebuehrenTableModel;
	}

}
