/**
 * CopyrightNoergler.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import javax.swing.JFrame;

import eu.gronos.beschriftungen.controller.BooleanConfigAction;
import eu.gronos.beschriftungen.model.BooleanConfigSchluessel;
import eu.gronos.beschriftungen.util.BooleanConfigOption;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;
import eu.gronos.kostenrechner.interfaces.Noergelnd;
import eu.gronos.kostenrechner.model.DatenschutzZustimmung;
import eu.gronos.kostenrechner.view.helpanderror.DatenschutzZustimmungDialog;

/**
 * Wenn der Datenschutzhinweis noch nicht akzeptiert wurde, dann nörgeln. Wenn
 * nicht akzeptiert wird, dann aufhören. Sonst Vermerk in Registry/Preferences
 * und weiter.
 *
 * @author Peter Schuster (setrok)
 * @date 21 Feb 2019
 *
 */
public class DatenschutzNoergler
		implements Noergelnd<BooleanConfigOption>, AbfrageLieferant<DatenschutzZustimmungDialog> {

	private final JFrame owner;
	private int level;
	private BooleanConfigOption basisOption;
	private BooleanConfigOption anhangOption;
	private DatenschutzZustimmung zustimmung;
	private DatenschutzZustimmungDialog dialog = null;

	/**
	 * @param owner ein {@link JFrame}
	 * @param level entweder {@link DatenschutzZustimmung#OPTION_DSGVO_EMAIL_BASIS}
	 *              oder {@link DatenschutzZustimmung#OPTION_DSGVO_EMAIL_ANHANG}
	 */
	public DatenschutzNoergler(JFrame owner, int level) {
		this.owner = owner;
		this.level = level;
		setZustimmung(new DatenschutzZustimmung());
		basisOption = new BooleanConfigOption(new BooleanConfigSchluessel("dsgvo-email-basis", false));
		getZustimmung().emailBasisZugestimmt = basisOption.getValue();
		anhangOption = new BooleanConfigOption(new BooleanConfigSchluessel("dsgvo-email-anhang", false));
		getZustimmung().emailAnhangZugestimmt = anhangOption.getValue();
	}

	/**
	 * Wenn der Datenschutzhinweis schon akzeptiert wurde, dann <code>true</code>.
	 * Sonst nörgeln. Wenn nicht akzeptiert wird, dann <code>false</code>
	 * zurückgeben. Sonst Vermerk in Registry/Preferences und <code>true</code>
	 * zurückgeben.
	 * 
	 * @return <code>true</code>, wenn bereits zuvor akzeptiert oder nach
	 *         Aufforderung akzeptiert. Sonst false
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Noergelnd#noergle()
	 */
	@Override
	public boolean noergle() {
		if (getZustimmung().get(level)) {
			// Laut Registry schon akzeptiert TODO differenzieren
			return true;
		} else {
			setZustimmung(setDialog(new DatenschutzZustimmungDialog(owner, level, getZustimmung())).showDialogAndAsk());
			setDialog(null);
			if (getZustimmung() != null) {
				basisOption.setValue(getZustimmung().emailBasisZugestimmt);
				anhangOption.setValue(getZustimmung().emailAnhangZugestimmt);
				return getZustimmung().get(level);
			}
			return false;
		}
	}

	/**
	 * Für das Speichern in der Registry oder den Preferences nötige Felder
	 * 
	 * @return gibt {@link option} als {@link BooleanConfigAction} zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Noergelnd#getOption()
	 */
	@Override
	public BooleanConfigOption getOption() {
		if (DatenschutzZustimmung.OPTION_DSGVO_EMAIL_ANHANG == level) {
			return anhangOption;
		} else {
			return basisOption;
		}
	}

	/**
	 * Die Methode setzt wieder den Standardwert der {@link #getOption()} in die
	 * Registry/Preferences.
	 */
	public void reset() {
		getOption().reset();
	}

	/**
	 * @return gibt {@link #zustimmung} als {@link DatenschutzZustimmung} zurück.
	 */
	public DatenschutzZustimmung getZustimmung() {
		return zustimmung;
	}

	/**
	 * @param zustimmung d. {@link #zustimmung}, d. gesetzt werden soll als
	 *                   {@link DatenschutzZustimmung}.
	 */
	public void setZustimmung(DatenschutzZustimmung zustimmung) {
		this.zustimmung = zustimmung;
	}

	/**
	 * @return gibt {@link #dialog} als {@link DatenschutzZustimmungDialog} zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	@Override
	public DatenschutzZustimmungDialog getDialog() {
		return dialog;
	}

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als
	 *               {@link DatenschutzZustimmungDialog}.
	 */
	private DatenschutzZustimmungDialog setDialog(DatenschutzZustimmungDialog dialog) {
		this.dialog = dialog;
		return this.dialog;
	}
}
