/*
 * FehlerHelper.java
 * eu.gronos.kostenrechner.view.helpanderror (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import java.io.StringWriter;
import java.util.logging.Level;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.util.files.ThrowableWrapper;
import eu.gronos.kostenrechner.view.helpanderror.FehlerDialog;

/**
 * Klasse dient dazu die Message, LocalizedMessage und den StackTrace eines
 * Fehlers in einen {@link String} zu schreiben.
 * 
 * Außerdem steuert die Klasse über {@link #zeigeFehler(String, Throwable)} den
 * Aufruf des {@link FehlerDialog}s.
 *
 * @author Peter Schuster (setrok)
 * @date 24 Apr 2019
 *
 */
public class FehlerHelper {

	private final Throwable throwable;
	private final String meldung;
	private final BeschriebeneAktion emailSchreibenFehler;
	private static FehlerDialog dialog = null;

	/**
	 * @param throwable ein {@link Throwable}, um die {@link StackTraceElement}s
	 *                  auszulesen
	 * @param meldung   die Fehlermeldung
	 */
	public FehlerHelper(Throwable throwable, String meldung) {
		this.throwable = throwable;

		StackTraceElement stack = throwable.getStackTrace()[0];
		String klasse = stack.getClassName();
		this.meldung = repariereMeldung(meldung, throwable);
		String methode = stack.getMethodName();
		Kostenrechner.getLogger().logp(Level.WARNING, klasse, methode, meldung, throwable);
		emailSchreibenFehler = new SchreibeEmailFehler(
				(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(72045), this);
	}

	/**
	 * Die Methode dient dazu, TODO (kommentieren)
	 * 
	 * @param meldung
	 * @param stack   TODO (kommentieren)
	 */
	private String repariereMeldung(String meldung, Throwable throwable) {
		if (meldung == null || meldung.isEmpty()) {
			if (throwable.getLocalizedMessage() != null && !throwable.getLocalizedMessage().isEmpty()) {
				return throwable.getLocalizedMessage();
			} else if (throwable.getMessage() != null && !throwable.getMessage().isEmpty()) {
				return throwable.getMessage();
			} else {
				return throwable.getClass().toString();
			}
		} else {
			return meldung;
		}
	}

	/**
	 * Die Methode zeigt über {@link FehlerDialog} einen Fehlertext an. Wenn
	 * {@link Kostenrechner} noch nicht existiert oder noch nicht angezeigt wird,
	 * zeigt sie den Fehlertext auf der Konsole an.
	 * 
	 * @param meldung   den anzuzeigenden Fehlertext
	 * @param throwable das {@link Throwable} (Oberklasse von {@link Exception}),
	 *                  das geworfen wurde.
	 */
	public static void zeigeFehler(String meldung, Throwable throwable) {
		if (Kostenrechner.getInstance() == null || !Kostenrechner.getInstance().isVisible()) {
			if (Kostenrechner.getLogger() == null) {
				System.err.println(meldung);
				throwable.printStackTrace();
			} else {
				StackTraceElement stack = throwable.getStackTrace()[0];
				String klasse = stack.getClassName();
				String methode = stack.getMethodName();
				Kostenrechner.getLogger().logp(Level.SEVERE, klasse, methode, meldung, throwable);
			}
		} else {
			setDialog(new FehlerDialog(meldung, new FehlerHelper(throwable, meldung))).showDialog();
			setDialog(null);
		}
	}

	/**
	 * Die Methode marshallt den {@link Throwable}, also dessen
	 * {@link Throwable#getMessage()}, die {@link Throwable#getLocalizedMessage()}
	 * und {@link Throwable#getStackTrace()}, in einen XML-{@link String}.
	 * 
	 * @param formatted schaltet, ob das XML formatiert sein soll, also ob der
	 *                  Marshaller Zeilenumbrüche und Einrückungen bauen soll, siehe
	 *                  {@link Marshaller#JAXB_FORMATTED_OUTPUT}
	 * @return einen String mit den XML-Daten oder einen Fehlertext, falls die
	 *         Konvertiertung nicht klappt
	 */
	public String baueFehlerXmlString(boolean formatted) {
		StringWriter sw = new StringWriter();
		try {
			JAXBContext context = JAXBContext.newInstance(ThrowableWrapper.class);
			Marshaller marshaller = context.createMarshaller();
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(formatted));
			marshaller.marshal(new ThrowableWrapper(throwable), sw);
		} catch (JAXBException e) {
			e.printStackTrace();
			return "Fehler beim Konvertieren in XML: " + e.getLocalizedMessage();
		}
		return sw.toString();
	}

	/**
	 * @return gibt {@link #dialog} als {@link FehlerDialog} zurück.
	 */
	public static FehlerDialog getDialog() {
		return dialog;
	}

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als
	 *               {@link FehlerDialog}.
	 */
	public static FehlerDialog setDialog(FehlerDialog dialog) {
		FehlerHelper.dialog = dialog;
		return dialog;
	}

	/**
	 * @return gibt {@link #emailSchreibenFehler} als {@link BeschriebeneAktion}
	 *         zurück.
	 */
	public BeschriebeneAktion getEmailAktion() {
		return emailSchreibenFehler;
	}

	/**
	 * @return gibt {@link #meldung} als {@link String} zurück, also die
	 *         Fehlermeldung
	 */
	public String getMeldung() {
		return meldung;
	}

	/**
	 * @return gibt {@link #throwable} als {@link Throwable} zurück.
	 */
	public Throwable getThrowable() {
		return throwable;
	}
}