/*
 * Schwellenwert92Action.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import java.awt.event.ActionEvent;

import javax.swing.AbstractButton;
import javax.swing.JCheckBoxMenuItem;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.controller.BooleanConfigAction;
import eu.gronos.beschriftungen.interfaces.Vorsorgend;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.LangVorsorgeBeschriftung;
import eu.gronos.beschriftungen.model.SchluesselBeschriftung;
import eu.gronos.beschriftungen.util.IntegerConfigOption;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;
import eu.gronos.kostenrechner.view.menus.ExtrasJMenu;
import eu.gronos.kostenrechner.view.menus.Schwellenwert92Dialog;

/**
 * Eine {@link BooleanConfigAction}, die über die Oberklasse den Einsatz von §
 * 92 II ZPO überhaupt überwacht, bei Aktivierung den
 * {@link Schwellenwert92Dialog} aufruft und danach die
 * {@link IntegerConfigOption} anpasst.
 *
 * @author Peter Schuster (setrok)
 * @date 19.03.2020
 *
 */
public class Paragraph92Action extends BooleanConfigAction implements AbfrageLieferant<Schwellenwert92Dialog> {
	private static final long serialVersionUID = -4432788831023684181L;
	private final IntegerConfigOption schwellenOption;
	private Schwellenwert92Dialog dialog = null;
	private final SchluesselBeschriftung schwellenBeschriftung;

	/**
	 * Konstruktor:
	 * 
	 * @param beschriftung          die {@link LangBeschriftung} für die
	 *                              {@link BeschriebeneAktion} selbst.
	 * @param schwellenBeschriftung die {@link SchluesselBeschriftung} für den
	 *                              {@link HinzufuegenDialog}, der aufgerufen wird
	 * @param schwellenOption       die {@link IntegerConfigOption}, aus der die
	 *                              {@link BeschriebeneAktion} den aktuellen Wert
	 *                              holt. Die muss im {@link ExtrasJMenu} bleiben,
	 *                              damit sie global abgefragt werden kann.
	 */
	public Paragraph92Action(SchluesselBeschriftung beschriftung, SchluesselBeschriftung schwellenBeschriftung,
			IntegerConfigOption schwellenOption) {
		super(beschriftung);
		this.schwellenOption = schwellenOption;
		this.schwellenBeschriftung = schwellenBeschriftung;
	}

	/**
	 * Die Methode ruft den {@link Schwellenwert92Dialog} auf.
	 * 
	 * @param e ein {@link ActionEvent}
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		// Erst das Häkchen anpassen
		super.actionPerformed(e);
		// Dialog nur aufrufen, wenn jetzt ein Häkchen gesetzt
		if (!super.getValue().booleanValue()) {
			setButtonText();
			return;
		}

		Integer value = setDialog(new Schwellenwert92Dialog(Kostenrechner.getInstance(), schwellenBeschriftung,
				schwellenOption.getValue())).showDialogAndAsk();
		setDialog(null);

		if (value != null) {
			schwellenOption.setValue(value);
			setButtonText();
		}
	}

	/**
	 * 
	 * @return gibt den {@link Schwellenwert92Dialog} zurück oder <code>null</code>,
	 *         wenn der gerade nicht geöffnet ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	@Override
	public Schwellenwert92Dialog getDialog() {
		return dialog;
	}

	/**
	 * Die Methode überschreibt {@link BooleanConfigAction#setButtonText()} und
	 * formattiert den Text des {@link JCheckBoxMenuItem}, so dass auch die Schwelle
	 * darin steht.
	 * 
	 * Dadurch verhindert sie auch das unformattierte Beschriften mit
	 * {@link BooleanConfigAction#setButtonText()}.
	 */
	@Override
	protected void setButtonText() {
		setButtonText(super.getValue(), schwellenOption.getValue());
	}

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als
	 *               {@link Schwellenwert92Dialog}.
	 * 
	 * @return den {@link Schwellenwert92Dialog} zum Weiterarbeiten
	 */
	private Schwellenwert92Dialog setDialog(Schwellenwert92Dialog dialog) {
		this.dialog = dialog;
		return dialog;
	}

	/**
	 * Beim Klicken wird der Text des {@link AbstractButton} aus den
	 * {@link LangVorsorgeBeschriftung#getAlternativeTitles()} der Auswahl
	 * angepasst, bei {@link Boolean#TRUE} der index 1, bei {@link Boolean#FALSE}
	 * der index 0.
	 * 
	 * Die Beschriftung wird auch formattiert.
	 * 
	 * @param selected ob 92 II überhaupt aktiviert ist aus der Oberklasse
	 *                 {@link BooleanConfigAction#getValue()}
	 */
	private void setButtonText(boolean selected, Integer schwelle) {
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		if (selected) {
			beschrifter.changeTo((Vorsorgend<?>) getBeschriftung(), 1, schwelle);
		} else {
			beschrifter.changeTo((Vorsorgend<?>) getBeschriftung(), 0);
		}
		beschrifter.beschrifte(this, getBeschriftung());
	}

}
