/**
 * EmailUtils.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import java.awt.Desktop;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;

import javax.swing.Action;
import javax.xml.bind.JAXB;

import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.data.EmailData;
import eu.gronos.kostenrechner.data.Ersetzung;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;
import eu.gronos.kostenrechner.model.DatenschutzZustimmung;
import eu.gronos.kostenrechner.util.ErsetzungComparator;
import eu.gronos.kostenrechner.view.helpanderror.DatenschutzZustimmungDialog;

/**
 * Die Oberklassen für alle SchreibeEmail-Klassen hält Methoden für
 * Feedback-E-Mails bereit: {@link #schreibe(String, String)} und
 * {@link #urlEncode(String)}
 *
 * @author Peter Schuster (setrok)
 * @date 28.03.2016
 *
 */
public abstract class SchreibeEmail extends BeschriebeneAktion
		implements AbfrageLieferant<DatenschutzZustimmungDialog> {

	private static final long serialVersionUID = 9027687725799709474L;
	private static final String E_MAIL_FEHLER_URI_SYNTAX = "URI-Syntax falsch: ";
	private static final String E_MAIL_FEHLER_ENCODING = "Zeichenkodierung nicht unterstützt: ";
	private static final String E_MAIL_FEHLER_TEXT = "Fehler beim Aufruf des Standard-E-Mail-Programms: ";
	private final EmailData eMailData;
	/**
	 * Die URI-Einleitung für die Betreffzeile als String.
	 */
	protected static final String SUBJECT2 = "?subject=";
	/**
	 * Die URI-Einleitung für den E-Mail-Text als String.
	 */
	protected static final String BODY2 = "&BODY=";
	/**
	 * Die URI-Einleitung für E-Mails als String.
	 */
	protected static final String MAILTO = "mailto:";

	/**
	 * Ein doppelter Zeilenumbruch als String.
	 */
	protected static final String NEUER_ABSATZ = "\n\n";

	private DatenschutzNoergler noergler = null;

	/**
	 * Konstruktor.
	 * 
	 * @param name             {@link Action#NAME}
	 * @param shortDescription {@link Action#SHORT_DESCRIPTION}
	 * @param mnemonic         {@link Action#MNEMONIC_KEY}
	 * @param actionCommandKey {@link Action#ACTION_COMMAND_KEY}
	 */
	protected SchreibeEmail(LangBeschriftung beschriftung) {
		super(beschriftung);

		eMailData = JAXB.unmarshal(getClass().getClassLoader().getResourceAsStream("resources/email.xml"),
				EmailData.class);
		eMailData.replaceWiths.sort(new ErsetzungComparator());
		// TODO entfernen JAXB.marshal(eMailData, System.out);
	}

	/**
	 * @return gibt den {@link #getDialog()} des {@link DatenschutzNoergler}s zurück
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	@Override
	public DatenschutzZustimmungDialog getDialog() {
		if (getNoergler() == null)
			return null;
		else
			return getNoergler().getDialog();
	}

	/**
	 * emailSchreiben lässt eine Beschwerde-E-Mail im Standard-E-Mail-Client
	 * schreiben.
	 * 
	 * @param subject die Betreffzeile der E-Mail als String
	 * @param body    der Inhalt der E-Mail als String
	 * 
	 * @url "http://stackoverflow.com/questions/12449654/clickable-email-address-as-a-jlabel-in-java"
	 * @url "http://www.2ality.com/2010/12/simple-way-of-sending-emails-in-java.html"
	 */
	protected final void schreibe(final String subject, final String body) {
		try {
			final URI mailtoURI = new URI(
					MAILTO + eMailData.helpDeskEmail + SUBJECT2 + urlEncode(subject) + BODY2 + urlEncode(body));
			// Launches the mail composing window of the user default mail
			// client (Dateien mitschicken geht wohl nicht...)
			Desktop.getDesktop().mail(mailtoURI);
			// den Browser aufzurufen dauert länger:
			// Desktop.getDesktop().browse(mailtoURI);
		} catch (UnsupportedEncodingException uee) {
			FehlerHelper.zeigeFehler(E_MAIL_FEHLER_TEXT + E_MAIL_FEHLER_ENCODING + uee.getLocalizedMessage(), uee);
		} catch (URISyntaxException use) {
			FehlerHelper.zeigeFehler(E_MAIL_FEHLER_TEXT + E_MAIL_FEHLER_URI_SYNTAX + use.getLocalizedMessage(), use);
		} catch (IOException ioe) {
			FehlerHelper.zeigeFehler(E_MAIL_FEHLER_TEXT + ioe.getLocalizedMessage(), ioe);
		}
	}

	/**
	 * Die Methode dient dazu, zum E-Mail-Versand eine Zeichenkette URL-konform zu
	 * kodieren.
	 * 
	 * @param str die zu kodierende Zeichenkette
	 * @url "http://www.2ality.com/2010/12/simple-way-of-sending-emails-in-java.html"
	 * @return die kodierte Zeichenkette
	 */
	protected final String urlEncode(String str) {
		try {
			return URLEncoder.encode(str, "UTF-8").replace("+", "%20");
		} catch (UnsupportedEncodingException uee) {
			throw new RuntimeException(uee);
		}
	}

	/**
	 * Die Methode ruft den {@link DatenschutzNoergler} auf und differenziert
	 * danach, ob es nur den Text aus der <code>basisFrage</code> schickt oder auch
	 * den <code>erweiterterText</code>.
	 * 
	 * So muss sich die abgeleitete Klasse nicht um den {@link DatenschutzNoergler}
	 * und das level der {@link DatenschutzZustimmung} kümmern.
	 * 
	 * @param subject         den immer gesendeten Betreff der E-Mail
	 * @param basisFrage      der Text, der geschickt wird, wenn jedenfalls
	 *                        {@link DatenschutzZustimmung#emailBasisZugestimmt}
	 *                        wurde
	 * @param erweiterterText der Text, der zusätzlich geschickt wird, nur dann,
	 *                        wenn auch
	 *                        {@link DatenschutzZustimmung#emailAnhangZugestimmt}
	 *                        zugestimmt wurde.
	 */
	protected void schreibeDifferenziert(final String subject, String basisFrage, String erweiterterText) {
		setNoergler(
				new DatenschutzNoergler(Kostenrechner.getInstance(), DatenschutzZustimmung.OPTION_DSGVO_EMAIL_ANHANG))
						.noergle();
		DatenschutzZustimmung zustimmung = getNoergler().getZustimmung();
		setNoergler(null);
		if (zustimmung != null && zustimmung.emailBasisZugestimmt) {
			// Wenn dem Anhang nicht zugestimmt wurde, reinen E-Mail-Text schreiben
			final StringBuilder body = new StringBuilder(basisFrage);
			if (zustimmung.emailAnhangZugestimmt) {
				// Wenn auch dem Anhang zugestimmt wird, auch weitere Daten schicken
				body.append(erweiterterText);
			}
			schreibe(subject, body.toString());
		}
	}

	/**
	 * Die Methode ersetzt die XML-Tags gemäß {@link EmailData#replaceWiths} bzw.
	 * {@link Ersetzung#getReplace()} und {@link Ersetzung#getWith()} und mal den
	 * XML-Text damit kürzer.
	 * 
	 * @param lang der Ursprungs-XML-String
	 * @return den verkleinerten XML-String
	 */
	protected final String verkleinere(String lang) {
		String string = lang;
		eMailData.replaceWiths.sort(new ErsetzungComparator());

		for (Ersetzung e : eMailData.replaceWiths) {
			string = string.replaceAll(e.getReplace(), e.getWith());
		}
		if (!eMailData.replaceWiths.isEmpty())
			System.out.println(eMailData.replaceWiths.get(0).getReplace() + " / "
					+ eMailData.replaceWiths.get(eMailData.replaceWiths.size() - 1).getReplace());
		return string;
	}

	/**
	 * @return gibt {@link #noergler} als {@link DatenschutzNoergler} zurück.
	 */
	protected DatenschutzNoergler getNoergler() {
		return noergler;
	}

	/**
	 * @param noergler d. {@link #noergler}, d. gesetzt werden soll als
	 *                 {@link DatenschutzNoergler}.
	 */
	protected DatenschutzNoergler setNoergler(DatenschutzNoergler noergler) {
		this.noergler = noergler;
		return this.noergler;
	}
}
