package eu.gronos.kostenrechner.data.baumbach;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlIDREF;

import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus.Numerus;
import eu.gronos.kostenrechner.logic.TenorToken;

/**
 * Ein {@link Beteiligter}, der auf mehrere {@link Beteiligter} verweist.
 *
 * @author Peter Schuster (setrok)
 * @date 24.10.2021
 *
 */
public class MehrfachBeteiligter extends Beteiligter {

	private static final long serialVersionUID = -2607492610898941487L;
	private List<Beteiligter> beteiligte;
	private final boolean gesamtschuldnerisch;

	/**
	 * Konstruktor: akzeptiert eine {@link List} aus {@link Beteiligter}n
	 * 
	 * @param beteiligte          {@link List} aus {@link Beteiligter}n
	 * @param gesamtschuldnerisch <code>true</code>, wenn die {@link Beteiligter}n
	 *                            gesamtschuldnerisch verbunden sind
	 */
	public MehrfachBeteiligter(List<Beteiligter> beteiligte, boolean gesamtschuldnerisch) {
		super();
		setBeteiligte(beteiligte);
		this.gesamtschuldnerisch = gesamtschuldnerisch;

		// Typ, Genus/Numerus und so setzen
		setTyp(beteiligte.get(0).getTyp());
		final Numerus pl = beteiligte.size() > 1 ? Numerus.PLURAL : Numerus.SINGULAR;
		// setGenusNumerus(beteiligte.get(0).getGenusNumerus() + pl);
		setGenusNumerus(GenusNumerus.of(beteiligte.get(0).getGenusNumerus().getGenus(), pl));
	}

	/**
	 * Standard-Konstruktor erstellt eine leere {@link #beteiligte} {@link List}
	 * 
	 */
	public MehrfachBeteiligter() {
		this(new ArrayList<>(), false);
	}

	@Override
	public boolean isPlural() {
		return beteiligte.size() > 1;
	}

	/**
	 * @return gibt true zurück, wenn es sich um einen BaumbachGesamtschuldner
	 *         handelt (hier also immer true).
	 */
	public boolean isGesamtschuldnerschaft() {
		return this.gesamtschuldnerisch;
	}

	/**
	 * @return gibt {@link #gesamtschuldnerisch} als {@link boolean} zurück.
	 */
	public boolean isGesamtschuldnerisch() {
		return gesamtschuldnerisch;
	}

	/**
	 * @return die {@link List#size()} der internen {@link #beteiligte}
	 */
	public int size() {
		return beteiligte.size();
	}

	/**
	 * @param other eine {@link List} von {@link Beteiligter}
	 * @return
	 */
	public boolean listEquals(List<Beteiligter> other) {
		return beteiligte.equals(other);
	}

	/**
	 * @return eine Kopie von {@link #beteiligte}
	 * 
	 */
//	@XmlTransient
	@XmlElement(name="beteiligtenVerweis")
	@XmlIDREF
	public List<Beteiligter> getBeteiligte() {
		return new ArrayList<Beteiligter>(beteiligte);
	}

	/**
	 * @return
	 */
	@XmlAttribute(name = "ids")
	public String getIds() {
		if (beteiligte == null || beteiligte.isEmpty())
			return null;
		return beteiligte.stream()//
				.map(b -> b.getId())//
				.collect(Collectors.joining(","));
	}

	@Override
	public String parteiBezeichner(Casus casus, int laufendeNummer, boolean einzigerSeinerArt) {
		// public String parteiBezeichner(int casus, int laufendeNummer, boolean
		// einzigerSeinerArt) {
		final StringBuilder builder = new StringBuilder(parteiBezeichnerListe(beteiligte, beteiligte, casus));
		formatGesamtschuldner(builder);
		return builder.toString();
	}

	@Override
	public String kurzBezeichner(Casus casus) {
		final StringBuilder builder = new StringBuilder();
		for (final Beteiligter bet : beteiligte) {
			builder.append(bet.kurzBezeichner(casus));
			builder.append(", ");
		}

		behandleKomma(builder);
		formatGesamtschuldner(builder);
		return builder.toString();
	}

	@Override
	public String langBezeichnung() {
		final StringBuilder builder = new StringBuilder();
		for (final Beteiligter bet : beteiligte) {
			builder.append(bet.langBezeichnung());
			builder.append(", ");
		}
		behandleKomma(builder);
		formatGesamtschuldner(builder);
		return builder.toString();
	}

	/**
	 * Die Methode kurzBezeichnerListe dient dazu, eine Liste der repräsentierten
	 * {@link List} mittels deren kurzBezeichner zu erzeugen.
	 * 
	 * @param alleBeteiligte die Gesamt-{@link List}e aller {@link Beteiligter}n,
	 *                       auch die nicht in den Bezeichner aufzunehmenden, für
	 *                       eine Prüfung, ob sie eindeutig sind.
	 * @param beteiligte     die {@link Beteiligter}n, aus denen die
	 *                       Bezeichner-Liste erstellt werden soll
	 * @param casus          eine der Konstanten NOMINATIV, GENITIV, DATIV,
	 *                       AKKUSATIV
	 * @return einen String mit einem passenden Bezeichner etwa Kläger-1, Beklagte-2
	 * 
	 * @see eu.gronos.kostenrechner#Beteiligter#kurzBezeichner(int)
	 */
	static CharSequence kurzBezeichnerListe(List<Beteiligter> alleBeteiligte, List<Beteiligter> beteiligte,
			Casus casus) {
		final StringBuilder builder = new StringBuilder();
//		int typ = -1;
		BeteiligtenTyp typ = null;
		for (final Beteiligter bet : beteiligte) {
			if (bet.getTyp() != typ) {
				typ = bet.getTyp();
				builder.append(bet.kurzBezeichner(casus));
			}
			if (!istEinzigerSeinerArt(bet, alleBeteiligte)) {
				builder.append('-');
				builder.append(bet.getLfdNr());
			}
			builder.append(", ");
		}
		behandleKomma(builder);
		return builder;
	}

	/**
	 * Die Methode entfernt das überflüssige Komma am Ende und ersetzt ein weiteres
	 * Komma und "und"
	 * 
	 * @param builder der bisher erstellte {@link StringBuilder}
	 */
	private static void behandleKomma(StringBuilder builder) {
		builder.deleteCharAt(builder.lastIndexOf(","));
		builder.deleteCharAt(builder.lastIndexOf(TenorToken.LEER.toString()));
		final int pos = builder.lastIndexOf(",");
		if (pos > -1)
			builder.replace(pos, pos + 1, " und");
		final int pos2 = builder.lastIndexOf("  ");
		if (pos2 > -1)
			builder.replace(pos2, pos2 + 2, TenorToken.LEER.toString());
	}

	/**
	 * Die Methode ergänzt nach Bedarf ein BaumbachToken.GESAMTSCHULDNERISCH
	 * 
	 * @param builder der bisher erstellte {@link StringBuilder}
	 */
	private void formatGesamtschuldner(final StringBuilder builder) {
		if (isGesamtschuldnerisch())
			builder.append(TenorToken.LEER.toString() + TenorToken.GESAMTSCHULDNERISCH);
	}

	/**
	 * @param beteiligte d. {@link #beteiligte}, d. gesetzt werden soll als
	 *                   {@link List<Beteiligter>}.
	 */
	private void setBeteiligte(List<Beteiligter> beteiligte) {
		this.beteiligte = new ArrayList<>();
		if (beteiligte != null && beteiligte.size() > 0)
			this.beteiligte.addAll(beteiligte);
	}

	/**
	 * Die Methode ermittelt, ob der angegebene {@link Beteiligter} in der
	 * {@link List}e und der einzige für seinen Beteiligten-Typ ist. Gesamtschuldner
	 * zählen nicht.
	 * 
	 * @param beteiligter der {@link Beteiligter}
	 * @param beteiligte  {@link List} der anderen {@link Beteiligter}n
	 * @return {@code true}, wenn es der einzige Nicht-Gesamtschuldner seines
	 *         {@link Beteiligter#getTyp()}s ist.
	 */
	public static boolean istEinzigerSeinerArt(Beteiligter beteiligter, List<Beteiligter> beteiligte) {
		final BeteiligtenTyp typ = beteiligter.getTyp();
		int count = 0;
		for (Beteiligter b : beteiligte) {
			if (b.getTyp() == typ)
				count++;
		}
		return count < 2;
	}

	/**
	 * Die Methode parteiBezeichnerListe dient dazu, eine Liste der repräsentierten
	 * Beteiligten zu erzeugen.
	 * 
	 * @param alleBeteiligte die Gesamt-{@link List}e aller {@link Beteiligter}n,
	 *                       auch die nicht in den Bezeichner aufzunehmenden, für
	 *                       eine Prüfung, ob sie eindeutig sind.
	 * @param beteiligte     die {@link Beteiligter}n, aus denen die Liste erstellt
	 *                       werden soll
	 * @param casus          eine der Konstanten NOMINATIV, GENITIV, DATIV,
	 *                       AKKUSATIV
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)".
	 * 
	 */
	public static CharSequence parteiBezeichnerListe(List<Beteiligter> alleBeteiligte, List<Beteiligter> beteiligte,
			Casus casus) {
		final StringBuilder builder = new StringBuilder();
//		int typ = -1;
		BeteiligtenTyp typ = null;
		for (final Beteiligter bet : beteiligte) {
			if (bet.getTyp() != typ) {
				// Wenn der beteiligtenTyp zwischenzeitlich wechselt, den Bezeichner
				// ausgeben ...
				typ = bet.getTyp();
				final boolean istEinzigerSeinerArt = istEinzigerSeinerArt(bet, alleBeteiligte);
				final Numerus pl = istEinzigerSeinerArt(bet, beteiligte) ? Numerus.SINGULAR : Numerus.PLURAL;
				builder.append(Beteiligter.parteiBezeichner(typ, GenusNumerus.of(bet.getGenusNumerus().getGenus(), pl),
						bet.getLfdNr(), casus, istEinzigerSeinerArt));
			} else {
				// Laufende Nummer, bezogen auf den Typ berechnen...
				builder.append(fuegeLaufendeNummerHinzu(bet.getLfdNr()));
			}
			builder.append(", ");
		}
		behandleKomma(builder);
		return builder;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + ((beteiligte == null) ? 0 : beteiligte.hashCode());
		result = prime * result + (gesamtschuldnerisch ? 1231 : 1237);
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof MehrfachBeteiligter)) {
			return false;
		}
		MehrfachBeteiligter other = (MehrfachBeteiligter) obj;
		if (beteiligte == null) {
			if (other.beteiligte != null) {
				return false;
			}
		} else if (!getIds().equals(other.getIds())) {
			return false;
		}
		if (gesamtschuldnerisch != other.gesamtschuldnerisch) {
			return false;
		}
		return true;
	}

	/**
	 * Die Methode erstellt einen einfachen {@link Beteiligter}n oder einen
	 * {@link MehrfachBeteiligter}n, je nachdem, ob in der {@link List} der
	 * {@link Beteiligter}n einer {@link Beteiligter} enthalten ist (dann einfacher
	 * {@link Beteiligter} als Rückgabe) oder mehrere enthalten sind (dann ein
	 * {@link MehrfachBeteiligter}).
	 * 
	 * @param beteiligte          {@link List} der {@link Beteiligter}n
	 * @param gesamtschuldnerisch soll
	 *                            {@link MehrfachBeteiligter#isGesamtschuldnerisch()}
	 *                            gesetzt werden
	 * @return ein einfacher {@link Beteiligter}, wenn in der {@link List} der
	 *         {@link Beteiligter}n einer {@link Beteiligter} enthalten ist, oder
	 *         ein {@link MehrfachBeteiligter}, wenn mehrere enthalten sind
	 */
	public static Beteiligter from(List<Beteiligter> beteiligte, boolean gesamtschuldnerisch) {
		if (beteiligte == null || beteiligte.size() < 1)
			return null;
		if (beteiligte.size() == 1)
			return beteiligte.get(0);

		return new MehrfachBeteiligter(beteiligte, gesamtschuldnerisch);

	}
}
