/**
 * HauptklageForderung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.forderungen;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * Klasse für Hauptklageforderungen im Gegensatz zu Hilfsklageforderungen
 * 
 * @author Peter Felix Schuster (setrok)
 * @date: 31.07.2014
 *
 */
public class KlageForderung extends Forderung {
	private static final long serialVersionUID = 1087072460398990941L;
	private static final String BESCHREIBUNG = "Hauptklageforderung";
	private boolean wirtschaftlichIdentisch;

	/**
	 * Konstruktor ohne Paramter für {@link JAXB}
	 * 
	 */
	public KlageForderung() {
		super();
		super.setGegenforderung(false);
	}

	/**
	 * Der Standardkonstruktor belegt vor, dass es sich nicht um eine wirtschaftlich
	 * identische Klageforderung handelt.
	 * 
	 * @param wert   die Höhe der Forderung, also ihren Wert.
	 * @param erfolg inwieweit die Forderung Erfolg hatte.
	 * 
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	public KlageForderung(Euro wert, Euro erfolg) throws IllegalArgumentException {
		this(wert, erfolg, false);
	}

	/**
	 * Konstruktor:
	 * 
	 * @param wert                    die Höhe der Forderung, also ihren Wert.
	 * @param erfolg                  inwieweit die Forderung Erfolg hatte.
	 * @param wirtschaftlichIdentisch ob die Hauptklageforderung wirtschaftlich
	 *                                identisch zu der jeweils in der Liste
	 *                                vorangehenden Klageforderung ist.
	 * 
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	private KlageForderung(Euro wert, Euro erfolg, boolean wirtschaftlichIdentisch) throws IllegalArgumentException {
		super(false, wert, erfolg);
		setWirtschaftlichIdentisch(wirtschaftlichIdentisch);
	}


	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		
		builder.append(", ");
		builder.append("wirtschaftlichIdentisch=");
		builder.append(wirtschaftlichIdentisch);

		StringBuilder superBuilder = new StringBuilder(super.toString().replace("Forderung", "KlageForderung"));
		superBuilder.insert(superBuilder.lastIndexOf("]"), builder);
		return superBuilder.toString();
	}

	/**
	 * Gibt eine Beschreibung der Forderungsart zurück
	 * 
	 * @return "Hauptklageforderung"
	 * 
	 * @see eu.gronos.kostenrechner.data.forderungen.Forderung#getBeschreibung()
	 */
	@Override
	public String getBeschreibung() {
		return BESCHREIBUNG;
	}

	/**
	 * @return <code>boolean</code>, ob die Hauptklageforderung wirtschaftlich
	 *         identisch zu der jeweils in der Liste vorangehenden Klageforderung
	 *         ist. TODO derzeit an der Oberfläche nicht genutzt.
	 */
	@XmlAttribute(name = "p45i3GKG")
	public boolean isWirtschaftlichIdentisch() {
		return wirtschaftlichIdentisch;
	}

	/**
	 * @param wirtschaftlichIdentisch the wirtschaftlichIdentisch to set
	 */
	public void setWirtschaftlichIdentisch(boolean wirtschaftlichIdentisch) {
		this.wirtschaftlichIdentisch = wirtschaftlichIdentisch;
	}

}
