/*
 * GebuehrenVerzeichnis.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.util.ArrayList;

import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;

/**
 * Von {@link ArrayList} abgeleitete und aus {@link GebuehrenVerzeichnis}
 * hervorgegangene Klasse zur Übergabe von {@link GebuehrenTatbestand}en im
 * {@link TenorDatenContainer}, die auch Methoden zur Berechnung enthält,
 * nämlich die statische Methode
 * {@link #errechneGebuehr(GebuehrenTatbestand, long)} und für die Summe
 * {@link #errechneGebuehrenSumme(long)} sowie
 * {@link #errechneGebuehrenSummeMitSteuer(long)}.
 *
 * @author Peter Schuster (setrok)
 * @date 23.06.2019
 *
 */
public class GebuehrenAuflistung extends ArrayList<GebuehrenTatbestand> {

	private static final long serialVersionUID = 6564487211842926610L;

	public GebuehrenAuflistung() {
		super();
	}

	/**
	 * Die Methode dient dazu, die passende Gebühr oder Auslage zu berechnen.
	 * 
	 * @param gt         einen GebuehrenSatzTatbestand oder AuslagenTatbestand,
	 *                   andere Ableger von GebuehrenTatbestand werden ignoriert.
	 * @param streitwert Streitwert (wird bei Auslagen ignoriert)
	 * @return die Höhe der Gebühr (dann errechnet nach 1,0 Gebühr für den
	 *         streitwert, multipliziert mit dem Gebührensatz nach gt) bzw. Auslage
	 */
	public static Euro errechneGebuehr(GebuehrenTatbestand gt, Euro streitwert) {
		if (gt != null && gt instanceof AuslagenTatbestand)
			return ((AuslagenTatbestand) gt).getBetrag();
		else if (gt != null && gt instanceof PauschalTatbestand)
			return ((PauschalTatbestand) gt).getBetrag();
		else if (gt != null && gt instanceof GebuehrenSatzTatbestand)
			return GebuehrenAuflistung.findeGebuehrenTabelle(gt).errechneGebuehr(streitwert,
					((GebuehrenSatzTatbestand) gt).getSatz());
		else
			return Euro.ofCents(0L);
	}

	/**
	 * Die Methode soll die Summe aller Gebühren liefern. Mehrwertsteuer wird
	 * ignoriert.
	 * 
	 * @param streitwert ein long mit dem Streitwert
	 * @return ein double mit der Summe der Gebühren für diesen Streitwert
	 */
	public Euro errechneGebuehrenSumme(Euro streitwert) {
		Euro summe = Euro.ofCents(0L);// double = 0
		for (GebuehrenTatbestand gt : this) {
			/* summe um die Gebühr erhöhen */
			// summe += GebuehrenAuflistung.errechneGebuehr(gt, streitwert);
			summe = summe.add(GebuehrenAuflistung.errechneGebuehr(gt, streitwert));
		}
		return summe;
	}

	/**
	 * Die Methode soll die Summe aller Gebühren der Tabelle liefern und auch die
	 * Umsatzsteuer ausrechnen, wenn sie im {@link GebuehrenAuflistung} vorkommen.
	 * 
	 * @param streitwert ein long mit dem Streitwert
	 * @return ein double mit der Summe der Gebühren für diesen Streitwert
	 */
	public Euro errechneGebuehrenSummeMitSteuer(Euro streitwert) {
		Euro summe = Euro.ofCents(0L);// double summe = 0;
		Euro rvgsumme = Euro.ofCents(0L);// double rvgsumme = 0;
		for (GebuehrenTatbestand gt : this) {
			/*
			 * errechneGebuehr ignoriert MehrwertsteuerTatbestände; diese muss deshalb extra
			 * gerechnet werden.
			 */
			if (gt instanceof MehrwertsteuerTatbestand) {
				MehrwertsteuerTatbestand mwst = (MehrwertsteuerTatbestand) gt;
				summe = summe.add(mwst.errechneSteuer(rvgsumme));
				/*
				 * Wenn die Gebühren isoliert werden, die RVG-Summe wieder auf 0 setzen.
				 */
				if (mwst.isIsoliert())
					rvgsumme = Euro.ofCents(0L);//0.0;
			} else {
				/* summe um die Gebühr erhöhen */
				summe = summe.add(GebuehrenAuflistung.errechneGebuehr(gt, streitwert));
				/*
				 * rvgsumme nur erhöhen, wenn RVG-Gebühr (wird für MwSt gebraucht).
				 */
				if (gt.getGebuehrenKlasse() == AnwaltsGebuehrenTabelle.class)
					rvgsumme = rvgsumme.add(GebuehrenAuflistung.errechneGebuehr(gt, streitwert));
			}
		}
		return summe;
	}

	/**
	 * Die Methode findeGebuehrenTabelle findet heraus, ob ein Gebührentatbestand
	 * zum RVG oder GKG gehört und gibt eine Instanz dieser Tabelle zurück.
	 * 
	 * @param gt der GebuehrenTatbestand
	 * @return die zum GebuehrenTatbestand gehörende GebuehrenTabelle
	 */
	private static GebuehrenTabelle findeGebuehrenTabelle(GebuehrenTatbestand gt) {
		if (AnwaltsGebuehrenTabelle.class.equals(gt.getGebuehrenKlasse()))
			return GebuehrenGesetzesSammlung.getRvg();
		else if (GerichtsGebuehrenTabelle.class.equals(gt.getGebuehrenKlasse()))
			return GebuehrenGesetzesSammlung.getGkg();
		return null;
	}

}