/**
 * GebuehrenGrundZeile.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.util.GregorianCalendar;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlTransient;

import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * Eine Klasse, die die Sockelwerte einer Gebührenordnung speichert, die sich
 * nach Streit- bzw. Gegenstandstandswert richten. Maßgeblich ist die
 * Wertgebührenvorschrift der jeweiligen Gebührenordnung (z.B. § 13 RVG, § 34
 * GKG).
 * 
 * Gebührenordnungen sind so aufgebaut, dass die 1,0-Gebühr bis zum einem
 * bestimmten untersten Gegenstands-/Streitwert ( {@link GebuehrenGrundZeile}.
 * <code>sprung</code>) einen bestimmten Sockelbetrag (
 * {@link GebuehrenGrundZeile}.<code>hoehe</code>) beträgt. Das ist die erste
 * Zeile (diese Klasse).
 * 
 * Diese Gebühr erhöht sich dann Gegenstandswert bis <code>grenze</code> Euro,
 * für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
 * <code>hoehe</code> Euro. Das sind die weiteren Zeilen ({@link GebuehrenZeile}
 * ).
 *
 * @author Peter Schuster (setrok)
 * @date 01.01.2015
 *
 */
public class GebuehrenGrundZeile extends GebuehrenZeile {

	private static final long serialVersionUID = 8970361335078930471L;
	private String gesetz = null;
	private GregorianCalendar gueltigAb = null;
	private GregorianCalendar gueltigBis = null;

	/**
	 * Der Konstruktor setzt alle notwendigen Variablen.
	 * 
	 * Beispiel nach GKG (§ 34): Wenn sich die Gebühren nach dem Streitwert
	 * richten, beträgt die Gebühr bei einem Streitwert bis
	 * <code>grundSprung</code> EUR <code>grundGebuehr</code> EUR.
	 * 
	 * @param grundSprung
	 *            der Streitwert, bis zu dem die Grundgebühr gilt.
	 * @param grundGebuehr
	 *            die Gebühr, auf die die Erhöhungen iterativ aufgerechnet
	 *            werden müssen (s. Beispiel oben).
	 * @param gueltigAb
	 *            Gültigkeitszeitraum des jeweils einschlägigen Gesetzes
	 *            (Beginn)
	 * @param gueltigBis
	 *            Gültigkeitszeitraum des jeweils einschlägigen Gesetzes (Ende)
	 * @param gesetz
	 *            die Bezeichnung der Gebührenordnung als {@link String}
	 */
	public GebuehrenGrundZeile(Euro grundSprung, Euro grundHoehe,
			GregorianCalendar gueltigAb, GregorianCalendar gueltigBis,
			String gesetz) {
		//		super(0L, grundSprung, grundHoehe);
		super(Euro.ofCents(0L), grundSprung, grundHoehe);
		this.gueltigAb = gueltigAb;
		this.gueltigBis = gueltigBis;
		this.gesetz = gesetz;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenGrundZeile() {
		super();
		setGrenze(Euro.ofCents(0L));
	}

	/**
	 * @return immer 0L
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenZeile#getGrenze()
	 */
	@Override
	@XmlTransient
	public Euro getGrenze() {
		return Euro.ofCents(0L);
	}

	/**
	 * 
	 * @param grenze
	 *            wird ignoriert. Wird immer auf 0L gesetzt.
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenZeile#setGrenze(long)
	 */
	@Override
	public void setGrenze(Euro grenze) {
		super.setGrenze(Euro.ofCents(0L));
	}

	/**
	 * @return gibt {@link #gesetz} als String zurück.
	 */
	@XmlAttribute(name = "gebuehrenordnung")
	public String getGesetz() {
		return gesetz;
	}

	/**
	 * @param gesetz
	 *            d. {@link #gesetz}, d. gesetzt werden soll als String
	 */
	public void setGesetz(String gesetz) {
		this.gesetz = gesetz;
	}

	/**
	 * @return gibt {@link #gueltigAb} als GregorianCalendar zurück.
	 */
	@XmlAttribute(name = "gueltigAb")
	public GregorianCalendar getGueltigAb() {
		return gueltigAb;
	}

	/**
	 * @param gueltigAb
	 *            d. {@link #gueltigAb}, d. gesetzt werden soll als
	 *            GregorianCalendar
	 */
	public void setGueltigAb(GregorianCalendar gueltigAb) {
		this.gueltigAb = gueltigAb;
	}

	/**
	 * @return gibt {@link #gueltigBis} als GregorianCalendar zurück.
	 */
	@XmlAttribute(name = "gueltigBis")
	public GregorianCalendar getGueltigBis() {
		return gueltigBis;
	}

	/**
	 * @param gueltigBis
	 *            d. {@link #gueltigBis}, d. gesetzt werden soll als
	 *            GregorianCalendar
	 */
	public void setGueltigBis(GregorianCalendar gueltigBis) {
		this.gueltigBis = gueltigBis;
	}

}
