/*
 * GebuehrenRahmenTatbestand.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

/**
 * Eine Klasse, die Wertgebühren mit Rahmen abbilden soll.
 *
 * @author Peter Schuster (setrok)
 * @date 20 Jun 2019
 *
 */
public class GebuehrenRahmenTatbestand extends GebuehrenSatzTatbestand implements Cloneable {

	public static final String ART = "Gebührenrahmen";
	private static final long serialVersionUID = 1682428886423302356L;
	private double untergrenze;
	private double obergrenze;
	private double mittel;

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenRahmenTatbestand() {
		super();
	}

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten.
	 * 
	 * Die Mittelgebühr wird aus dem Mittelwert von {@link #getUntergrenze()} und
	 * {@link #getObergrenze()} ermittelt, der {@link #getSatz()} auch.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param untergrenze     {@link #getUntergrenze()}
	 * @param obergrenze      {@link #getObergrenze()}
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 */
	public GebuehrenRahmenTatbestand(String vorschrift, String bezeichnung, double untergrenze, double obergrenze,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		this(vorschrift, bezeichnung, untergrenze, obergrenze, mittle(untergrenze, obergrenze), gebuehrenKlasse);
	}

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten. Der
	 * {@link #getSatz()} wird auf {@link #getMittel()} gesetzt.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param untergrenze     {@link #getUntergrenze()}
	 * @param obergrenze      {@link #getObergrenze()}
	 * @param mittel          {@link #getMittel()}
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 */
	public GebuehrenRahmenTatbestand(String vorschrift, String bezeichnung, double untergrenze, double obergrenze,
			double mittel, Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, mittel, gebuehrenKlasse);
		setUntergrenze(untergrenze);
		setObergrenze(obergrenze);
		setMittel(mittel);
	}

	/**
	 * @return gibt {@link #untergrenze} als {@link double} zurück.
	 */
	@XmlAttribute(name = "untergrenze")
	public double getUntergrenze() {
		return untergrenze;
	}

	/**
	 * @param untergrenze d. {@link #untergrenze}, d. gesetzt werden soll als
	 *                    {@link double}.
	 */
	public void setUntergrenze(double untergrenze) {
		this.untergrenze = untergrenze;
	}

	/**
	 * @return gibt {@link #obergrenze} als {@link double} zurück.
	 */
	@XmlAttribute(name = "obergrenze")
	public double getObergrenze() {
		return obergrenze;
	}

	/**
	 * @param obergrenze d. {@link #obergrenze}, d. gesetzt werden soll als
	 *                   {@link double}.
	 */
	public void setObergrenze(double obergrenze) {
		this.obergrenze = obergrenze;
	}

	/**
	 * In bestimmten Gebührentatbeständen liegt die Mittelgebühr nicht beim
	 * rechnerischen Mittelwert, sondern ist gesetzlich festgelegt, etwa bei Ziff.
	 * 2300 VV RVG: Eine Gebühr von mehr als 1,3 kann nur gefordert werden, wenn die
	 * Tätigkeit umfangreich oder schwierig war.
	 * 
	 * @return gibt die Mittelgebühr {@link #mittel} als {@link double} zurück.
	 */
	@XmlAttribute(name = "mittelGebuehr")
	public double getMittel() {
		return mittel;
	}

	/**
	 * @param mittel d. {@link #mittel}, d. gesetzt werden soll als {@link double}.
	 */
	public void setMittel(double mittel) {
		this.mittel = mittel;
	}

	@Override
	public String toString() {
		return String.format(
				"GebuehrenRahmenTatbestand [vorschrift=%s, bezeichnung=%s, untergrenze=%s, obergrenze=%s, mittel=%s, gebuehrenKlasse=%s]",
				getVorschrift(), getBezeichnung(), untergrenze, obergrenze, mittel, getGebuehrenKlasse());
	}

	/**
	 * Die Methode dient dazu, einen {@link Object#hashCode()} zu erstellen.
	 * 
	 * @return einen <code>int</code>, der sich aus
	 *         {@link GebuehrenSatzTatbestand#hashCode()} sowie
	 *         {@link #getUntergrenze()}, {@link #getObergrenze()} und
	 *         {@link #getMittel()} berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		long temp;
		temp = Double.doubleToLongBits(mittel);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(obergrenze);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(untergrenze);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * Die Methode vergleicht anhand von {@link #getMittel()},
	 * {@link #getUntergrenze()} und {@link #getObergrenze()} sowie
	 * {@link GebuehrenSatzTatbestand#equals(Object)} die Gleichheit
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof GebuehrenRahmenTatbestand)) {
			return false;
		}
		GebuehrenRahmenTatbestand other = (GebuehrenRahmenTatbestand) obj;
		if (Double.doubleToLongBits(mittel) != Double.doubleToLongBits(other.mittel)) {
			return false;
		}
		if (Double.doubleToLongBits(obergrenze) != Double.doubleToLongBits(other.obergrenze)) {
			return false;
		}
		if (Double.doubleToLongBits(untergrenze) != Double.doubleToLongBits(other.untergrenze)) {
			return false;
		}
		return true;
	}

	@Override
	public GebuehrenRahmenTatbestand clone() {
		return new GebuehrenRahmenTatbestand(getVorschrift(), getBezeichnung(), getUntergrenze(), getObergrenze(),
				getMittel(), getGebuehrenKlasse());
	}

	/**
	 * Die Methode dient dazu, TODO (kommentieren)
	 * 
	 * @param untergrenze
	 * @param obergrenze
	 * @return TODO (kommentieren)
	 */
	private static double mittle(double untergrenze, double obergrenze) {
		return (untergrenze + obergrenze) / 2;
	}

}
