/**
 * Beteiligter.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.tenordaten;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlID;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.interfaces.Hinzufuegbar;

/**
 * Die Oberklasse für Beteiligte hält die Methoden für die parteiBezeichner
 * bereit
 * 
 * @author Peter Schuster (setrok)
 * @date 12.06.2014
 * 
 * @todo TODO könnte Xjustiz-Daten speichern...
 */
public class Beteiligter implements Comparable<Beteiligter>, Serializable, Hinzufuegbar {

	private static final long serialVersionUID = -6275833726367356470L;
	private static int zaehler = Integer.MIN_VALUE;

	/**
	 * ein {@link Enum} für die grammatikalische Beugung der Bezeichnungen, also
	 * {@link #MAENNLICH_SINGULAR}, {@link #MAENNLICH_PLURAL},
	 * {@link #WEIBLICH_SINGULAR}, {@link #WEIBLICH_PLURAL}.
	 * 
	 * Mit der Methode {@link #of(Genus, Numerus)} lässt sich die passende Form
	 * anhand von {@link Genus} und {@link Number} finden.
	 */
	public static enum GenusNumerus {
		MAENNLICH_SINGULAR(Genus.MAENNLICH, Numerus.SINGULAR), // 0
		WEIBLICH_SINGULAR(Genus.WEIBLICH, Numerus.SINGULAR), // 1
		MAENNLICH_PLURAL(Genus.MAENNLICH, Numerus.PLURAL), // 2),
		WEIBLICH_PLURAL(Genus.WEIBLICH, Numerus.PLURAL);// 3;

		public static enum Genus {
			MAENNLICH, WEIBLICH;
		}

		public static enum Numerus {
			SINGULAR, PLURAL;
		}

		private final Genus genus;
		private final Numerus numerus;

		private GenusNumerus(Genus genus, Numerus numerus) {
			this.genus = genus;
			this.numerus = numerus;
		}

		/**
		 * Sozusagen eine Fabrikmethode
		 * 
		 * @param genus   ein {@link Genus}
		 * @param numerus ein {@link Numerus}
		 * @return den dazu passenden {@link GenusNumerus}
		 */
		public static GenusNumerus of(Genus genus, Numerus numerus) {
			switch (genus) {
			case MAENNLICH:
				switch (numerus) {
				case SINGULAR:
					return MAENNLICH_SINGULAR;
				case PLURAL:
					return MAENNLICH_PLURAL;
				}
			case WEIBLICH:
				switch (numerus) {
				case SINGULAR:
					return WEIBLICH_SINGULAR;
				case PLURAL:
					return WEIBLICH_PLURAL;
				}
			}
			return null;
		}

		/**
		 * @return the genus
		 */
		public Genus getGenus() {
			return genus;
		}

		/**
		 * @return the numerus
		 */
		public Numerus getNumerus() {
			return numerus;
		}

		/**
		 * @param ordinal die Ordnungszahl des enums, was {@link #ordinal()} zurückgeben
		 *                würde
		 * @return der passende {@link GenusNumerus}
		 */
		public static GenusNumerus of(int ordinal) {
			if (ordinal < values().length)
				return values()[ordinal];
			else
				return null;
		}
	}

	/**
	 * ein {@link Enum} für grammatikalische Beugungen, also {@link #NOMINATIV},
	 * {@link #GENITIV}, {@link #DATIV}, {@link #AKKUSATIV}.
	 */
	public static enum Casus {
		NOMINATIV, // 0
		GENITIV, // 1
		DATIV, // 2
		AKKUSATIV;// 3
	}

	/**
	 * ein {@link Enum} für {@link Beteiligter#getTyp()}, kann die Werte
	 * {@link #KLAEGER}, {@link #DRITTWIDERBEKLAGTE} oder {@link #BEKLAGTE}
	 * annehmen.
	 */
	public static enum BeteiligtenTyp {
		KLAEGER('k', // 0
				new String[][] { { "der Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" },
						{ "des Klägers", "der Klägerin", "der Kläger", "der Klägerinnen" },
						{ "dem Kläger", "der Klägerin", "den Klägern", "den Klägerinnen" }, {
								"den Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" } }),
		DRITTWIDERBEKLAGTE('d', // 1
				new String[][] {
						{ "der Drittwiderbeklagte", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
								"die Drittwiderbeklagten" },
						{ "des Drittwiderbeklagten", "der Drittwiderbeklagten", "der Drittwiderbeklagten",
								"der Drittwiderbeklagten" },
						{ "dem Drittwiderbeklagte", "der Drittwiderbeklagten", "den Drittwiderbeklagten",
								"den Drittwiderbeklagten" },
						{ "den Drittwiderbeklagten", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
								"die Drittwiderbeklagten" } }),
		BEKLAGTE('b', // 2
				new String[][] { { "der Beklagte", "die Beklagte", "die Beklagten", "die Beklagten" },
						{ "des Beklagten", "der Beklagten", "der Beklagten", "der Beklagten" },
						{ "dem Beklagten", "der Beklagten", "den Beklagten", "den Beklagten" },
						{ "den Beklagten", "die Beklagte", "die Beklagten", "die Beklagten" } }),
		GERICHTSKOSTEN('g', // = 3, war 8
				new String[][] { { "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" },
						{ "des Gerichts", "des Gerichts", "der Gerichte", "der Gerichte" },
						{ "dem Gericht", "dem Gericht", "den Gerichten", "den Gerichten" },
						{ "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" } });

		private final String[][] beugung;
		private final char kurz;

		private BeteiligtenTyp(char kurz, String[][] beugung) {
			this.beugung = beugung;
			this.kurz = kurz;
		}

		/**
		 * @return die entsprechenden Beugeformen
		 */
		public String[][] getBeugung() {
			return beugung;
		}

		/**
		 * 
		 * Die Methode erzeugt eine gebeugte Form des {@link BeteiligtenTyp}s
		 * 
		 * @param genusNumerus als {@link GenusNumerus}
		 * @param casus        als {@link Casus}
		 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
		 *         Kläger" - natürlich ohne eine laufende Nummer
		 */
		public String beuge(GenusNumerus genusNumerus, Casus casus) {
			return getBeugung()[casus.ordinal()][genusNumerus.ordinal()];
		}

		/**
		 * @param typ der {@link BeteiligtenTyp#ordinal()}
		 * @return den dazu passenden {@link BeteiligtenTyp}
		 */
		public static BeteiligtenTyp of(int typ) {
			if (typ == 8)// früher war das der
				return GERICHTSKOSTEN;
			if (typ < values().length)
				return values()[typ];
			else
				return null;
		}

		/**
		 * @param kurz die Kurzform {@link #kurz} des {@link BeteiligtenTyp}s
		 * @return den dazu passenden {@link BeteiligtenTyp}
		 */
		public static BeteiligtenTyp of(char kurz) {
			for (BeteiligtenTyp typ : values()) {
				if (typ.kurz() == kurz)
					return typ;
			}
			return null;
		}

		/**
		 * @return the kurz
		 */
		public char kurz() {
			return kurz;
		}
	}

	// public static final int SINGULAR = 0;
//	public static final GenusNumerus.Numerus SINGULAR = GenusNumerus.Numerus.SINGULAR;
	// public static final int PLURAL = 2;
//	public static final GenusNumerus.Numerus PLURAL = GenusNumerus.Numerus.PLURAL;
	// public static final int MAENNLICH = 0;
//	public static final GenusNumerus.Genus MAENNLICH = GenusNumerus.Genus.MAENNLICH;
	// public static final int WEIBLICH = 1;
//	public static final GenusNumerus.Genus WEIBLICH = GenusNumerus.Genus.WEIBLICH;
	// public static final int NOMINATIV = 0;
//	public static final Casus NOMINATIV = Casus.NOMINATIV;
	// public static final int GENITIV = 1;
//	public static final Casus GENITIV = Casus.GENITIV;
	// public static final int DATIV = 2;
//	public static final Casus DATIV = Casus.DATIV;
	// public static final int AKKUSATIV = 3;
//	public static final Casus AKKUSATIV = Casus.AKKUSATIV;
	// public static final int KLAEGER = 0;
//	public static final BeteiligtenTyp KLAEGER = BeteiligtenTyp.KLAEGER;
	// public static final int DRITTWIDERBEKLAGTE = 1;
//	public static final BeteiligtenTyp DRITTWIDERBEKLAGTE = BeteiligtenTyp.DRITTWIDERBEKLAGTE;
	// public static final int BEKLAGTE = 2;
//	public static final BeteiligtenTyp BEKLAGTE = BeteiligtenTyp.BEKLAGTE;
	// public static final int GERICHTSKOSTEN = 8;
//	public static final BeteiligtenTyp GERICHTSKOSTEN = BeteiligtenTyp.GERICHTSKOSTEN;

//	private static final String[][] BEKLAGTEN_BEUGUNG = new String[][] {
//			{ "der Beklagte", "die Beklagte", "die Beklagten", "die Beklagten" },
//			{ "des Beklagten", "der Beklagten", "der Beklagten", "der Beklagten" },
//			{ "dem Beklagten", "der Beklagten", "den Beklagten", "den Beklagten" },
//			{ "den Beklagten", "die Beklagte", "die Beklagten", "die Beklagten" } };
//	private static final String[][] DRITTWIDERBEKLAGTEN_BEUGUNG = new String[][] {
//			{ "der Drittwiderbeklagte", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
//					"die Drittwiderbeklagten" },
//			{ "des Drittwiderbeklagten", "der Drittwiderbeklagten", "der Drittwiderbeklagten",
//					"der Drittwiderbeklagten" },
//			{ "dem Drittwiderbeklagte", "der Drittwiderbeklagten", "den Drittwiderbeklagten",
//					"den Drittwiderbeklagten" },
//			{ "den Drittwiderbeklagten", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
//					"die Drittwiderbeklagten" } };
//	private static final String[][] KLAEGER_BEUGUNG = new String[][] {
//			{ "der Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" },
//			{ "des Klägers", "der Klägerin", "der Kläger", "der Klägerinnen" },
//			{ "dem Kläger", "der Klägerin", "den Klägern", "den Klägerinnen" },
//			{ "den Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" } };
//	private static final String[][] GERICHTSKOSTEN_BEUGUNG = new String[][] {
//			{ "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" },
//			{ "des Gerichts", "des Gerichts", "der Gerichte", "der Gerichte" },
//			{ "dem Gericht", "dem Gericht", "den Gerichten", "den Gerichten" },
//			{ "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" } };
	/**
	 * Der int mit dem Namen genusNumerus speichert das grammatikalische Geschlecht
	 * des Beteiligten sowie, ob es einer oder mehrere sind. Sein Wert entspricht
	 * einer der Konstanten MAENNLICH oder WEIBLICH, gegebenenfalls zuzüglich
	 * PLURAL.
	 */
	private GenusNumerus genusNumerus;
	/**
	 * Der int mit dem Namen typ speichert den Beteiligtentyp des Beteiligten,
	 * entspricht also einer der Konstanten Beteiligter.KLAEGER,
	 * Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 */
	private BeteiligtenTyp typ;
	/**
	 * Der int mit dem Namen lfdNr cachet die lfdNr, so dass sie für Hash-Vergleiche
	 * benutzt werden kann.
	 */
	private int lfdNr = -1;// ++zaehler;//

	private int tempNr;

	/*
	 * Der Konstruktor richtet eine Instanz von Beteiligter mit den übergegebenen
	 * Werten ein.
	 * 
	 * @param typ eine der Konstanten Beteiligter.KLAEGER,
	 * Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 * 
	 * @param genusNumerus eine der Konstanten MAENNLICH oder WEIBLICH
	 */
//	@Deprecated
//	public Beteiligter(int typ, int genusNumerus) {
//		super();
//		setTyp(typ);
//		setGenusNumerus(genusNumerus);
//	}

	/**
	 * Der Konstruktor richtet eine Instanz von Beteiligter mit den übergegebenen
	 * Werten ein.
	 * 
	 * @param typ          ein {@link BeteiligtenTyp}, also
	 *                     {@link BeteiligtenTyp#KLAEGER},
	 *                     {@link BeteiligtenTyp#DRITTWIDERBEKLAGTE},
	 *                     {@link BeteiligtenTyp#BEKLAGTE}
	 * @param genusNumerus ein {@link GenusNumerus}
	 */
	public Beteiligter(BeteiligtenTyp beteiligtenTyp, GenusNumerus genusNumerus) {
		this();
		setTyp(beteiligtenTyp);
		setGenusNumerus(genusNumerus);
	}

	/**
	 * Dieser Konstruktor richtet eine Instanz von Beteiligter mit denselben Werten
	 * wie der übergegebene Beteiligte ein (soll die Arbeit bei abgeleiteten Klassen
	 * und mit CellEditor erleichtern).
	 * 
	 * @param beteiligter ein Beteiligter.
	 */
	public Beteiligter(Beteiligter beteiligter) {
		this();
		setTyp(beteiligter.getTyp());
		setGenusNumerus(beteiligter.getGenusNumerus());
	}

	/**
	 * Konstruktor für JAXB
	 * 
	 */
	public Beteiligter() {
		super();
		tempNr = zaehler++;
		if (tempNr >= 0) {
			tempNr = zaehler = Integer.MIN_VALUE;
		}
	};

	/*
	 * Die statische Methode parteiBezeichner dient dazu, einen passenden Bezeichner
	 * zu den übergebenen Angaben zu einer bestimmten Partei zu erzeugen. Diese kann
	 * dann von den nicht-statischen Methoden aufgerufen werden, aber auch denen zur
	 * Array-Erzeugung.
	 * 
	 * @param beteiligtenTyp eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER,
	 * BEKLAGTER
	 * 
	 * @param genusNumerus
	 * 
	 * @param laufendeNummer die laufende Nummer der Partei bzgl. ihres Typs
	 * 
	 * @param casus einen passenden Bezeichner zu einer bestimmten Partei zu
	 * erzeugen.
	 * 
	 * @param einzigerSeinerArt true, wenn es nur eine Partei dieses Beteiligtentyps
	 * gibt.
	 * 
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 * Kläger", "der Beklagte zu 1.)" oder null, wenn es den Beteiligtentyp nicht
	 * gibt.
	 */
//	@Deprecated
//	public static String parteiBezeichner(int beteiligtenTyp, int genusNumerus, int laufendeNummer, int casus,
//			boolean einzigerSeinerArt) {
//		String[][] zwischen = getBezeichnerBeugungen(beteiligtenTyp);
//		if (zwischen == null)
//			return null;
//		return zwischen[casus][genusNumerus] + (einzigerSeinerArt ? "" : fuegeLaufendeNummerHinzu(laufendeNummer));
//	}

	/**
	 * Die statische Methode parteiBezeichner dient dazu, einen passenden Bezeichner
	 * zu den übergebenen Angaben zu einer bestimmten Partei zu erzeugen. Diese kann
	 * dann von den nicht-statischen Methoden aufgerufen werden, aber auch denen zur
	 * Array-Erzeugung.
	 * 
	 * @param beteiligtenTyp    eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER,
	 *                          BEKLAGTER
	 * @param genusNumerus
	 * @param laufendeNummer    die laufende Nummer der Partei bzgl. ihres Typs
	 * @param casus             einen passenden Bezeichner zu einer bestimmten
	 *                          Partei zu erzeugen.
	 * @param einzigerSeinerArt true, wenn es nur eine Partei dieses Beteiligtentyps
	 *                          gibt.
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)" oder null, wenn es den Beteiligtentyp
	 *         nicht gibt.
	 */
	public static String parteiBezeichner(BeteiligtenTyp beteiligtenTyp, GenusNumerus genusNumerus, int laufendeNummer,
			Casus casus, boolean einzigerSeinerArt) {
//		String[][] zwischen = beteiligtenTyp.getBeugung();
		return beteiligtenTyp.beuge(genusNumerus, casus) // war: zwischen[casus.ordinal()][genusNumerus.ordinal()]
				+ (einzigerSeinerArt ? "" : fuegeLaufendeNummerHinzu(laufendeNummer));
	}

	/**
	 * @return typ gibt den typ als int zurück, also eine der Konstanten
	 *         Beteiligter.KLAEGER, Beteiligter.DRITTWIDERBEKLAGTE,
	 *         Beteiligter.BEKLAGTE
	 */
	@XmlAttribute(name = "typ")
	@XmlJavaTypeAdapter(BeteiligtenTypAdapter.class)
	public BeteiligtenTyp getTyp() {
		return typ;
	}

	/*
	 * @return typ gibt den typ als int zurück, also {@link
	 * BeteiligtenTyp#ordinal()}
	 */
//	@Deprecated
//	public int beteiligtenTypAsInt() {
//		return typ.ordinal();
//	}

	/*
	 * Die Methode dient dazu, den Typ nachträglich zu verändern.
	 * 
	 * @param typ eine der Konstanten {@link Beteiligter#KLAEGER}, {@link
	 * Beteiligter#DRITTWIDERBEKLAGTE}, {@link Beteiligter#BEKLAGTE}
	 */
//	@Deprecated
//	public void setTyp(int typ) {
//		if (typ >= KLAEGER && typ <= BEKLAGTE)
//			this.typ = typ;
//		if (typ == GERICHTSKOSTEN && this instanceof GerichtsKostenBeteiligter)
//			this.typ = typ;
//		if (typ >= BeteiligtenTyp.GERICHTSKOSTEN.ordinal() && !(this instanceof GerichtsKostenBeteiligter))
//			return;// do nothing
//		setTyp(BeteiligtenTyp.values()[typ]);
//	}

	/**
	 * Die Methode dient dazu, den Typ nachträglich zu verändern.
	 * 
	 * @param typ ein {@link BeteiligtenTyp}
	 */
	public void setTyp(BeteiligtenTyp typ) {
		this.typ = typ;
	}

	/**
	 * @return den genusNumerus als {@link GenusNumerus}
	 */
	@XmlAttribute(name = "genusNumerus")
	@XmlJavaTypeAdapter(GenusNumerusAdapter.class)
	public GenusNumerus getGenusNumerus() {
		return genusNumerus;
	}

	/*
	 * @return den genusNumerus als int, also {@link GenusNumerus#ordinal()}
	 */
//	@Deprecated
//	public int genusNumerusAsInt() {
//		return genusNumerus.ordinal();
//	}
//
//	@Deprecated
//	public void setGenusNumerus(int genusNumerus) {
//		setGenusNumerus(GenusNumerus.values()[genusNumerus]);
//	}

	/**
	 * @param genusNumerus the genusNumerus to set
	 */
	public void setGenusNumerus(GenusNumerus genusNumerus) {
		this.genusNumerus = genusNumerus;
	}

	/**
	 * Die Methode gibt zurück, ob die Instanz ein oder mehrere "echte" Personen
	 * repräsentiert.
	 * 
	 * @return true, wenn genusNumerus eine Pluralform ist.
	 */
	public boolean isPlural() {
		if (this.genusNumerus == null)
			return false;
		return this.genusNumerus.getNumerus() == GenusNumerus.Numerus.PLURAL;
	}

	/**
	 * @return gibt den lfdNr als int zurück.
	 */
	@XmlAttribute(name = "laufendeNummer")
	public int getLfdNr() {
		return lfdNr;
	}

	/**
	 * Der Setter für lfdNr dient dazu, die lfdNr zu cashen, so dass sie für
	 * Hash-Vergleiche benutzt werden kann.
	 * 
	 * @param lfdNr d. lfdNr, d. gesetzt werden soll
	 */
	public void setLfdNr(int lfdNr) {
		this.lfdNr = lfdNr;
	}

	/**
	 * @return the tempNr
	 */
	protected int getTempNr() {
		return tempNr;
	}

	/**
	 * @return {@link BeteiligtenTyp#kurz} + {@link #getLfdNr()}, zB "k1"
	 */
	@XmlAttribute(name = "id")
	@XmlID
	public String getId() {
		if (this.typ == null)// ||
			return null;

		if (lfdNr > 0)
			return "" + typ.kurz() + lfdNr;
		else
			return "" + typ.kurz() + tempNr;
	}

	/**
	 * @param id
	 */
	public void setId(String id) {
		if (id == null || id.length() < 2)
			return;
		// das erste Zeichen ist immer der Typ
		setTyp(BeteiligtenTyp.of(id.charAt(0)));
		// der Rest ist eine Laufende Nummer
		final int nr = Integer.parseInt(id.substring(1));
		if (nr > 0)
			setLfdNr(nr);
	}

	/**
	 * Die Methode parteiBezeichner dient dazu, einen passenden Bezeichner zu dem
	 * repräsentierten Beteiligten zu erzeugen.
	 * 
	 * @param casus             eine der Konstanten NOMINATIV, GENITIV, DATIV,
	 *                          AKKUSATIV
	 * @param laufendeNummer    die laufende Nummer der Partei bzgl. ihres Typs
	 * @param einzigerSeinerArt true, wenn es nur eine Partei dieses Beteiligtentyps
	 *                          gibt.
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)".
	 */
	public String parteiBezeichner(Casus casus, int laufendeNummer, boolean einzigerSeinerArt) {
		setLfdNr(laufendeNummer);
//		return parteiBezeichner(beteiligtenTypAsInt(), genusNumerusAsInt(), laufendeNummer, casus, einzigerSeinerArt);
		return parteiBezeichner(getTyp(), getGenusNumerus(), laufendeNummer, casus, einzigerSeinerArt);
	}

//	@Deprecated
//	public String parteiBezeichner(int casus, int laufendeNummer, boolean einzigerSeinerArt) {
//		return parteiBezeichner(Casus.values()[casus], laufendeNummer, einzigerSeinerArt);
//	}

	/**
	 * Die Methode fuegeLaufendeNummerHinzu dient dazu, aus der übergebenen
	 * laufenden Nummer einen String zu machen, der einem "nackten" ParteiBezeichner
	 * hinzugefügt werden kann.
	 * 
	 * @param laufendeNummer die laufende Nummer eines Beteiligten als int
	 * @return einen formatierten String, z.B. " zu 1.)" (führendes Leerzeichen)
	 */
	public static String fuegeLaufendeNummerHinzu(int laufendeNummer) {
		return String.format(" zu %d.)", laufendeNummer);
	}

//	@Deprecated
//	public static Beteiligter[] getBeteiligtenAuswahlListe(int beteiligtenTyp, boolean mitPlural) {
//		return getBeteiligtenAuswahlListe(BeteiligtenTyp.values()[beteiligtenTyp], mitPlural);
//	}

	/**
	 * Die Methode getBeteiligtenAuswahlListe dient dazu, ein Array von
	 * Beteiligter[] zu haben, um eine JComboBox<Beteiligter> füllen zu können.
	 * 
	 * @param beteiligtenTyp eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER,
	 *                       BEKLAGTER
	 * @param mitPlural      der Schalter regelt, ob auch die Pluralformen
	 *                       ausgegeben werden sollen; wenn "true", werden insgesamt
	 *                       4 Strings ins String[] gepackt, sonst nur die zwei
	 *                       Singularformen.
	 * @return ein Array Beteiligter[4] mit Beteiligten eines
	 *         <code>beteiligtenTyp</code>s in allen Formen männlich/weiblich,
	 *         ggfls. Singular/Plural.
	 */
	public static Beteiligter[] getBeteiligtenAuswahlListe(BeteiligtenTyp beteiligtenTyp, boolean mitPlural) {
		Beteiligter[] zwischen;// = new Beteiligter[4];
		if (mitPlural)
			zwischen = new Beteiligter[] { new Beteiligter(beteiligtenTyp, GenusNumerus.MAENNLICH_SINGULAR),
					new Beteiligter(beteiligtenTyp, GenusNumerus.WEIBLICH_SINGULAR),
					new Beteiligter(beteiligtenTyp, GenusNumerus.MAENNLICH_PLURAL),
					new Beteiligter(beteiligtenTyp, GenusNumerus.WEIBLICH_PLURAL) };
		else
			zwischen = new Beteiligter[] { new Beteiligter(beteiligtenTyp, GenusNumerus.MAENNLICH_SINGULAR),
					new Beteiligter(beteiligtenTyp, GenusNumerus.WEIBLICH_SINGULAR) };
		/*
		 * Da compareTo() und equals() auf die laufendeNr schauen, muss ich die
		 * unterschiedlich setzen. Sonst können JComboBox-Instanzen das nicht mehr
		 * auswerten, die über sort() oder so zu gehen scheinen...
		 */
		for (Beteiligter b : zwischen)
			b.setLfdNr(b.getGenusNumerus().ordinal());
		return zwischen[0] == null ? null : zwischen;
	}

	/*
	 * Die Methode kurzBezeichner dient dazu, einen parteiBezeichner ohne bestimmten
	 * Artikel zurückzugeben.
	 * 
	 * @param casus eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * 
	 * @return einen String mit dem Bezeichner ohne Artikel und ohne laufende Nummer
	 */
//	@Deprecated
//	public String kurzBezeichner(int casus) {
//		String zwischen = getBezeichnerBeugungen(getTyp())[casus][genusNumerusAsInt()];
//		return zwischen.substring(4, zwischen.length());
//	}

	/**
	 * Die Methode kurzBezeichner dient dazu, einen parteiBezeichner ohne bestimmten
	 * Artikel zurückzugeben.
	 * 
	 * @param casus eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @return einen String mit dem Bezeichner ohne Artikel und ohne laufende Nummer
	 */
	public String kurzBezeichner(Casus casus) {
		String zwischen = getTyp().getBeugung()[casus.ordinal()][getGenusNumerus().ordinal()];
		return zwischen.substring(4, zwischen.length());
	}

	/**
	 * Die Methode erzeugt eine String-Repräsentation des Beteiligten durch Aufruf
	 * der Methode {@link kurzBezeichner(int)} im Nominativ ohne zusätzliche
	 * laufende Nummer.
	 * 
	 * @return einen Partei-Bezeichner als String ohne laufende Nummer und Artikel
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Hinzufuegbar#langBezeichnung()
	 */
	@Override
	public String langBezeichnung() {
		// damit das Feld lfdNr nicht "kaputt" geht, wird der Getter aufgerufen.
		// return parteiBezeichner(NOMINATIV, getLfdNr(), true);
		return kurzBezeichner(Casus.NOMINATIV);
	}

	/**
	 * Die Methode erzeugt eine {@link String}-Repräsentation des Beteiligten, die
	 * knapp alle Werte der Instanz aufführt. Nach Umbau in <code>0.6.1</code>
	 * eignet sich das Ergebnis nicht mehr zur Verarbeitung in einem Tenortext,
	 * dafür besser zur Ausgabe in Logbüchern oder für Testverfahren. Zur
	 * Textausgabe bitte {@link #langBezeichnung()} nehmen.
	 * 
	 * @return einen String, der die Werte {@link #beteiligtenTypAsInt()},
	 *         {@link #getLfdNr()}, {@link #genusNumerusAsInt()} und zur
	 *         Klarstellung auch {@link #isPlural()} aufnimmt.
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("Beteiligter [typ=");
		builder.append(typ);
		builder.append(", lfdNr=");
		builder.append(lfdNr);
		builder.append(", genusNumerus=");
		builder.append(genusNumerus);
		builder.append("(isPlural=");
		builder.append(isPlural() + ")");
		builder.append("]");
		return builder.toString();
	}

	/**
	 * Die Methode vergleicht zwei Beteiligte auf Gleichheit. Dabei stellt sie
	 * (außer in Fällen von Identität und null) auf Beteiligtentyp und
	 * LaufendeNummer ab.
	 * 
	 * @param andererBeteiligter the reference object with which to compare.
	 * @return true if this object is the same as the obj argument; false otherwise.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!(obj instanceof Beteiligter)) {
			return false;
		}
		Beteiligter other = (Beteiligter) obj;
		if (lfdNr != other.lfdNr) {
			return false;
		}
		if (typ != other.typ) {
			return false;
		}
		return true;
	}
//	@Override
//	public boolean equals(Object andererBeteiligter) {
//		if (this == andererBeteiligter) {
//			return true;
//		}
//		if (andererBeteiligter == null) {
//			return false;
//		}
//		if (getClass() != andererBeteiligter.getClass()) {
//			return false;
//		}
//		Beteiligter other = (Beteiligter) andererBeteiligter;
//		if (typ != other.typ) {
//			return false;
//		}
//		if (lfdNr != other.lfdNr) {
//			return false;
//		}
//		return true;
//	}
//
//	@Override
//	public int hashCode() {
//		final int prime = 31;
//		int result = 1;
//		result = prime * result + typ.ordinal();
//		result = prime * result + lfdNr;
//		return result;
//	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + lfdNr;
		result = prime * result + ((typ == null) ? 0 : typ.hashCode());
		return result;
	}

	/**
	 * Vergleicht zwei Beteiligter-Objekte, in erster Linie anhand des
	 * Beteiligtentyps, in zweiter Linie nach laufender Nummer. Dient nur für ein
	 * sort.
	 * 
	 * @param andererBeteiligter the object to be compared.
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Beteiligter andererBeteiligter) {
		// Wenn die Objekte gleich sind, 0 zurückgeben
		if (equals(andererBeteiligter))
			return 0;
		// Nur wenn die Typen identisch sind, auf die Laufende Nummer abstellen
		if (getTyp() == andererBeteiligter.getTyp()) {
			// an Integer.compareTo(o) delegieren
			return (new Integer(getLfdNr())).compareTo(new Integer(andererBeteiligter.getLfdNr()));
		} else {
			// Wenn die Typen ungleich sind, auf diese abstellen
			return getTyp().compareTo(andererBeteiligter.getTyp());
//			return (new Integer(getTyp()))
//					.compareTo(new Integer)
		}
	}

	/*
	 * Die Methode ist ein Getter für die privaten konstanten zweidimensionalen
	 * String[][]-Arrays mit den Beugeformen für die Beteiligtentypen.
	 * 
	 * @param beteiligtenTyp eine der Konstanten Beteiligter.KLAEGER,
	 * Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 * 
	 * @return ein String[][]-Array mit den Beugeformen des Beteiligtentyps, der
	 * dann mit String[casus][genusNumerus] benutzt werden kann.
	 */
//	@Deprecated
//	private static String[][] getBezeichnerBeugungen(int beteiligtenTyp) {
//		String[][] zwischen = null;
//		switch (beteiligtenTyp) {
//		case GERICHTSKOSTEN:
//			zwischen = GERICHTSKOSTEN_BEUGUNG;
//			break;
//		case BEKLAGTE:
//			zwischen = BEKLAGTEN_BEUGUNG;
//			break;
//		case DRITTWIDERBEKLAGTE:
//			zwischen = DRITTWIDERBEKLAGTEN_BEUGUNG;
//			break;
//		case KLAEGER:
//			zwischen = KLAEGER_BEUGUNG;
//			break;
//		default:
//			zwischen = null;
//		}
//		return zwischen;
//		return BeteiligtenTyp.values()[beteiligtenTyp].getBeugung();
//	}

}