package eu.gronos.kostenrechner.data.tenordaten;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import javax.xml.bind.annotation.XmlID;

import eu.gronos.kostenrechner.interfaces.Calculable;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;

/**
 * 
 * Eine {@link Number}, die einen Bruchteil repräsentiert.
 * 
 * Die Attribute der Klasse sind Zaehler und Nenner des Bruchs
 *
 * @author Peter Schuster (setrok)
 * @date 01.12.2021
 */
public class Fraction extends Number
		implements Comparable<Fraction>, Calculable<Fraction, Fraction>, HtmlRtfFormattierend {

	private static final long serialVersionUID = 3162722941330446595L;
	private static final String BRUCHSTRICH = "/";

	/**
	 * Englisch für den Zähler
	 */
	private final long numerator;
	/**
	 * Also der Nenner
	 */
	private final long denominator;

	public static final Fraction ZERO = new Fraction(0L, 1L);
	public static final Fraction ONE = new Fraction(1L, 1L);

	private static final Map<Long, Map<Long, Fraction>> POOL = new HashMap<>();
	private static final Map<Fraction, String> UNICODE_MAP = new HashMap<>();

	static {

		// 1
		HashMap<Long, Fraction> map = new HashMap<Long, Fraction>();
		map.put(0L, ZERO); // 0/1
		UNICODE_MAP.put(ZERO, "0");
		map.put(1L, ONE); // 1/1
		UNICODE_MAP.put(ONE, "1");
		POOL.put(1L, map);

		// 2
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 2L));
		UNICODE_MAP.put(map.get(1L), "\u00BD"); // ½
		POOL.put(2L, map);

		// 3
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 3L));
		UNICODE_MAP.put(map.get(1L), "\u2153"); // ⅓
		map.put(2L, new Fraction(2L, 3L));
		UNICODE_MAP.put(map.get(2L), "\u2154"); // ⅔
		POOL.put(3L, map);

		// 4
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 4L));
		UNICODE_MAP.put(map.get(1L), "\u00BC"); // ¼
		map.put(3L, new Fraction(3L, 4L));
		UNICODE_MAP.put(map.get(3L), "\u00BE"); // ¾
		POOL.put(4L, map);

		// 5
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 5L));
		UNICODE_MAP.put(map.get(1L), "\u2155"); // ⅕
		map.put(2L, new Fraction(2L, 5L));
		UNICODE_MAP.put(map.get(2L), "\u2156"); // ⅖
		map.put(3L, new Fraction(3L, 5L));
		UNICODE_MAP.put(map.get(3L), "\u2157"); // ⅗
		map.put(4L, new Fraction(4L, 5L));
		UNICODE_MAP.put(map.get(4L), "\u2158"); // ⅘
		POOL.put(5L, map);

		// 6
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 6L));
		UNICODE_MAP.put(map.get(1L), "\u2159"); // ⅙
		map.put(5L, new Fraction(5L, 6L));
		UNICODE_MAP.put(map.get(5L), "\u215A"); // ⅚
		POOL.put(6L, map);

		// 8
		map = new HashMap<Long, Fraction>();
		map.put(1L, new Fraction(1L, 8L));
		UNICODE_MAP.put(map.get(1L), "\u215B"); // ⅛
		map.put(3L, new Fraction(3L, 8L));
		UNICODE_MAP.put(map.get(3L), "\u215C"); // ⅜
		map.put(5L, new Fraction(5L, 8L));
		UNICODE_MAP.put(map.get(5L), "\u215D"); // ⅝
		map.put(7L, new Fraction(7L, 8L));
		UNICODE_MAP.put(map.get(7L), "\u215E"); // ⅞
		POOL.put(8L, map);

		map = null;
	}

	/**
	 * Der Konstruktor erstellt eine neue {@link Fraction} aus Zähler und Nenner,
	 * die er auch gleich kürzt.
	 * 
	 * @param numerator   der Zähler des neuen Bruchs als {@code long}
	 * @param denominator der Nenner des neuen Bruchs als {@code long}
	 */
	private Fraction(long numerator, long denominator) {
		if (denominator == 0)
			throw new ArithmeticException("denominator of a fraction must not be zero.");

		// denominator always positive
		if (denominator < 0) {
			numerator = -numerator;
			denominator = -denominator;
		}

		if (denominator == 1L) {
			this.numerator = numerator;
			this.denominator = 1L;
		} else {
			// Bruch kürzen
			long gcd = greatestCommonDivisor(numerator, denominator);
			this.numerator = numerator / gcd;
			this.denominator = denominator / gcd;
		}
	}

	/**
	 * Die Methode erzeugt eine Prozent-Angabe
	 * 
	 * @return ein {@code double}
	 */
	public double toPercent() {
		return 100.0 * doubleValue();
	}

	/**
	 * Die Methode formatiert einen Dezimalbruch als Prozent-Angabe
	 * 
	 * @param dezimalBruch ein {@code double} zwischen 0.0 und 1.0
	 * @return einen formattierten {@link String} im Stil "98,00%"
	 */
	public String toPercentString() {// private alsProzent(final double dezimalBruch)
		return String.format("%.2f%%", toPercent());// (dezimalBruch * 100.0)
	}

	/**
	 * The reciprocal of a fraction is another fraction with the numerator and
	 * denominator exchanged.
	 * 
	 * @return a new {@link Fraction} with {@link #getNumerator()} and
	 *         {@link #getDenomitator()} exchanged.
	 */
	public Fraction reciprocal() {
		return new Fraction(this.denominator, this.numerator);
	}

	/**
	 * Returns a {@code Fraction} instance representing the specified {@code double}
	 * value.
	 * 
	 * @see java.math.BigDecimal#valueOf(double)
	 * @see java.math.BigDecimal#pow(int)
	 * @see java.math.BigDecimal#scale
	 * 
	 * @see java.lang.Double#valueOf(double)
	 * @param d a double value.
	 * @return a {@code Fraction} instance representing {@code d}.
	 */
	public static Fraction valueOf(double d) {
		final Fraction f;
		/* https://de.wikipedia.org/wiki/Dezimalsystem#Dezimalbruchentwicklung */

		// Wert vor dem Komma abschneiden und merken
		final long whole = (long) d;

		// danach nur noch mit dem Nachkommawert weiter
		// final double remainder = Math.abs(d) - Math.abs(whole);
		final BigDecimal decimal = BigDecimal.valueOf(Math.abs(d) - Math.abs(whole));
		/*
		 * Wie viele Zehnerstellen der Dezimalbruch hat, liefert BigDecimal#scale
		 */
		final int scale = decimal.scale();

		// Stellen ermitteln:
		final byte[] array = digits(decimal, scale);// (remainder);

		if (scale > 3) {
			f = infiniteDecimalExpansion(decimal, scale, array)//
					.orElse(finiteDecimal(decimal, scale));
		} else {
			f = finiteDecimal(decimal, scale);// (remainder);
		}

		// Wert vor dem Komma mit hinzurechnen, wenn > 0
		if (whole > 0L) {//
			return new Fraction(whole, 1L).add(f);
		} else if (whole < 0L) {
			// negative Zahlen
			return new Fraction(whole, 1L).subtract(f);
		} else {
			return f;
		}
	}

	/**
	 * erstellt eine neue {@link Fraction} aus Zähler und Nenner, die er auch gleich
	 * kürzt.
	 * 
	 * @param numerator   der Zähler des neuen Bruchs als {@code long}
	 * @param denominator der Nenner des neuen Bruchs als {@code long}
	 */
	public static Fraction valueOf(long numerator, long denominator) {
		if (denominator == 0L)
			throw new ArithmeticException("Denominator must not be zero!");

		// mit dem Konstruktor einen neuen Bruch anlegen; der erledigt auch das Kürzen
		final Fraction fraction = new Fraction(numerator, denominator);

		// aus dem Pool holen, wenn vorhanden
		final Map<Long, Fraction> inner = POOL.get(fraction.denominator);
		if (inner != null) {
			final Fraction f = inner.get(fraction.numerator);
			if (f != null) {
				return f;
			}
		}

		// Keine passende Instanz im Pool, also den einfachen Bruch liefern
		return fraction;
	}

	/**
	 * @param s
	 * @return
	 */
	public static Fraction valueOf(String s) {
		if (s == null || s.trim().isEmpty())
			throw new IllegalArgumentException("String is null or empty");

		s = s.trim();

		if (s.contains("/")) {
			String[] parts = s.split("/");
			if (parts.length != 2)
				throw new NumberFormatException("Invalid fraction: " + s);

			long numerator = Integer.parseInt(parts[0].trim());
			long denominator = Integer.parseInt(parts[1].trim());

			return valueOf(numerator, denominator);
		}

		if (s.endsWith("%")) {
			String numberPart = s.substring(0, s.length() - 1).trim().replace(",", ".");
			try {
				return Fraction.valueOf(Double.parseDouble(numberPart) / 100.0);
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException("Ungültiges Prozentformat: " + s, e);
			}
		}

		// Kommazahl → Bruch
		// Optional: Kommazahl als unechter Bruch behandeln?
		try {
			return Fraction.valueOf(Double.parseDouble(s.replace(",", ".")));
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException("Kann String nicht in Fraction umwandeln: " + s, e);
		}

//		double decimal = Double.parseDouble(s);
//		return Fraction.of(decimal); // falls du so eine Methode hast
	}

	/**
	 * Mit Hilfe der Dezimalbruchentwicklung kann man jeder reellen Zahl eine Folge
	 * von Ziffern zuordnen. Jeder endliche Teil dieser Folge definiert einen
	 * Dezimalbruch, der eine Näherung der reellen Zahl ist. Man erhält die reelle
	 * Zahl selbst, wenn man von den endlichen Summen der Teile zur unendlichen
	 * Reihe über alle Ziffern übergeht (infinite decimal expansion)
	 * 
	 * https://de.wikipedia.org/wiki/Dezimalsystem#Dezimalbruchentwicklung
	 * 
	 * @param decimal der umzuwandelnde {@link BigDecimal}
	 * @param scale   die Zahl der Nachkommastellen aus {@link BigDecimal#scale()}
	 * @param array   ein Array aus byte mit allen Nachkommastellen des
	 *                Dezimalbruchs
	 * @return ein {@link Optional}, das eine {@link Fraction} enthalten kann oder
	 *         {@link Optional#empty()}, wenn nur Nullen oder kein periodischer
	 *         Bruch
	 */
	private static Optional<Fraction> infiniteDecimalExpansion(final BigDecimal decimal, final int scale,
			final byte[] array) {
		int begin = afterLeadingZeros(array);
		// Wenn nach dem Komma nur Nullen, dann Ganzzahl als Fraction zurückgeben
		if (begin == -1) {
			return Optional.empty();
		}

		int end = findPeriodFrom(array, begin);
		// wenn keine "reine" Wiederholung, dann beginnt die Periode vielleicht nicht
		// sofort nach dem Komma, dann eine Stelle weiter merken (also begin erhöhen
		// und
		// nochmal prüfen)
		while (end < begin && begin < array.length) {
			end = findPeriodFrom(array, ++begin);
		}

		// Nach Schleife: Sub-Array zu Bruch bauen,
		if (begin < array.length && end >= begin) {
			if (begin > 0) {
				return Optional.of(//
						infiniteDecimalExpansion(//
								Arrays.copyOfRange(array, 0, begin), //
								Arrays.copyOfRange(array, begin, end + 1)));
			} else {
				return Optional.of(//
						infiniteDecimalExpansion(//
								Arrays.copyOfRange(array, 0, end + 1)));
			}
		}
		return Optional.empty();
	}

	/**
	 * https://de.wikipedia.org/wiki/Dezimalsystem#Dezimalbruchentwicklung
	 * 
	 * Etwas komplizierter ist die Rechnung, wenn die Periode nicht unmittelbar auf
	 * das Komma folgt
	 * 
	 * @param beforePeriod ein Array aus byte mit einer oder mehr Nachkommastellen
	 *                     vor den periodischen
	 * @param period       ein Array aus byte mit einer oder mehr Nachkommastellen,
	 *                     die eine Periode darstellen
	 * @return eine {@link Fraction}
	 */
	private static Fraction infiniteDecimalExpansion(byte[] beforePeriod, byte[] period) {
		long pow1 = (long) Math.pow(10, beforePeriod.length);
		long pow2 = (long) Math.pow(10, period.length) - 1L;

		// wird der numerator des Bruchs
		// denominator ist dann (Basis ^ Sub-Array-Länge) - 1
		final long denominator = pow1 * pow2;

		final long numerator = (pow2 * combineDigits(beforePeriod)) + combineDigits(period);

		return new Fraction(numerator, denominator);
	}

	/**
	 * https://de.wikipedia.org/wiki/Dezimalsystem#Dezimalbruchentwicklung
	 * 
	 * Mit Hilfe der Dezimalbruchentwicklung kann man jeder reellen Zahl eine Folge
	 * von Ziffern zuordnen. Jeder endliche Teil dieser Folge definiert einen
	 * Dezimalbruch, der eine Näherung der reellen Zahl ist. Man erhält die reelle
	 * Zahl selbst, wenn man von den endlichen Summen der Teile zur unendlichen
	 * Reihe über alle Ziffern übergeht.
	 * 
	 * 
	 * Zur Umformung periodischer Dezimalbruchentwicklungen verwendet man die
	 * Beziehungen... Die Periode wird jeweils in den Zähler übernommen. Im Nenner
	 * stehen so viele Neunen, wie die Periode Stellen hat. Gegebenenfalls sollte
	 * der entstandene Bruch noch gekürzt werden.
	 * 
	 * @param period ein Array aus byte mit einer oder mehr Nachkommastellen, die
	 *               eine Periode darstellen
	 * @return eine {@link Fraction} mit einer Zehnerpotenz minus 1 als Nenner
	 */
	private static Fraction infiniteDecimalExpansion(byte[] period) {
		final long numerator = combineDigits(period);
		// wird der numerator des Bruchs
		// denominator ist dann (Basis ^ Sub-Array-Länge) - 1
		final long denominator = (long) Math.pow(10, period.length) - 1L;
		return new Fraction(numerator, denominator);
	}

	/**
	 * Liefert bei einem endlichen Dezimalbruch eine {@link Fraction} mit einer
	 * Zehnerpotenz als Nenner zurück.
	 * 
	 * @see java.math.BigDecimal#valueOf(double)
	 * @see java.math.BigDecimal#pow(int)
	 * @see java.math.BigDecimal#scale
	 * 
	 * @param decimal der endliche Dezimalbruch als {@link BigDecimal}
	 * @param scale   die Zahl der Stellen hinter dem Komma aus
	 *                {@link BigDecimal#scale()}
	 * @return eine neue {@link Fraction}
	 */
	private static Fraction finiteDecimal(final BigDecimal decimal, final int scale) {
		final Fraction f;
		/*
		 * wenn "glatt", also nicht periodisch: wie gehabt: Wie viele Zehnerstellen der
		 * Dezimalbruch hat, liefert BigDecimal#scale
		 */
		f = valueOf(// new Fraction
				decimal.scaleByPowerOfTen(scale).longValue(), //
				BigDecimal.TEN.pow(scale).longValue());
		return f;
	}

	/**
	 * Prüft im array ab einem bestimmten offset begin, ob eine periodische
	 * Wiederholung im Dezimalbruch vorliegt und liefert dessen Ende als int zurück.
	 * 
	 * @param array ein Array aus byte mit allen Nachkommastellen des Dezimalbruchs
	 * @param begin der Offset des Beginns, ab dem eine Periode vorliegen könnte
	 * @return den Offset des Endes der Periode als int oder -1 bei Fehlanzeige
	 */
	private static int findPeriodFrom(final byte[] array, int begin) {
		byte[] period;
		// Schleife, bis Wiederholung erkannt:
		for (int end = begin; end < array.length; end++) {
			// wenn Schleife bei letzter Zahl angelangt und noch keine Wiederholung
			// gefunden: Fehlanzeige zurückgeben
			if (end >= array.length - 1)
				return -1;
			// sub-array von 0 bis Zähler nehmen (i+1, weil exklusiv)
			period = Arrays.copyOfRange(array, begin, end + 1);
			// Prüfen, ob sich dieses sub-array darauffolgend wiederholt
			// wenn ja: Schleife abbrechen und Sub-Array merken
			if (isPeriodic(array, period, end + 1)) {
				return end;
			}
			// sonst weiter
		}
		// Fehlanzeige
		return -1;
	}

	/**
	 * Prüft im array ab einem bestimmten offset begin, ob eine periodische
	 * Wiederholung im Dezimalbruch in Form von period vorliegt
	 * 
	 * @param array  ein Array aus byte mit allen Nachkommastellen des Dezimalbruchs
	 * @param period ein Array aus byte mit den Nachkommastellen, die eine Periode
	 *               darstellen könnten
	 * @param begin  der Offset des Beginns, ab dem eine Periode vorliegen könnte
	 * @return {@code true}, wenn sich period im Rest des array ab begin wiederholt
	 */
	private static boolean isPeriodic(byte[] array, byte[] period, int begin) {
		/*
		 * im array bei index begin anfangen
		 */
		for (int i = begin; i < array.length; i += period.length) {
			/*
			 * Teilkopie bilden, die period.length lang ist, aber gedeckelt, wenn der Rest
			 * des array kürzer ist
			 */
			final int end = (begin + period.length) <= array.length - 1 //
					? begin + period.length //
					: array.length;
			final byte[] remainingPart = Arrays.copyOfRange(array, begin, end);
			// final byte[] part = Arrays.copyOfRange(array, i, i + period.length);
			final byte[] periodToCompare = remainingPart.length < period.length//
					? Arrays.copyOfRange(period, 0, remainingPart.length)//
					: period;
			/*
			 * auf Gleichheit mit period prüfen wenn ja: wiederholen, bis Ende des array
			 * erreicht; sonst: false zurückgeben
			 */
			if (!Arrays.equals(remainingPart, periodToCompare))
				return false;
		}
		return true;
	}

	/**
	 * Prüft ein array aus byte auf führende Nullen
	 * 
	 * @param array ein Array aus byte mit allen Nachkommastellen des Dezimalbruchs
	 * @return den Offset der ersten Stelle, die keine Null mehr ist als {@code int}
	 */
	private static int afterLeadingZeros(final byte[] array) {
		int begin = 0;
		// Wahrscheinlich muss ich führende Nullen ignorieren
		while (array[begin] == 0) {
			begin++;
			if (begin > array.length - 1) {
				// signalisieren, dass nur Nullen
				return -1;
			}
		}
		return begin;
	}

	/**
	 * Array aus einzelnen Nachkommastellen aufbauen: remainder immer wieder mit
	 * Basis (10) multiplizieren mit (int) remainder aufsammeln
	 * 
	 * @param remainder der umzuwandelnde {@link BigDecimal}
	 * @param scale     die Zahl der Nachkommastellen aus {@link BigDecimal#scale()}
	 * @return ein Array aus byte mit allen Nachkommastellen des Dezimalbruchs mit
	 *         scale Elementen
	 */
	private static byte[] digits(BigDecimal remainder, final int scale) {
		final BigDecimal base = BigDecimal.TEN;
		final byte[] bs = new byte[scale];

		for (int i = 0; i < scale; i++) {
			byte vorKomma = remainder.multiply(base).byteValue();
			bs[i] = vorKomma;
			remainder = remainder.multiply(base).subtract(BigDecimal.valueOf(vorKomma));
		}
		return bs;
	}

	/**
	 * @param bytes ein Array aus byte mit den umzuwandelnden Nachkommastellen des
	 *              Dezimalbruchs
	 * @return den daraus ermittelten Zähler als {@code long}
	 */
	private static long combineDigits(byte[] bytes) {
		final BigInteger base = BigInteger.TEN;
		BigInteger result = BigInteger.ZERO;

		for (int i = 0; i < bytes.length; i++) {
			result = result.multiply(base);
			result = result.add(BigInteger.valueOf(bytes[i]));
		}

		return result.longValue();
	}

	/**
	 * Benutzt {@link BigInteger#gcd(BigInteger)}, um den größten gemeinsamen Teiler
	 * (gcd) von zwei Nennern zu ermitteln.
	 * 
	 * @param a erster Nenner als {@code long}
	 * @param b zweiter Nenner als {@code long}
	 * @return den ggT
	 */
	private static long greatestCommonDivisor(long a, long b) {
		return BigInteger.valueOf(a).gcd(BigInteger.valueOf(b)).longValue();
	}

	/**
	 * The lcm is the lowest common denominator that can be used before fractions
	 * can be added subtracted or compared
	 * 
	 * @param denominator1 der Nenner des ersten Bruchs
	 * @param denominator2 der Nenner des zweiten Bruchs
	 * @return the least common multiple or lowest common denominator
	 * 
	 * @url https://en.wikipedia.org/wiki/Least_common_multiple
	 */
	private static long lowestCommonMultiple(final long denominator1, final long denominator2) {
		if (denominator1 == 0L && denominator2 == 0L)
			throw new ArithmeticException("Es dürfen nicht beide Nenner == 0 sein!");
		return BigInteger.valueOf(denominator1)//
				.multiply(BigInteger.valueOf(denominator2))//
				.divide(BigInteger.valueOf(greatestCommonDivisor(denominator1, denominator2)))//
				.longValue();
	}

	/**
	 * Um Brüche mit zu großen Nennern zu vermeiden (zB 13577/31569), kann diese
	 * Methode den Bruch runden bis maximal zu dem angegebenen Wert
	 * 
	 * @param maxDenominator der maximal zulässige Nenner, z.B. 99
	 * @return den gerundeten Bruch oder den ursprünglichen Bruch, wenn keine
	 *         Rundung möglich war.
	 * 
	 */
	public Fraction approx(long maxDenominator) {
		if (this.denominator <= maxDenominator)
			return this;

		long bestNumerator = this.numerator;
		long bestDenominator = this.denominator;

		double bestError = Double.MAX_VALUE;
		double value = doubleValue();

		for (long denom = 1L; denom <= maxDenominator; denom++) {
			long numer = Math.round(value * denom);

			// Überspringen, wenn durch Rundung der Zähler 0 wird
			if (numer == 0)
				continue;

			double approx = (double) numer / denom;
			double error = Math.abs(approx - value);

			if (error < bestError) {
				bestNumerator = numer;
				bestDenominator = denom;
				bestError = error;
			}
		}

		// Falls keine Verbesserung möglich war oder der Bruch trivialisiert würde,
		// gib auf und gib den Ursprungsbruch zurück
		if (bestNumerator == 0 || bestDenominator > maxDenominator)
			return this;

		return new Fraction(bestNumerator, bestDenominator);
	}

	/**
	 * Die Methode ermittelt den größeren von diesem und dem anderen
	 * {@link Fraction}-Bruch
	 * 
	 * @param other der zu vergleichende {@link Fraction}-Bruch
	 * @return {@code this} oder {@code other}
	 * 
	 * @see eu.gronos.kostenrechnerjunits.BaumbachAngriffeTest.Calculable#max(java.lang.Number)
	 */
	@Override
	public Fraction max(Fraction other) {
		return compareTo(other) > 0 ? this : other;
	}

	@Override
	public int intValue() {
		return (int) (numerator / denominator);
	}

	@Override
	public long longValue() {
		return (numerator / denominator);
	}

	@Override
	public float floatValue() {
		return (float) numerator / denominator;
	}

	/**
	 * Die Methode wandelt den Bruch in einen Dezimalbruch um
	 * 
	 * @return ein {@code double}
	 * 
	 * @see java.lang.Number#doubleValue()
	 */
	@Override
	public double doubleValue() {
		return (double) numerator / denominator;
	}

	/**
	 * Die Methode erzeugt einen hashCode anhand der gekürzten Zähler und Nenner.
	 * 
	 * @return einen {@code int}
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@XmlID
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (int) (this.denominator ^ (this.denominator >>> 32));
		result = prime * result + (int) (this.numerator ^ (this.numerator >>> 32));
		return result;
	}

	/**
	 * Die Methode vergleicht die Brüche über {@link #compareTo(Fraction)}.
	 * 
	 * @param other ein {@link Object}, am besten eine {@link Fraction}
	 * 
	 * @return {@code true}, wenn es sich bei {@code other} auch um eine
	 *         {@link Fraction} handelt und {@link #compareTo(Fraction)} == 0
	 *         ergibt.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof Fraction))
			return false;

		Fraction other = (Fraction) obj;
		return // denominator == other.denominator && numerator == other.numerator;
		compareTo(other) == 0;
	}

	/**
	 * Die Methode vergleicht den {@link Fraction}-Bruch mit einem anderen
	 * 
	 * 
	 * Compares two {@link Fraction} values numerically. The value returned is
	 * identical to what would be returned by:
	 * 
	 * <pre>
	 * Fraction.of(firstNumerator, firstDenomitator).compareTo(Fraction.of(secondNumerator, secondDenomitator))
	 * </pre>
	 * 
	 * @param other die andere {@link Fraction}
	 * @return the value {@code 0} if {@code this == other}; a value less than
	 *         {@code 0} if {@code this < other}; and a value greater than {@code 0}
	 *         if {@code this > other}
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 * @see java.lang.Long#compare(long, long)
	 */
	@Override
	public int compareTo(Fraction other) {
		/*
		 * Compares two {@link Fraction} values numerically. The value returned is
		 * identical to what would be returned by:
		 * 
		 * <pre> Fraction.of(firstNumerator,
		 * firstDenomitator).compareTo(Fraction.of(secondNumerator, secondDenomitator))
		 * </pre>
		 * 
		 * @param firstNumerator the {@link #getNumerator()} of the first {@link
		 * Fraction} ({@code x}) to compare
		 * 
		 * @param firstDenomitator the {@link #getDenomitator()} of the first {@link
		 * Fraction} ({@code x}) to compare
		 * 
		 * @param secondNumerator the {@link #getNumerator()} of the second {@link
		 * Fraction} ({@code y}) to compare
		 * 
		 * @param secondDenomitator the {@link #getDenomitator()} of the second {@link
		 * Fraction} ({@code y}) to compare
		 * 
		 * @return the value {@code 0} if {@code x == y}; a value less than {@code 0} if
		 * {@code x < y}; and a value greater than {@code 0} if {@code x > y}
		 */
		return Long.compare(this.numerator * other.denominator, other.numerator * this.denominator);
	}

	/**
	 * Die Methode erzeugt eine {@link String}-Darstellung des Bruchs
	 * 
	 * @return eine {@link String}-Darstellung des Bruchs im Format &quot;1/2&quot;
	 *         (also &lt;Zähler>/&lt;Nenner>)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return numerator == 0 ? "0" : //
				denominator == 1 ? "" + numerator : //
						numerator + BRUCHSTRICH + denominator;
	}

	/**
	 * @return einen String mit HTML-Formattierungsanweisung für einen
	 *         hochgestellten Zähler und einen tiefergestellten Nenner.
	 */
	@Override
	public String toHtmlString() {
		if (UNICODE_MAP.containsKey(this))
			return UNICODE_MAP.get(this);
		return "<html><sup><small>" + numerator + "</small></sup>&frasl;<sub><small>" + denominator
				+ "</small></sub></html>";
	}//

	/**
	 * @return einen String mit RTF-Formattierungsanweisung für einen hochgestellten
	 *         Zähler und einen tiefergestellten Nenner.
	 */
	@Override
	public String toRtfString() {
		if (UNICODE_MAP.containsKey(this))
			return UNICODE_MAP.get(this);
		return "{\\super " + numerator + "}\\'2f{\\sub " + denominator + "}";
	}

	/**
	 * Die Methode addiert zum vorliegenden einen weiteren {@link Fraction}-Bruch
	 * 
	 * @param summand eine weitere {@link Fraction}
	 * @return {@code this +} {@code summand} über das kleinste gemeinsame Vielfache
	 *         {@link #lowestCommonMultiple(long, long)}
	 * @see java.math.BigInteger#add(java.math.BigInteger)
	 */
	@Override
	public Fraction add(Fraction summand) {
		return add(this.numerator, this.denominator, summand.numerator, summand.denominator);
	}

	/**
	 * Die statische Methode addiert zwei Brüche, die als ihre Bestandteile
	 * übergeben werden.
	 * 
	 * @param firstNumerator    Zähler des ersten Bruchs als {@code long}
	 * @param firstDenomitator  Nenner des ersten Bruchs als {@code long}
	 * @param secondNumerator   Zähler des zweiten Bruchs als {@code long}
	 * @param secondDenomitator Nenner des zweiten Bruchs als {@code long}
	 * 
	 * @return die Summe als ein neuer, gekürzter {@link Fraction}-Bruch
	 */
	private static Fraction add(final long firstNumerator, final long firstDenomitator, final long secondNumerator,
			final long secondDenomitator) {
		final long lCM = lowestCommonMultiple(firstDenomitator, secondDenomitator);
		return new Fraction((firstNumerator * lCM / firstDenomitator) + (secondNumerator * lCM / secondDenomitator),
				lCM);
	}

	/**
	 * Die Methode zieht vom vorliegenden einen anderen {@link Fraction}-Bruch ab,
	 * dies über den {@link #lowestCommonMultiple(long, long)}
	 * 
	 * @param subtrahend der abzuziehende {@link Fraction}-Bruch
	 * @return die Differenz als neuer, gekürzter {@link Fraction}-Bruch
	 */
	@Override
	public Fraction subtract(Fraction subtrahend) {
		return add(this.numerator, this.denominator, //
				-subtrahend.numerator, subtrahend.denominator);
	}

	/**
	 * Die statische Methode multipliziert zwei Brüche, die als ihre Bestandteile
	 * übergeben werden.
	 * 
	 * @param firstNumerator    Zähler des ersten Bruchs als {@code long}
	 * @param firstDenomitator  Nenner des ersten Bruchs als {@code long}
	 * @param secondNumerator   Zähler des zweiten Bruchs als {@code long}
	 * @param secondDenomitator Nenner des zweiten Bruchs als {@code long}
	 * 
	 * @return die Summe als ein neuer, gekürzter {@link Fraction}-Bruch
	 */
	private static Fraction multiply(final long firstNumerator, final long firstDenomitator, final long secondNumerator,
			final long secondDenomitator) {
		return //
		new Fraction(firstNumerator * secondNumerator, firstDenomitator * secondDenomitator);
	}

	/**
	 * Die Methode multipliziert den vorliegenden Bruch mit einem anderen.
	 * 
	 * @param factor eine {@link Fraction}, mit dem {@code this} multipliziert
	 *               werden soll
	 * @return {@code this * factor} als neue {@link Fraction}
	 * 
	 * @see java.math.BigInteger#multiply(java.math.BigInteger)
	 */
	@Override
	public Fraction multiply(Fraction factor) {
		return multiply(this.numerator, this.denominator, factor.numerator, factor.denominator);
	}

	/**
	 * Die Methode erzeugt eine neue {@link Fraction} aus {@code this / divisor}
	 * durch Multiplikation mit dem Kehrwert
	 * 
	 * @param divisor die {@link Fraction}, durch die geteilt werden soll
	 * @return {@code this / divisor} als neue {@link Fraction}
	 * 
	 * @see java.math.BigInteger#divide(java.math.BigInteger)
	 */
	@Override
	public Fraction divide(Fraction divisor) {
		return multiply(this.numerator, this.denominator, divisor.denominator, divisor.numerator);
	}

	/**
	 * Die Methode vergleicht den {@link Fraction}-Bruch mit einem anderen
	 * 
	 * @param other die andere {@link Fraction}
	 * @return {@code true} wenn der vorliegende Bruch größer ist als der andere
	 * 
	 * @see eu.gronos.kostenrechnerjunits.BaumbachAngriffeTest.Calculable#greaterThan(java.lang.Number)
	 * @see java.lang.Long#compare(long, long)
	 */
	@Override
	public boolean greaterThan(Fraction other) {
		return // compare(this.numerator, this.denominator, other.numerator, other.denominator)
		compareTo(other) > 0;
	}

	/**
	 * Die Methode vergleicht den {@link Fraction}-Bruch mit einem anderen
	 * 
	 * @param other die andere {@link Fraction}
	 * @return {@code true} wenn der vorliegende Bruch kleiner ist als der andere
	 * 
	 * @see eu.gronos.kostenrechnerjunits.BaumbachAngriffeTest.Calculable#lessThan(java.lang.Number)
	 * @see java.lang.Long#compare(long, long)
	 */
	@Override
	public boolean lessThan(Fraction other) {
		return // compare(this.numerator, this.denominator, other.numerator, other.denominator)
		compareTo(other) < 0;
	}

	@Override
	public boolean greaterThanOrEqualTo(Fraction other) {
		return compareTo(other) >= 0;
	}

	@Override
	public boolean lessThanOrEqualTo(Fraction other) {
		return compareTo(other) <= 0;
	}

}