/**
 * KostenTenorContainer.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.tenordaten;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.datatype.XMLGregorianCalendar;

import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;

/**
 * Containerklasse, um alle Zustände im XML-Format speichern zu können.
 *
 * @author Peter Schuster (setrok)
 * @date 26.12.2014
 *
 */
@XmlRootElement(name = "SchustersKleinerKostenTenorXML", namespace = "http://www.kostentenor.de/")
public class VerfahrensDatenContainer {

	@XmlElement(name = "programminfo")
	public Dateikopf dateikopf = new Dateikopf();

	@XmlElement(name = "allgemein")
	public Allgemein allgemein = new Allgemein();

	@XmlElement(name = "teilklageruecknahme")
	public Teilklageruecknahme teilklageruecknahme = new Teilklageruecknahme();

	@XmlElement(name = "streitgenossen")
	public Streitgenossen streitgenossen = new Streitgenossen();

	@XmlElement(name = "forderungsstaffelung")
	public ForderungsStaffelung staffelung = new ForderungsStaffelung();

	@XmlElement(name = "gebuehrenberechnung")
	public GebuehrenBerechnung gebuehrenBerechnung = new GebuehrenBerechnung();

	public VerfahrensDatenContainer() {
		super();
	}

	/**
	 * Convenience-Konstruktor, die für Programmteile ein neues Parameterobjekt für
	 * die Verfahrensdaten erzeugt, die früher VerfahrensDatenContainer genutzt
	 * haben, vor allem für den {@link BaumbachAssistentDialog}.
	 * 
	 * @param klaeger             die BaumbachBeteiligtenListe mit den Klägern
	 * @param drittwiderbeklagten die BaumbachBeteiligtenListe mit den
	 *                            Drittwiderbeklagten
	 * @param beklagten           die BaumbachBeteiligtenListe mit den Beklagten
	 * @param streitwert          den Streitwert als {@link Euro}
	 * @param aktenzeichen        das Aktenzeichen als String
	 * @param speicherDatum       {@link XmlTransferHandler#speicherDatum()}
	 */
	public VerfahrensDatenContainer(BaumbachBeteiligtenListe klaeger, BaumbachBeteiligtenListe drittwiderbeklagte,
			BaumbachBeteiligtenListe beklagte, Euro streitwert, String aktenzeichen,
			XMLGregorianCalendar speicherDatum) {
		this();
		/* Erst die BaumbachBeteiligtenListen einpacken */
		streitgenossen.klaeger = klaeger;
		streitgenossen.drittwiderbeklagte = drittwiderbeklagte;
		streitgenossen.beklagte = beklagte;
		/*
		 * Vorher gab's kein Feld fürs Vorhandensein einer Widerklage, jetzt gibt's
		 * eins, dann muss es auch richtig gefüllt werden. Die Methode bleibt dieselbe.
		 */
		streitgenossen.widerklage = streitgenossen.gibtsDrittwiderbeklagte()
				|| (streitgenossen.gibtsKlaeger() && streitgenossen.klaeger.enthaeltAnWiderklageBeteiligte())
				|| (streitgenossen.gibtsBeklagte() && streitgenossen.beklagte.enthaeltAnWiderklageBeteiligte());
		/* Dann die allgemeinen Teile einpacken */
		allgemein.speicherdatum = speicherDatum;
		allgemein.aktenzeichen = aktenzeichen;
		allgemein.streitwert = streitwert;
		allgemein.selectedPanel = Streitgenossen.TAB_ID;
		/* Damit der Rest nicht leer ist */
		kopiereBeteiligteAusStreitgenossen();
		alleStreitwerteHinauf();
	}

	/**
	 * Die Methode dient dazu, die Daten aus den Untercontainern {@link Dateikopf}
	 * und {@link Allgemein} zu einem String zusammenzufassen
	 * 
	 * @todo TODO {@link Allgemein#toString()} und {@link Dateikopf#toString()}
	 *       nutzen!
	 * 
	 * @return ein String
	 */
	public String createFileInfo() {
		GregorianCalendar gregorianCalendar = new GregorianCalendar();
		if (allgemein.speicherdatum != null) {
			gregorianCalendar = allgemein.speicherdatum.toGregorianCalendar();
		}
		return String.format(
				"Datei-/Systeminfo [speicherdatum=%s, aktenzeichen=%s, version=%s, osName=%s, osArch=%s, osVersion=%s, javaVendor=%s, javaName=%s, javaVersion=%s]",
				DateFormat.getDateInstance(DateFormat.MEDIUM, Locale.GERMAN).format(gregorianCalendar.getTime()),
				allgemein.aktenzeichen, dateikopf.version, dateikopf.osName, dateikopf.osArch, dateikopf.osVersion,
				dateikopf.javaVendor, dateikopf.javaName, dateikopf.javaVersion);
	}

	/**
	 * Die Methode sucht den höchsten Streitwert aus den UnterContainern heraus und
	 * schreibt ihn in {@link Allgemein#streitwert}
	 *
	 */
	public void setzeStreitwertInAllgemein() {
		allgemein.streitwert = // Euro.ofEuros()
				maxStreitwert();
	}

	/**
	 * Die Methode setzt alle Streitwertangaben auf den im
	 * {@link VerfahrensDatenContainer} angegebenen höchsten Streitwert der
	 * Streitgenossen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 * @return den "aufgefüllten" {@link VerfahrensDatenContainer}
	 */
	private VerfahrensDatenContainer alleStreitwerteHinauf() {
		final Euro streitwert = allgemein.streitwert;// double
		if (streitwert.compareTo(Euro.ZERO_CENTS) <= 0)//
			return this;
		/* Zuerst auf Registerkarte 1 verteilen */
		Teilklageruecknahme t = teilklageruecknahme;
		t.streitwerteUndObsiegen = new ArrayList<Euro>();// Long
		t.streitwerteUndObsiegen.add(streitwert);// (long)
		t.streitwerteUndObsiegen.add(streitwert);// (long)
		t.streitwerteUndObsiegen.add(streitwert);// (long)
		/* Dann auf Registerkarte 4 */
		gebuehrenBerechnung.streitwert = streitwert;// (long)
		/* und Rückgabe */
		return this;
	}

	/**
	 * Die Methode dient dazu, die Beteiligten hinsichtlich genusNumerus anhand des
	 * Inhalts der {@link BaumbachBeteiligtenListe BaumbachBeteiligtenListen} der
	 * Streitgenossen zu setzen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 * @return den "aufgefüllten" {@link VerfahrensDatenContainer}
	 */
	private VerfahrensDatenContainer kopiereBeteiligteAusStreitgenossen() {
		/* Erst den Genus/Numerus zu ermitteln */
		BaumbachBeteiligtenListe klaegerseite = new BaumbachBeteiligtenListe();
		klaegerseite.addAll(streitgenossen.klaeger);
		if (streitgenossen.drittwiderbeklagte != null)
			klaegerseite.addAll(streitgenossen.drittwiderbeklagte);
		Beteiligter kl = klaegerseite.ermittleGenusNumerusForAll(BeteiligtenTyp.KLAEGER);
		BaumbachBeteiligtenListe beklagtenseite = new BaumbachBeteiligtenListe();
		beklagtenseite.addAll(streitgenossen.beklagte);
		Beteiligter bk = beklagtenseite.ermittleGenusNumerusForAll(BeteiligtenTyp.BEKLAGTE);
		ArrayList<Beteiligter> parteien = new ArrayList<Beteiligter>();
		/* Dann das Ergebnis einsetzen */
		parteien.add(kl);
		parteien.add(bk);
		teilklageruecknahme.parteien = parteien;
		staffelung.parteien = parteien;
		/* Und zurückgeben */
		return this;
	}

	/**
	 * Die Methode sucht den höchsten Streitwert aus den UnterContainern heraus
	 * 
	 * @return den maximalen Streitwert
	 */
	private Euro maxStreitwert() {
//		return Math.max(streitgenossen.klaeger.getHoechstenStreitwert(BeteiligtenTyp.KLAEGER).doubleValue(),
//				Math.max(streitgenossen.drittwiderbeklagte.getHoechstenStreitwert(BeteiligtenTyp.DRITTWIDERBEKLAGTE).doubleValue(),
//						Math.max(streitgenossen.beklagte.getHoechstenStreitwert(BeteiligtenTyp.BEKLAGTE).doubleValue(),
//								Math.max(gebuehrenBerechnung.streitwert.doubleValue(), // (double)
//										teilklageruecknahme.streitwerteUndObsiegen.get(0).doubleValue()))));
		return Euro.max(streitgenossen.klaeger.getHoechstenStreitwert(BeteiligtenTyp.KLAEGER),
				Euro.max(streitgenossen.drittwiderbeklagte.getHoechstenStreitwert(BeteiligtenTyp.DRITTWIDERBEKLAGTE),
						Euro.max(streitgenossen.beklagte.getHoechstenStreitwert(BeteiligtenTyp.BEKLAGTE),
								Euro.max(gebuehrenBerechnung.streitwert, // (double)
										teilklageruecknahme.streitwerteUndObsiegen.get(0)))));
	}

}
