/**
 * VollstreckbarkeitsListe.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.tenordaten;

import java.util.ArrayList;

import eu.gronos.kostenrechner.util.VollstreckbarkeitsPruefer;

/**
 * Liste für {@link Vollstreckbarkeit}en
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 10.11.2014
 */
public class VollstreckbarkeitsListe extends ArrayList<Vollstreckbarkeit> {

	private static final long serialVersionUID = 3941238377555955826L;
	private final VollstreckbarkeitsPruefer pruefer = new VollstreckbarkeitsPruefer();

	/**
	 * Die Methode prüft, ob §§ 708, 713 ZPO einschlägig ist.
	 * 
	 * @return <code>true</code>, wenn die Liste aus nur einem Element besteht und
	 *         dieses das Flag {@link Vollstreckbarkeit#b713} gesetzt hat.
	 */
	public boolean sindAlle708iVm713() {
		if (size() == 1 && get(0).isB713()) {
			return true;
		}
		return false;
	}

	/**
	 * Die Methode prüft, ob alle Elemente der Liste die Flags für § 709 S. 1 ZPO
	 * gesetzt haben.
	 * 
	 * @return <code>true</code>, wenn alle Elemente
	 *         {@link Vollstreckbarkeit#b709s1} gesetzt haben.
	 */
	public boolean sindAlle709() {
		boolean alle709 = false;
		/* Erst einmal gucken, ob überhaupt ein Element § 709 S. 1 hat. */
		if (!isEmpty()) {
			for (Vollstreckbarkeit vo : this) {
				if (vo.isB709s1())
					alle709 = true;
			}
		}
		/* Wenn nicht, war's das schon. */
		if (!alle709)
			return alle709;
		/*
		 * Dann gucken, ob welche dabei sind, die weder leer sind noch § 709 S. 1 ZPO.
		 * Wenn solche gefunden werden, ist der Rückgabewert false.
		 */
		if (!isEmpty()) {
			for (Vollstreckbarkeit vo : this) {
				if (!vo.isB709s1() && vo.hatEintraege())
					alle709 = false;
			}
		}
		return alle709;
	}

	/**
	 * Die Methode prüft, ob alle Elemente der Liste die Flags für §§ 708, 711 ZPO
	 * gesetzt haben.
	 * 
	 * @return <code>true</code>, wenn alle Elemente {@link Vollstreckbarkeit#b711}
	 *         gesetzt haben.
	 */
	public boolean sindAlle708iVm711() {
		boolean alle708iVm711 = false;
		/* Erst einmal gucken, ob überhaupt ein Element § 711 hat. */
		if (size() > 0) {
			for (Vollstreckbarkeit vo : this) {
				if (vo.isB711())
					alle708iVm711 = true;
			}
		}
		/* Wenn nicht, war's das schon. */
		if (!alle708iVm711)
			return alle708iVm711;
		/*
		 * Dann gucken, ob welche dabei sind, die weder leer sind noch § 711 ZPO. Wenn
		 * solche gefunden werden, ist der Rückgabewert false.
		 */
		if (size() > 0) {
			for (Vollstreckbarkeit vo : this) {
				if (!vo.isB711() && vo.hatEintraege())
					alle708iVm711 = false;
			}
		}
		return alle708iVm711;
	}

	/**
	 * Die Methode dient dazu, ein Element der Liste hinzufügen und vorher die
	 * Plausis zu checken.
	 * 
	 * @param vo das hinzufügende {@link Vollstreckbarkeit}
	 * @return <code>false</code>, wenn
	 *         {@link Vollstreckbarkeit#pruefeEingabe(Vollstreckbarkeit)}
	 *         <code>false</code> liefert, sonst <code>true</code>
	 * 
	 * @see java.util.ArrayList#add(java.lang.Object)
	 */
	@Override
	public boolean add(Vollstreckbarkeit vo) {
		if (!pruefer.pruefeEingabe(vo))
			return false;
		else
			return super.add(vo);
	}

	/**
	 * Die Methode, die alle verwenden Vorschriften herausfindet und als String
	 * zurückgibt.
	 * 
	 * @return &quot;Die Entscheidung über die vorläufige Vollstreckbarkeit beruht
	 *         auf &quot; als {@link String}.
	 */
	@Override
	public String toString() {
		boolean b708 = false;
		boolean b709s1 = false;
		boolean b709s2 = false;
		boolean b711 = false;
		boolean b713 = false;
		/* Liste durchgehen und Vorschriften sammeln */
		for (Vollstreckbarkeit vo : this) {
			b708 = vo.isB708() || b708;
			b709s1 = vo.isB709s1() || b709s1;
			b709s2 = vo.isB709s2() || b709s2;
			b711 = vo.isB711() || b711;
			b713 = vo.isB713() || b713;
		}
		/* Text zusammensetzen */
		StringBuilder string = new StringBuilder("Die Entscheidung über die vorläufige Vollstreckbarkeit beruht auf ");
		if (b708)
			string.append("§ 708 ZPO");
		/*
		 * Wenn beide Sätze von § 709 enthalten sind, die ganze Vorschrift zitieren,
		 * sonst nur den einzelnen Satz
		 */
		if (b709s1 && b709s2)
			string.append("§ 709 ZPO");
		else if (b709s1)
			string.append("§ 709 Satz 1 ZPO");
		else if (b709s2)
			string.append("§ 709 Satz 2 ZPO");
		if (b711)
			string.append("§ 711 ZPO");
		if (b713)
			string.append("§ 713 ZPO");
		/*
		 * Wenn der Offset des ersten § in der Zeichenkette sich vom Offset des letzten
		 * unterscheidet, dann gibt es mehr als eines und es muss "§§" statt nur "§"
		 * heißen.
		 */
		final int indexOfSectionSign = string.indexOf("§");
		if (indexOfSectionSign != -1 && indexOfSectionSign != string.lastIndexOf("§"))
			string.insert(indexOfSectionSign, '§');
		string.append(".");
		return string.toString().replace("ZPO§", ", ").replaceAll("\\s+", " ").replace(" , ", ", ");
	}
}
