package eu.gronos.kostenrechner.interfaces;

/**
 * Das Interface markiert {@link Number}-Unterklassen, die mathematische
 * Grundfunktionen wie {@link Calculable#add(Number)},
 * {@link Calculable#subtract(Number)}, {@link Calculable#multiply(Number)} und
 * {@link Calculable#divide(Number)} beherrschen
 *
 * @author Peter Schuster (setrok)
 * @date 06.12.2021
 *
 * @param <N> Verweis auf die Kind-Klasse selbst
 * @param <S> Verweis auf die Kind-Klasse selbst oder Parameter-Typ der
 *            {@link Calculable#multiply(Number)} bzw. Rückgabetyp der
 *            {@link Calculable#divide(Number)}-Methode.
 */
public interface Calculable<N extends Number & Comparable<N>, S extends Number> {

	/**
	 * Die Methode dient dazu, einen {@link Calculable} zum Vorhandenen zu addieren.
	 * 
	 * @param summand ein {@link Calculable} vom Typ des Parameters {@code N}, das
	 *                hinzugerechnet werden soll.
	 * @return ein neues {@link Calculable} aus {@code this + summand}
	 * @see java.math.BigInteger#add(java.math.BigInteger)
	 */
	N add(N summand);

	/**
	 * Die Methode dient dazu, einen {@link Calculable} vom Vorhandenen abzuziehen.
	 * 
	 * @param subtrahend ein {@link Calculable} vom Typ des Parameters {@code N},
	 *                   das abgezogen werden soll.
	 * @return ein neues {@link Calculable} aus {@code this - subtrahend}
	 * @see java.math.BigInteger#subtract(java.math.BigInteger)
	 */
	N subtract(N subtrahend);

	/**
	 * Die Methode dient dazu, den größeren von diesem und dem anderen
	 * {@link Calculable} zu ermitteln
	 * 
	 * @param other der zu vergleichende {@link Calculable}
	 * @return {@code this} oder {@code other}
	 * 
	 * @see java.math.BigInteger#max(java.math.BigInteger)
	 */
	N max(N other);

	/**
	 * Die Methode dient dazu, den vorhandenen {@link Calculable} mit dem
	 * {@code factor} zu multiplizieren.
	 * 
	 * @param factor eine {@link Number} vom Typ des Parameters {@code S}, mit dem
	 *               zu multiplizieren ist
	 * @return ein neues {@link Calculable} aus {@code this * factor}
	 * @see java.math.BigInteger#multiply(java.math.BigInteger)
	 */
	N multiply(S factor);

	/**
	 * Die Methode dient dazu, den vorhandenen {@link Calculable} durch den
	 * {@code divisor} zu teilen.
	 * 
	 * @param divisor ein {@link Calculable} vom Typ des Parameters {@code N} durch
	 *                den geteilt werden soll
	 * @return eine {@link Number} vom Typ des Parameters {@code S} aus
	 *         {@code this / divisor}
	 * @see java.math.BigInteger#divide(java.math.BigInteger)
	 */
	S divide(N divisor);

	/**
	 * Die Methode vergleicht ein anderes {@link Calculable}
	 * 
	 * @param other das andere {@link Calculable} vom selben Typ
	 * @return {@code true}, wenn {@code this > other}
	 * 
	 * @see javafx.beans.binding.IntegerBinding#greaterThan(double)
	 */
	boolean greaterThan(N other);

	/**
	 * Die Methode vergleicht ein anderes {@link Calculable}
	 * 
	 * @param other das andere {@link Calculable} vom selben Typ
	 * @return {@code true}, wenn {@code this < other}
	 * @see javafx.beans.binding.IntegerBinding#lessThan(double)
	 */
	boolean lessThan(N other);
	
	/**
	 * Die Methode vergleicht ein anderes {@link Calculable}
	 * 
	 * @param other das andere {@link Calculable} vom selben Typ
	 * @return {@code true}, wenn {@code this >= other}
	 * @see javafx.beans.binding.IntegerBinding#greaterThanOrEqualTo(long) 
	 */
	boolean greaterThanOrEqualTo(N other);
	
	/**
	 * Die Methode vergleicht ein anderes {@link Calculable}
	 * 
	 * @param other das andere {@link Calculable} vom selben Typ
	 * @return {@code true}, wenn {@code this < other}
	 * @see javafx.beans.binding.IntegerBinding#lessThanOrEqualTo(long)
	 */
	boolean lessThanOrEqualTo(N other);
}