
package eu.gronos.kostenrechner.logic;

import java.util.ArrayList;
import java.util.List;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;

/**
 * VollstreckbarkeitsHelfer.java eu.gronos.kostenrechner.logic (Kostenrechner)
 * hilft dabei, strukturierte Daten für die Vollstreckbarkeitsdaten für einen
 * {@link TenorDatenContainer} zu bauen, vor allem
 * {@link VollstreckbarkeitsEntscheidungsElemente#prozessverhaeltnisse}
 *
 * @author Peter Schuster (setrok)
 * @date 8 Dec 2018
 *
 */
public class VollstreckbarkeitsHelfer {

	private final List<HauptsacheVerhaeltnis> hauptsacheVerhaeltnisse;
	private final List<KostenTragungsVerhaeltnis> kostenTragungsVerhaeltnisse;

	/**
	 * Der Konstruktor baut auf den Listen aus
	 * {@link HauptsacheEntscheidungsElemente} und
	 * {@link KostenGrundEntscheidungsElemente} auf.
	 * 
	 * @param hauptsacheVerhaeltnisse     {@link HauptsacheEntscheidungsElemente#prozessverhaeltnisse}
	 * @param kostenTragungsVerhaeltnisse {@link KostenGrundEntscheidungsElemente#prozessverhaeltnisse}
	 */
	public VollstreckbarkeitsHelfer(List<HauptsacheVerhaeltnis> hauptsacheVerhaeltnisse,
			List<KostenTragungsVerhaeltnis> kostenTragungsVerhaeltnisse) {
		this.hauptsacheVerhaeltnisse = hauptsacheVerhaeltnisse;
		this.kostenTragungsVerhaeltnisse = kostenTragungsVerhaeltnisse;
	}

	/**
	 * Wenn es nur um einen Kläger und einen Beklagten geht, baut diese Methode mit
	 * Hilfe der {@link VollstreckbarkeitsListe} dazu die
	 * {@link VollstreckbarkeitsEntscheidungsElemente#prozessverhaeltnisse}
	 * 
	 * @param klaeger                 ein {@link Beteiligter}
	 * @param beklagter               ein {@link Beteiligter}
	 * @param vollstreckbarkeitsListe eine {@link VollstreckbarkeitsListe}, deren 0.
	 *                                Element die Unanfechtbarkeit speichert, wenn
	 *                                nicht unanfechtbar das 1. Element das
	 *                                Verhältnis kl -> bk und das 2. Element das
	 *                                Verhältnis bk -> kl.
	 * @return eine List&lt;VollstreckungsVerhaeltnis> wie in
	 *         {@link VollstreckbarkeitsEntscheidungsElemente#prozessverhaeltnisse},
	 */
	public List<VollstreckungsVerhaeltnis> toVollstreckbarkeitsVerhaeltnisse(Beteiligter klaeger, Beteiligter beklagter,
			VollstreckbarkeitsListe vollstreckbarkeitsListe) {
		if (vollstreckbarkeitsListe == null || vollstreckbarkeitsListe.isEmpty())
			throw new IllegalArgumentException("Die Vollstreckbarkeitsliste darf nicht null oder leer sein!");
		List<VollstreckungsVerhaeltnis> liste = new ArrayList<VollstreckungsVerhaeltnis>();
		// Wenn das Urteil unanfechtbar ist, hat die vollstreckbarkeitsListe nur ein
		// Element
		if (vollstreckbarkeitsListe.sindAlle708iVm713()) {
			return errechneAlleVollstreckungsVerhaeltnisse(vollstreckbarkeitsListe.get(0));
		}
		// Das nächste Element der vollstreckbarkeitsListe beschreibt das Verhältnis
		// Kl-->Bk
		final Vollstreckbarkeit voKl = vollstreckbarkeitsListe.get(1);
		if (voKl.hatEintraege()) {
			VollstreckungsVerhaeltnis vollstreckungsVerhaeltnis = new VollstreckungsVerhaeltnis(klaeger, beklagter,
					voKl);
			Kostenrechner.getLogger().info(vollstreckungsVerhaeltnis.toString());
			liste.add(vollstreckungsVerhaeltnis);
		}
		// Das nächste Element der vollstreckbarkeitsListe beschreibt das Verhältnis
		// Bk-->Kl
		final Vollstreckbarkeit voBk = vollstreckbarkeitsListe.get(2);
		if (voBk.hatEintraege()) {
			VollstreckungsVerhaeltnis vollstreckungsVerhaeltnis = new VollstreckungsVerhaeltnis(beklagter, klaeger,
					voBk);
			Kostenrechner.getLogger().info(vollstreckungsVerhaeltnis.toString());
			liste.add(vollstreckungsVerhaeltnis);
		}
		return liste;
	}

	/**
	 * Die Methode baut eine Liste für die
	 * {@link VollstreckbarkeitsEntscheidungsElemente} aus allen Verhältnissen, wo
	 * ein Gläubiger vollstrecken kann, für den Fall, dass die Vollstreckbare
	 * einheitlich zu beurteilen ist.
	 * 
	 * @param vollstreckbarkeit ein {@link Vollstreckbarkeit}
	 * 
	 * @return eine List&lt;VollstreckungsVerhaeltnis> wie in
	 *         {@link VollstreckbarkeitsEntscheidungsElemente#prozessverhaeltnisse},
	 *         bei denen jeweils das Feld
	 *         {@link VollstreckungsVerhaeltnis#getVollstreckbarkeit()} mit dem
	 *         Parameter vollstreckbarkeit definiert ist.
	 */
	public List<VollstreckungsVerhaeltnis> errechneAlleVollstreckungsVerhaeltnisse(
			Vollstreckbarkeit vollstreckbarkeit) {
		List<VollstreckungsVerhaeltnis> liste = new ArrayList<VollstreckungsVerhaeltnis>();
		// Alle "Paarungen" aus der Hauptsache hinzufügen
		if (hauptsacheVerhaeltnisse != null && !hauptsacheVerhaeltnisse.isEmpty()) {
			for (HauptsacheVerhaeltnis hauptsacheVerhaeltnis : hauptsacheVerhaeltnisse) {
				if (hauptsacheVerhaeltnis.getVerurteilung().greaterThan(Euro.ZERO_CENTS)) {// > 0L
					VollstreckungsVerhaeltnis vollstreckungsVerhaeltnis = new VollstreckungsVerhaeltnis(
							hauptsacheVerhaeltnis.getGlaeubiger(), hauptsacheVerhaeltnis.getSchuldner(),
							vollstreckbarkeit);
					vollstreckungsVerhaeltnis.setDarueberHinaus(hauptsacheVerhaeltnis.isDarueberHinaus());
					liste.add(vollstreckungsVerhaeltnis);
					Kostenrechner.getLogger().info(vollstreckungsVerhaeltnis.toString());
				}
			}
		}
		// Alle "Paarungen" aus den Kosten hinzufügen, sofern nicht schon drin
		if (kostenTragungsVerhaeltnisse != null && !kostenTragungsVerhaeltnisse.isEmpty()) {
			for (KostenTragungsVerhaeltnis kostenTragungsVerhaeltnis : kostenTragungsVerhaeltnisse) {
				if (kostenTragungsVerhaeltnis.getKostenBruchteil().greaterThan(Fraction.ZERO)
						// (kostenTragungsVerhaeltnis.getKostenBruchteil() > 0.0
						&& !enthaeltPaarung(liste, kostenTragungsVerhaeltnis.getGlaeubiger(),
								kostenTragungsVerhaeltnis.getSchuldner())) {
					VollstreckungsVerhaeltnis vollstreckungsVerhaeltnis = new VollstreckungsVerhaeltnis(
							kostenTragungsVerhaeltnis.getGlaeubiger(), kostenTragungsVerhaeltnis.getSchuldner(),
							vollstreckbarkeit);
					vollstreckungsVerhaeltnis.setDarueberHinaus(kostenTragungsVerhaeltnis.isDarueberHinaus());
					liste.add(vollstreckungsVerhaeltnis);
					Kostenrechner.getLogger().info(vollstreckungsVerhaeltnis.toString());
				}
			}
		}
		return liste;
	}

	/**
	 * Die Methode prüft, ob eine gegebene "Paarung" schon in der Liste existiert
	 * 
	 * @param vollstreckungsVerhaeltnisse die zu prüfende
	 *                                    {@link List}&lt;VollstreckungsVerhaeltnis>
	 * @param glaeubiger                  {@link Beteiligter}
	 * @param schuldner                   {@link Beteiligter}
	 * @return
	 */
	private boolean enthaeltPaarung(List<VollstreckungsVerhaeltnis> vollstreckungsVerhaeltnisse, Beteiligter glaeubiger,
			Beteiligter schuldner) {
		if (!vollstreckungsVerhaeltnisse.isEmpty()) {
			for (VollstreckungsVerhaeltnis vollstreckungsVerhaeltnis : vollstreckungsVerhaeltnisse) {
				if (vollstreckungsVerhaeltnis.getGlaeubiger().equals(glaeubiger)
						&& vollstreckungsVerhaeltnis.getSchuldner().equals(schuldner)) {
					return true;
				}
			}
		}
		return false;
	}

}
