/**
 * BaumbachFormel.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.KLAEGER;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.xml.bind.annotation.XmlID;
import javax.xml.bind.annotation.XmlIDREF;

import eu.gronos.kostenrechner.data.baumbach.Angriff;
import eu.gronos.kostenrechner.data.baumbach.Angriff.AngriffArt;
import eu.gronos.kostenrechner.data.baumbach.AngriffListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachVollstreckbarkeit;
import eu.gronos.kostenrechner.data.baumbach.GerichtsKostenBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.ImUebrigen;
import eu.gronos.kostenrechner.data.baumbach.MehrfachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.baumbach.StreitgenossenAngriffe;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.tenordaten.Allgemein;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.ProzessVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;
import eu.gronos.kostenrechner.interfaces.AntragErfolgElement;
import eu.gronos.kostenrechner.interfaces.ParteiBeziehung;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.util.VerlusteBank;
import eu.gronos.kostenrechner.util.baumbach.AngriffeFilter;
import eu.gronos.kostenrechner.util.baumbach.ProzessVerhaeltnisseInTenorReihenfolge;
import eu.gronos.kostenrechner.util.baumbach.StreitgenossenAngriffePruefer;
import eu.gronos.kostenrechner.util.baumbach.StreitgenossenPruefer;

/**
 * Der Tenor f&uuml;r <b>Teilunterliegen bei Streitgenossenschaft</b> kann nach
 * der Baumbach&prime;schen Kostenformel erzeugt werden. Die
 * {@link TenorVorbereitend}e Klasse greift schon mal die Berechnungsweise nach
 * Baumbach vor, indem sie eine oder mehrere {@link VerlusteBank}en errechnet
 * und {@link ProzessVerhaeltnis}se erstellt.
 * 
 * @author Peter Schuster (setrok)
 * @date 29.04.2014
 * 
 */
public class BaumbachFormel implements TenorVorbereitend /* , Begruendend */ {

	private final TenorDatenContainer container;
	private Map<Beteiligter, VerlusteBank> quoten;
	private Map<Beteiligter, VerlusteBank> bereinigteQuoten;
	private final BaumbachTexter texter;
	private final BegruendungsTexter begruendungsTexter;//
	private final StreitgenossenAngriffe streit;

	/**
	 * Konstruktor:
	 * 
	 * @param verfahrensDaten          ein {@link VerfahrensDatenContainer} mit
	 *                                 einem
	 *                                 {@link StreitgenossenAngriffe}-{@link UnterContainerKlasse},
	 *                                 einer
	 *                                 {@link VerfahrensDatenContainer#allgemein}-{@link UnterContainerKlasse},
	 * @param b92iiZpo                 Soll § 92 II ZPO angewandt werden
	 *                                 (verhältnismäßig geringe Kostentragungen
	 *                                 werden unterschlagen)
	 * 
	 * @param alsBruch                 {@code boolean}, ob Kostenquoten als Bruch
	 *                                 dargestellt werden. Sonst werden sie als
	 *                                 Prozent geschrieben. TODO müsste in
	 *                                 {@link Allgemein}
	 * 
	 * 
	 * @param streitgenossen           {@link VerfahrensDatenContainer#streitgenossen}
	 * @param baumbachBeteiligtenListe ein Array BaumbachBeteiligter[] in der
	 *                                 Reihenfolge KLAEGER, DRITTWIDERBEKLAGTE
	 *                                 (falls vorhanden) und BEKLAGTE, mit
	 *                                 mindestens einem KLAEGER und einem BEKLAGTEn.
	 * 
	 * @param wirtschaftlichIdentisch  ist eine etwaige Widerklage wirtschaftlich
	 *                                 identisch mit der Hauptklageforderung?
	 * 
	 * @throws IllegalArgumentException wird geworfen, wenn eine der Vorgaben nicht
	 *                                  eingehalten wurde.
	 */
	public BaumbachFormel(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
		// public BaumbachAngriffFormel(StreitgenossenAngriffe streit, Allgemein
		// allgemein,
//            boolean alsBruch) {

//		private StreitgenossenAngriffConverter converter;
//		private EntscheidungsListenElementeAbgleicher converter2;
		new StreitgenossenPruefer().pruefeEingabe(verfahrensDaten.streitgenossen);//TODO gute Idee?
		this.streit = new StreitgenossenAngriffConverter().apply(verfahrensDaten.streitgenossen);
		new StreitgenossenAngriffePruefer().pruefeEingabe(streit);
		this.container = new TenorDatenContainer(verfahrensDaten);
		this.container.erweiterteKlasse = this.streit;
		// new TenorDatenContainer();
//        this.streit = streit;
		// alsBruch = false;// TODO gehört in verfahrensDaten.allgemein
		// this.container.allgemein = allgemein;
		this.container.allgemein = verfahrensDaten.allgemein;
		this.texter = new BaumbachTexter(this.streit, container.allgemein.alsBruch);
		this.begruendungsTexter = new BegruendungsTexter(this.streit);
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return das Ergebnis von {@link #erzeugeHauptsacheEntscheidung()} +
	 *         {@link #erzeugeKostenEntscheidung()}+
	 *         {@link #erzeugeStreitwertEntscheidung()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = begruendungsTexter.begruendungsElemente(); // zeilen.toBegruendungsElemente(getGruende());
		container.allgemein.selectedPanel = Streitgenossen.TAB_ID;

		return container;
	}

	/**
	 * Die Methode erzeugt den Hauptsachetenor, davon ausgehend, dass die
	 * Verurteilungen auf Zahlung lauten.
	 * 
	 * @return den Hauptsachetenor als
	 *         {@link EntscheidungsListenElemente<HauptsacheVerhaeltnis>}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung() {
		EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
		hauptsache.prozessverhaeltnisse = baueHauptsacheVerhaeltnisse(this.streit.angriffe);
		final List<List<Angriff>> gruppen = AngriffListe.gruppiereAngriffe(this.streit.angriffe);
		texter.setGruppen(gruppen);
		List<ImUebrigen> biue = new ArrayList<>();
		for (List<Angriff> gruppe : gruppen) {
			ImUebrigen iue = baueImUebrigen(gruppe);
			biue.add(iue);
		}
		texter.setImUebrigen(biue);
		Collections.sort(hauptsache.prozessverhaeltnisse, new ProzessVerhaeltnisseInTenorReihenfolge());

		hauptsache.text = texter.texteHauptsache(hauptsache);

		new EntscheidungsListenElementeAbgleicher().apply(this.streit.pseudo, hauptsache);

		return hauptsache;

	}

	/**
	 * Gibt den ganzen Kostentenor.
	 * 
	 * @return den Kostentenor als {@link EntscheidungsListenElemente} von
	 *         {@link KostenTragungsVerhaeltnis}se
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung() {
		EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();

		quoten = errechneAlleVerluste(this.streit.angriffe);
		/*
		 * Erst einmal die Quoten untersuchen und unwesentliches Unterliegen anderweitig
		 * verteilen (wenn ausgewählt)
		 */
		if (container.allgemein.b92ii) {
			bereinigteQuoten = entferneUnwesentlichesUnterliegen(quoten, container.allgemein.vomHundert92ii);
		} else {
			bereinigteQuoten = quoten;
		}

		kostenEntscheidung.prozessverhaeltnisse = baueKostenTragungsVerhaeltnisse(quoten, bereinigteQuoten,
				this.streit.angriffe);
		Collections.sort(kostenEntscheidung.prozessverhaeltnisse, new ProzessVerhaeltnisseInTenorReihenfolge());

		texter.setBereinigteQuoten(bereinigteQuoten);
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);

		// muss auch die Gründe erzeugen
		begruendungsTexter.baueGruende(quoten.get(new GerichtsKostenBeteiligter()));

		new EntscheidungsListenElementeAbgleicher().apply(this.streit.pseudo, kostenEntscheidung);

		return kostenEntscheidung;

	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit.
	 * 
	 * @return {@link EntscheidungsListenElemente} für
	 *         {@link VollstreckungsVerhaeltnis}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeVollstreckbarkeitsEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung() {
		EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();
		final VollstreckbarkeitsListe vollstreckbarkeit = berechneVollstreckbarkeit(this.streit.angriffe,
				bereinigteQuoten, quoten);
		vollEntscheidung.prozessverhaeltnisse = baueVollstreckungsVerhaeltnisse(vollstreckbarkeit,
				container.hauptsacheEntscheidung.prozessverhaeltnisse,
				container.kostenEntscheidung.prozessverhaeltnisse);
		begruendungsTexter.erweitereGruende(vollstreckbarkeit.toString());
		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeit);
		return vollEntscheidung;
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung, indem der Streitwert auf den
	 * höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
	 * 
	 * @return {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente streitwert = new StreitwertEntscheidungsElemente();
		for (Euro wert : errechneStreitwert(this.streit.angriffe)) {
			streitwert.streitwerte.add(wert);// (Long.valueOf(wert.longValue()));
		}
		streitwert.text = texter.texteStreitwert(streitwert);
		return streitwert;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link EntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public EntscheidungsElemente erzeugeSonstigeEntscheidung() {
		EntscheidungsElemente sonstige = new EntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return "Tenor bei Streitgenossen nach der sog. Baumbach'schen Formel (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor sowie Streitwertbeschluss).";
	}

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
	 */
	@Override
	public String getGruende() {
		return begruendungsTexter.getGruende();
	}

	/**
	 * Die Methode berechnet alle Kostenquoten, die gerichtlichen und die
	 * außergerichtlichen aller Beteiligten, denn sie errechnet alle
	 * {@link VerlusteBank}en zu allen Angriffen aller beteiligten
	 * {@link Beteiligter}n.
	 * 
	 * @param angriffe eine {@link List} aus {@link Angriff}en
	 * @return eine {@link Map} aus {@link Beteiligter}, {@link VerlusteBank}
	 */
	private Map<Beteiligter, VerlusteBank> errechneAlleVerluste(List<Angriff> angriffe) {
		final Map<Beteiligter, VerlusteBank> alleVerluste = new HashMap<>();
		final GerichtsKostenBeteiligter gerichtsKosten = new GerichtsKostenBeteiligter();
		alleVerluste.put(gerichtsKosten, errechneVerluste(angriffe, gerichtsKosten));
		for (Beteiligter b : ((AngriffListe) angriffe).getBeteiligte()) {
			alleVerluste.put(b, errechneVerluste(angriffe, b));
		}
		return alleVerluste;
	}

	/**
	 * Die Methode errechnet die Verluste aller an einem bestimmten
	 * Prozessverhältnis Beteiligten und gibt sie als {@link VerlusteBank} zurück.
	 * 
	 * @param angriffe    eine {@link List} aus {@link Angriff}en
	 * @param beteiligter nur {@link Angriff}e berücksichtigen, an denen der
	 *                    {@link Beteiligter} beteiligt ist; einen
	 *                    {@link GerichtsKostenBeteiligter}n übergeben, wenn ohne
	 *                    Einschränkungen
	 * @return eine {@link VerlusteBank}
	 */
	private VerlusteBank errechneVerluste(List<Angriff> angriffe, Beteiligter beteiligter) {
		VerlusteBank verluste = new VerlusteBank();

		final AngriffeFilter tester = new AngriffeFilter(beteiligter);

		for (Angriff angriff : angriffe) {
			if (tester.test(angriff)) {
				// in manchen Fällen darf nicht multipliziert werden:
				// darf nur multiplizieren, wenn beteiligter der
				// GerichtsKostenBeteiligter oder
				// der beteiligter in der Liste der angreifer ist
				final double multi = (beteiligter instanceof GerichtsKostenBeteiligter
						|| angriff.getAngreifer().contains(beteiligter)) ? angriff.getGegner().size() : 1;
				final Euro angreiferUnterliegen = angreiferUnterliegen(angriff);// long
				verluste.add(angriff.getAngreifer(), angreiferUnterliegen.multiply(multi));
				// multi * angreiferUnterliegen
				if (gegnerUnterliegen(angriff).getCents() > 0L) {
					// Man darf die Kosten eines (Wider-) Beklagten nicht beiden
					// (Wider-) Beklagten aufgebrummen; das widerspricht Baumbach: die
					// müssen zum Konto der einzelnen (Wider-) Beklagten selbst.
					List<Beteiligter> gegner;
					if (angriff.getGegner().size() > 1 && angriff.getGegner().contains(beteiligter)) {
						gegner = new ArrayList<Beteiligter>();
						gegner.add(beteiligter);
					} else {
						gegner = angriff.getGegner();
					}
					verluste.add(gegner, gegnerUnterliegen(angriff).multiply(multi));
				}
			}
		}
		verluste.removeBlanks();
		return verluste;
	}

	/**
	 * 
	 * @param angriff ein {@link AntragErfolgElement}
	 * @return {@link Angriff#getErfolgCent()} als Unterliegen der/des
	 *         {@link Angriff#getGegner()}
	 */
	private Euro gegnerUnterliegen(AntragErfolgElement angriff) {// long
		return angriff.getErfolg();
	}

	/**
	 * Die Methode errechnet das Unterliegen des {@link Angriff#getAngreifer()}s
	 * 
	 * @param angriff ein {@link AntragErfolgElement}
	 * @return das Unterliegen des {@link Angriff#getAngreifer()}s als Differenz
	 *         seines {@link Angriff#getAntragCent()} und
	 *         {@link Angriff#getErfolgCent()}
	 */
	private Euro angreiferUnterliegen(AntragErfolgElement angriff) {
		return angriff.getAntrag().subtract(gegnerUnterliegen(angriff));
	}

	/**
	 * Die Methode listet alle Hauptsacheverhältnisse der Beteiligten untereinander
	 * auf
	 * 
	 * @param angriffe die ungruppierte {@link List} der {@link Angriff}e
	 * 
	 * @return eine {@link List} aus {@link HauptsacheVerhaeltnis}sen
	 */
	private List<HauptsacheVerhaeltnis> baueHauptsacheVerhaeltnisse(List<Angriff> angriffe) {
		List<HauptsacheVerhaeltnis> verhaeltnisse = new ArrayList<HauptsacheVerhaeltnis>();

		/*
		 * die Liste in "Tenorreihenfolge" durchgehen (Beklagte vor übrigen,
		 * Gesamtschuldner vor einfachen)
		 */
		for (Angriff angriff : angriffe) {

			if (angriff.getErfolg().getCents() > 0L) {
				HauptsacheVerhaeltnis verhaeltnis = baueHauptsacheVerhaeltnis(angriffe, angriff);
				verhaeltnisse.add(verhaeltnis);
			}
		}

		return verhaeltnisse;
	}

	/**
	 * Die Methode baut ein einzelnes {@link HauptsacheVerhaeltnis} aus einem
	 * {@link Angriff}
	 * 
	 * @param angriffe alle {@link Angriff}e als {@link List}
	 * @param angriff  der einzelne {@link Angriff}
	 * @return ein einzelnes {@link HauptsacheVerhaeltnis}
	 */
	private HauptsacheVerhaeltnis baueHauptsacheVerhaeltnis(List<Angriff> angriffe, Angriff angriff) {
		return new HauptsacheVerhaeltnis(angreiferAusAngriff(angriff), gegnerAusAngriff(angriff),
				(findeDarueberHinaus(angriffe, angriff).size() > 0), angriff.getErfolg());
	}

	/**
	 * Die Methode listet alle Kostentragungsverhältnisse der Beteiligten
	 * untereinander auf.
	 * 
	 * @param quoten    eine {@link Map} aus {@link Beteiligter} und
	 *                  {@link VerlusteBank}
	 * @param angriffe  eine {@link List} von {@link Angriff}en
	 * @param allgemein @see {@link Allgemein#vomHundert92ii}
	 * @return eine {@link List}&lt;{@link KostenTragungsVerhaeltnis}>
	 */
	private List<KostenTragungsVerhaeltnis> baueKostenTragungsVerhaeltnisse(Map<Beteiligter, VerlusteBank> quoten,
			Map<Beteiligter, VerlusteBank> bereinigteQuoten, List<Angriff> angriffe) {
		List<KostenTragungsVerhaeltnis> verhaeltnisse = new ArrayList<>();
		List<Beteiligter> beteiligte = new ArrayList<>(bereinigteQuoten.keySet().size());
		bereinigteQuoten.keySet().forEach(beteiligte::add);
		for (Beteiligter b : bereinigteQuoten.keySet()) {
			// die erste Zeile betrifft die Gerichtskosten
			if (b instanceof GerichtsKostenBeteiligter)
				beteiligte.add(0, b);
			else
				beteiligte.add(b);
		}

		for (Beteiligter glaeubiger : beteiligte) {
			final VerlusteBank verluste = bereinigteQuoten.get(glaeubiger);

			for (List<Beteiligter> kontoInhaber : verluste.keySet()) {
				// Kosten des Gläubigers, die er selbst tragen muss, sind nicht
				// vollstreckbar
				if (kontoInhaber.contains(glaeubiger))
					continue;
				KostenTragungsVerhaeltnis verhaeltnis = baueKostenTragungsVerhaeltnis(angriffe, verluste, kontoInhaber,
						glaeubiger);
				// Sonst gibt es eine komische Dopplung
				if (!verhaeltnisse.contains(verhaeltnis))
					verhaeltnisse.add(verhaeltnis);
			}
		}

		return verhaeltnisse;
	}

	/**
	 * Die Methode baut ein einzelnes {@link KostenTragungsVerhaeltnis}
	 * 
	 * @param angriffe     eine {@link List} von {@link Angriff}en
	 * @param verluste     die {@link VerlusteBank}
	 * @param kontoInhaber eine {@link List} aus {@link Beteiligter}n als
	 *                     {@code key} für die {@link VerlusteBank}
	 * @param glaeubiger   der {@link Beteiligter} für
	 *                     {@link KostenTragungsVerhaeltnis#getGlaeubiger()}
	 * 
	 * @return ein einzelnes {@link KostenTragungsVerhaeltnis}
	 */
	private KostenTragungsVerhaeltnis baueKostenTragungsVerhaeltnis(List<Angriff> angriffe, VerlusteBank verluste,
			List<Beteiligter> kontoInhaber, Beteiligter glaeubiger) {
		return new KostenTragungsVerhaeltnis(glaeubiger, schuldnerAusVerlustKonto(kontoInhaber, angriffe),
				// darüber hinaus? Über verluste herausfinden!
				findeDarueberHinaus(kontoInhaber, verluste).size() > 0, // verluste.getQuotaFor(kontoInhaber));
				verluste.getFractionFor(kontoInhaber));
	}

	/**
	 * Die Methode baut einen {@link HauptsacheVerhaeltnis#getGlaeubiger()}
	 * 
	 * @param angriff der einzelne {@link Angriff}
	 * @return ein {@link Beteiligter}, der
	 *         {@link HauptsacheVerhaeltnis#getGlaeubiger()} werden kann
	 */
	private Beteiligter angreiferAusAngriff(Angriff angriff) {
		return MehrfachBeteiligter.from(angriff.getAngreifer(), false);
	}

	/**
	 * Die Methode baut einen {@link ProzessVerhaeltnis#getSchuldner()}
	 * 
	 * @param angriff der einzelne {@link Angriff}
	 * @return ein {@link Beteiligter}, ggfs {@link MehrfachBeteiligter} der
	 *         {@link HauptsacheVerhaeltnis#getSchuldner()} werden kann
	 */
	private Beteiligter gegnerAusAngriff(Angriff angriff) {
		return MehrfachBeteiligter.from(angriff.getGegner(), angriff.isGesamtSchuldnerisch());
	}

	/**
	 * Die Methode baut einen {@link KostenTragungsVerhaeltnis#getSchuldner()}
	 * 
	 * @param kontoInhaber eine {@link List} aus {@link Beteiligter}n als
	 *                     {@code key} für die {@link VerlusteBank}
	 * @param angriffe     eine {@link List} von {@link Angriff}en
	 * @return einen {@link Beteiligter}, der
	 *         {@link ProzessVerhaeltnis#getSchuldner()} werden kann
	 */
	private Beteiligter schuldnerAusVerlustKonto(List<Beteiligter> kontoInhaber, List<Angriff> angriffe) {
		return MehrfachBeteiligter.from(kontoInhaber,
				// gesamtschuldnerisch? Über angriffe herausfinden!
				istGesamtschuldnerisch(kontoInhaber, angriffe));
	}

	/**
	 * Die Methode konvertiert die {@link VollstreckbarkeitsListe} in eine Liste der
	 * {@link VollstreckungsVerhaeltnis}se. Die Liste ist sortiert nach Maßgabe von
	 * {@link VollstreckungsVerhaeltnis#compareTo(eu.gronos.kostenrechner.data.tenordaten.ProzessVerhaeltnis)}.
	 * 
	 * @param vollstreckbarkeit       eine {@link VollstreckbarkeitsListe}
	 * @param hauptsacheVerhaeltnisse eine {@link List} aus
	 *                                {@link HauptsacheVerhaeltnis}sen
	 * @param kostenVerhaeltnisse     eine {@link List} aus
	 *                                {@link KostenTragungsVerhaeltnis}sen
	 * @return eine {@link List}&lt;{@link VollstreckungsVerhaeltnis}&gt;
	 */
	private List<VollstreckungsVerhaeltnis> baueVollstreckungsVerhaeltnisse(VollstreckbarkeitsListe vollstreckbarkeit,
			List<HauptsacheVerhaeltnis> hauptsacheVerhaeltnisse, List<KostenTragungsVerhaeltnis> kostenVerhaeltnisse) {
		if (vollstreckbarkeit != null && !vollstreckbarkeit.isEmpty() && vollstreckbarkeit.sindAlle708iVm713()) {
			return new VollstreckbarkeitsHelfer(hauptsacheVerhaeltnisse, kostenVerhaeltnisse)
					.errechneAlleVollstreckungsVerhaeltnisse(vollstreckbarkeit.get(0));
		}
		List<VollstreckungsVerhaeltnis> verhaeltnisse = new ArrayList<VollstreckungsVerhaeltnis>();
		for (Vollstreckbarkeit vo : vollstreckbarkeit) {
			if (vo instanceof BaumbachVollstreckbarkeit) {
				BaumbachVollstreckbarkeit bvo = (BaumbachVollstreckbarkeit) vo;
				/* Das Gericht kann nicht vollstrecken */
				if (bvo.getGlaeubiger() instanceof GerichtsKostenBeteiligter)
					continue;
				/*
				 * Vor dem Hinzufügen müssen "leere" VollstreckbarkeitsObjekte und das Tragen
				 * eigener Kosten herausgefiltert werden
				 */
				if (bvo.hatEintraege() && !bvo.getGlaeubiger().equals(bvo.getSchuldner())) {
					VollstreckungsVerhaeltnis verhaeltnis = new VollstreckungsVerhaeltnis(bvo);
					verhaeltnisse.add(verhaeltnis);
				}
			}
		}
		Collections.sort(verhaeltnisse);
		return verhaeltnisse;
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * 
	 * Vorläufige Vollstreckbarkeit: erst einmal gucken, ob die Hauptsache über die
	 * Schwelle von §709 kommt.
	 * 
	 * Wenn nicht, dann bei den Kosten rechnen: Array double[][] durchgehen, Quote
	 * mit jeweiligen Kosten multiplizieren und für jede Partei zusammen rechnen.
	 * Die jeweils eigenen Kosten dabei auslassen.
	 * 
	 * TODO stattdessen Kostentragungsverhältnisse durchgehen?
	 * 
	 * @param angriffe         eine {@link List} von {@link Angriff}en
	 * @param bereinigteQuoten eine {@link Map} von {@link List} aus
	 *                         {@link Beteiligter}n zu {@link VerlusteBank}, darf
	 *                         bereinigt sein
	 * @param quoten           um die Unanfechtbarkeit zu prüfen, braucht die
	 *                         Methode die {@link VerlusteBank} mit den (nicht
	 *                         bereinigten) GerichtskostenQuoten
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, die weiteren Elemente speichern die
	 *         vorläufige Vollstreckbarkeit für die
	 *         Einzelvollstreckungsverhältnisse.
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit(List<Angriff> angriffe,
			Map<Beteiligter, VerlusteBank> bereinigteQuoten, Map<Beteiligter, VerlusteBank> quoten) {
		final VollstreckbarkeitsListe vollstreckbarkeit = new VollstreckbarkeitsListe();
		/* Unanfechtbarkeit prüfen und der Liste hinzufügen */
		vollstreckbarkeit.add(pruefeUnanfechtbarkeit(quoten.get(new GerichtsKostenBeteiligter())));
		/* Nach festgestellter Unanfechtbarkeit war's das schon. */
		if (vollstreckbarkeit.sindAlle708iVm713())
			return vollstreckbarkeit;
		/*
		 * Einzelne Vollstreckungsverhältnisse durchgehen und
		 * BaumbachVollstreckbarkeitsObjekt der Liste hinzufügen; an bereinigteQuoten
		 * entlanghangeln.
		 */
		final List<Euro> streitwerte = errechneStreitwert(angriffe);
		// Damit werden später die Kosten berechnet
		final Euro streitwert = streitwerte.get(0);// long (long)
		glaeubigerSchleife: // in der unbereinigten Map sind alle Gläubiger versammelt
		for (Beteiligter glaeubiger : quoten.keySet()) {
			if (glaeubiger == null)
				continue glaeubigerSchleife;
			final VerlusteBank glaeubigerQuoten = bereinigteQuoten.get(glaeubiger);
			schuldnerSchleife:
			// alleBeteiligteAus liefert alle potenziellen Schuldner
			for (Beteiligter schuldner : this.streit.beteiligte) {
				if (schuldner == null || schuldner.equals(glaeubiger))
					continue schuldnerSchleife;
				Euro hauptsache = Euro.ZERO_CENTS;
				Euro vollstreckbareKosten = Euro.ZERO_CENTS;
				angriffSchleife: // Für diese Paarung alle Angriffe durchgehen
				for (Angriff angriff : angriffe) {
					if (angriff == null)
						continue angriffSchleife;
					// Die Hauptsache steht im Angriff selbst
					if (angriff.getAngreifer().contains(glaeubiger) && angriff.getGegner().contains(schuldner)) {
						hauptsache = hauptsache.add(angriff.getErfolg());
					}
				}
				// Kosten aus (bereinigter) Map mit VerlusteBanken
				if (glaeubiger.getTyp() == KLAEGER) {
					/*
					 * Bei einem Kläger vorher die Gerichtskosten drauf KostenVerzeichnis.KV1210
					 */
//                    final Euro ofEuros = Euro.ofEuros(berechneGanzeQuote(schuldner,
//                            bereinigteQuoten
//                                    .get(new GerichtsKostenBeteiligter()))
//                            * GebuehrenAuflistung.errechneGebuehr(
//                                    TenorTexter.getKostenKlaeger().get(0),
//                                    streitwert));
					vollstreckbareKosten = vollstreckbareKosten.add(// //Euro.ofEuros(
							GebuehrenAuflistung.errechneGebuehr(TenorTexter.getKostenKlaeger().get(0), streitwert)// )
									.multiply(berechneGanzeQuote(schuldner,
											bereinigteQuoten.get(new GerichtsKostenBeteiligter()))));
				}
				/*
				 * Die Anwaltskosten (KOSTEN_BEKLAGTER) immer - auch bei Kl und Dwbk
				 */
//                vollstreckbareKosten = vollstreckbareKosten.add(
//                        Euro.ofEuros(berechneGanzeQuote(schuldner, glaeubigerQuoten)
//                                * (TenorTexter.getKostenBeklagter()
//                                        .errechneGebuehrenSumme(streitwert))));
				vollstreckbareKosten = vollstreckbareKosten // Euro.ofEuros()
						.add(TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwert)//
								.multiply(berechneGanzeQuote(schuldner, glaeubigerQuoten)));

				vollstreckbarkeit.add(BaumbachVollstreckbarkeit.pruefeSicherheitsleistung(//
						hauptsache.longValue(), vollstreckbareKosten.longValue(), glaeubiger, schuldner));
			}
		}
		return vollstreckbarkeit;
	}

	/**
	 * Die Methode dient dazu, die gesamte mögliche Quote zurückzugeben. Es wird die
	 * gesamte Quote zurückgegeben, die sich auch aus den Gesamtschuldnern ergibt,
	 * die auch den referenzierten Beteiligten repräsentieren.
	 * 
	 * @param schuldner der {@link Beteiligter}
	 * @param bank      eine {@link VerlusteBank} mit den Quoten zugunsten eines
	 *                  Gläubigers
	 * @return die Gesamtkostentragungslast als Dezimalbruch zwischen 0.0 und 1.0
	 *         als <code>double</code>
	 */
	private Fraction berechneGanzeQuote(Beteiligter schuldner, VerlusteBank bank) {
		if (schuldner == null || bank == null || (bank.getFiktiverStreitwert() == null)
				|| bank.getFiktiverStreitwert().getCents() == 0L) {
			return Fraction.ZERO;
		}
		Euro gesamt = Euro.ZERO_CENTS;
		for (List<Beteiligter> beteiligte : bank.keySet())
			if (beteiligte.contains(schuldner)) {
				gesamt = gesamt.add(bank.get(beteiligte));
			}
		return gesamt.divideAsFraction(bank.getFiktiverStreitwert());// divide
	}

	/**
	 * Die Methode findet in den Angriffen für die Gegner des Angriffs alle weiteren
	 * (zumindest teilweise erfolgreichen) Angriffe, in denen er vertreten ist.
	 * 
	 * @param angriffe die Gesamtliste der {@link Angriff}e als {@link List}
	 * @param angriff  den {@link Angriff}, auf dessen Gegner es ankommt
	 * @return eine {@link List} aller {@link Angriff}e, in denen der
	 *         {@link Angriff#getGegner()} noch Gegner ist.
	 */
	private List<Angriff> findeDarueberHinaus(List<Angriff> angriffe, ParteiBeziehung<List<Beteiligter>> angriff) {
		List<Angriff> liste = new ArrayList<>();
		if (angriffe == null || angriff == null || angriffe.size() < 1) {
			// do nothing, jump to return statement
		} else {
			for (Angriff a : angriffe) {
				// "eigenen" Angriff überspringen
				if (a.equals(angriff))
					continue;
				if (a.getAngreifer().equals(angriff.getAngreifer()) && a.getGegner().containsAll(angriff.getGegner())
				// nur darüber hinaus, wenn auch ein Erfolg da ist
						&& a.getErfolg().getCents() > 0L) {
					liste.add(a);
				}
			}
		}
		return liste;
	}

	/**
	 * Die Methode prüft, ob die Klage/Widerklage "im Übrigen" abgewiesen werden
	 * muss.
	 * 
	 * @return ein {@link ImUebrigen}
	 */
	private ImUebrigen baueImUebrigen(List<Angriff> gruppe) {
		/*
		 * Die Initialisierung geht davon aus, dass gar nicht verurteilt wurde, solange
		 * keine Verurteilung festgestellt ist
		 */
		ImUebrigen imUebrigen = new ImUebrigen(false, true, AngriffListe.klageOderWiderklage(gruppe));
		/*
		 * Wenn die Summe der Streitwerte so hoch ist wie die Summe des Unterliegens,
		 * dann hat der Angreifer voll obsiegt.
		 */
		final Euro gesamtErfolg = AngriffListe.addiereErfolg(gruppe);
		imUebrigen.setVollBegruendet(AngriffListe.addiereAntrag(gruppe).getCents() <= gesamtErfolg.getCents());
		/*
		 * Sobald eine Verurteilung vorliegt, wurde die Klage jedenfalls nicht mehr ganz
		 * abgewiesen.
		 */
		imUebrigen.setGanzUnbegruendet(gesamtErfolg.getCents() <= 0L);
		return imUebrigen;
	}

	/**
	 * Erst einmal die Quoten untersuchen und unwesentliches Unterliegen anderweitig
	 * verteilen (wenn ausgewählt) für die ganze {@link Map} aus {@link Beteiligter}
	 * und {@link VerlusteBank}.
	 * 
	 * Wenn eine der Quoten unter 10% (bzw. einem anderen Schwellenwert) ist, muss
	 * sie den verbleibenden sinnvoll "aufgeschlagen" werden:
	 * 
	 * <blockquote> § 92 Abs. 2 ZPO: Das Gericht kann der einen Partei die gesamten
	 * Prozesskosten auferlegen, wenn 1. die Zuvielforderung der anderen Partei
	 * verhältnismäßig geringfügig war und keine oder nur geringfügig höhere Kosten
	 * veranlasst hat oder 2. der Betrag der Forderung der anderen Partei von der
	 * Festsetzung durch richterliches Ermessen, von der Ermittlung durch
	 * Sachverständige oder von einer gegenseitigen Berechnung abhängig
	 * war.</blockquote>
	 * 
	 * Vorsicht! Für Begründungen usw. lässt sich die bereinigte
	 * {@link VerlusteBank} nicht nutzen, weil die Einzelverluste nicht stimmen!
	 * 
	 * @param quoten         eine {@link Map} aus {@link Beteiligter} und
	 *                       {@link VerlusteBank}
	 * @param vomHundert92ii der Schwellenwert aus @see
	 *                       {@link Allgemein#vomHundert92ii} als {@code int}
	 * @return eine neue {@link Map} aus {@link Beteiligter} und
	 *         {@link VerlusteBank}, in denen nur Verluste enthalten sind, die mehr
	 *         als unwesentlich sind
	 */
	private Map<Beteiligter, VerlusteBank> entferneUnwesentlichesUnterliegen(Map<Beteiligter, VerlusteBank> quoten,
			int vomHundert92ii) {
		Map<Beteiligter, VerlusteBank> bereinigteQuoten = new HashMap<>();
		for (Beteiligter key : quoten.keySet()) {
			bereinigteQuoten.put(key, entferneUnwesentlichesUnterliegen(quoten.get(key), vomHundert92ii));
		}
		return bereinigteQuoten;
	}

	/**
	 * Wenn eine der Quoten unter 10% (bzw. einem anderen Schwellenwert) ist, muss
	 * sie den verbleibenden sinnvoll "aufgeschlagen" werden:
	 * 
	 * <blockquote> § 92 Abs. 2 ZPO: Das Gericht kann der einen Partei die gesamten
	 * Prozesskosten auferlegen, wenn 1. die Zuvielforderung der anderen Partei
	 * verhältnismäßig geringfügig war und keine oder nur geringfügig höhere Kosten
	 * veranlasst hat oder 2. der Betrag der Forderung der anderen Partei von der
	 * Festsetzung durch richterliches Ermessen, von der Ermittlung durch
	 * Sachverständige oder von einer gegenseitigen Berechnung abhängig
	 * war.</blockquote>
	 * 
	 * Vorsicht! Für Begründungen usw. lässt sich die bereinigte
	 * {@link VerlusteBank} nicht nutzen, weil die Einzelverluste nicht stimmen!
	 * 
	 * @param quoten         eine {@link VerlusteBank} mit den quoten im Urzustand
	 * @param vomHundert92ii das {@link Allgemein#vomHundert92ii} als
	 *                       <code>int</code>
	 * @return eine {@link VerlusteBank} mit den quoten unter Berücksichtigung von
	 *         § 92 Abs. 2 ZPO
	 */
	private VerlusteBank entferneUnwesentlichesUnterliegen(VerlusteBank quoten, int vomHundert92ii) {
		VerlusteBank bereinigteQuoten = new VerlusteBank();
		// Erster Schritt: alle Quoten durchgehen
		for (List<Beteiligter> beteiligte : quoten.keySet()) {
			if ((100 * quoten.getQuotaFor(beteiligte)) < vomHundert92ii) {
				// ... und Einträge unter Schwellenwert ("verhältnismäßig gering")
				// nicht übernehmen
			} else {
				// ... andere hingegen von VerlusteBank aufsummieren lassen
				bereinigteQuoten.add(beteiligte, quoten.get(beteiligte));
			}
		}
		/*
		 * Wenn es keine Einzelquoten über dem Schwellenwert gab (dann ist
		 * getFiktiverStreitwertCent noch 0), dann die VerlusteBank unversehrt
		 * zurückgeben; ebenso, wenn der fiktive Streitwert an den alten heranreicht
		 */
		if (bereinigteQuoten.getFiktiverStreitwert().getCents() <= 0L
				|| quoten.getFiktiverStreitwert().equals(bereinigteQuoten.getFiktiverStreitwert()))
			return quoten;
		else
			return bereinigteQuoten;
	}

	/**
	 * Die Methode findet anhand der übergebenen {@link List} von {@link Angriff}en
	 * heraus, ob die {@link List} von {@link Beteiligter}n eine Gesamtschuld
	 * darstellt
	 * 
	 * @param beteiligte die {@link List} von {@link Beteiligter}n
	 * @param angriffe   {@link List} von {@link Angriff}en
	 * @return <code>true</code>, wenn Gesamtschuld
	 */
	private boolean istGesamtschuldnerisch(List<Beteiligter> beteiligte, List<Angriff> angriffe) {
		// bei einer leeren Liste oder nur einem Beteiligten kann es keine
		// Gesamtschuldnerschaft sein
		if (beteiligte == null || beteiligte.size() < 2)
			return false;
		// sonst den passenden Angriff heraussuchen und befragen
		for (Angriff angriff : angriffe) {
			if (angriff.getGegner().equals(beteiligte)) {
				return angriff.isGesamtSchuldnerisch();
			}
		}
		return false;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @param quoten die (nicht bereinigte!) VerlusteBank der Gerichtskosten
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit(VerlusteBank quoten) {
		/*
		 * müsste recht einfach mit der (nicht bereinigten!) VerlusteBank der
		 * Gerichtskosten gehen
		 */
		for (List<Beteiligter> key : quoten.keySet()) {
			Euro verluste = quoten.get(key);
			/*
			 * Sobald irgend etwas höher als die Berufungsgrenze ist, ist das Urteil
			 * anfechtbar.
			 */
			// ((verluste.intValue()) > TenorTexter.BERUFUNGS_GRENZE)// verluste / 100
			if (verluste.greaterThan(TenorTexter.BERUFUNGS_GRENZE))
				return new Vollstreckbarkeit();
			/*
			 * Nochmal prüfen, ob es Schuldnermehrheiten mit zusätzlichem Verlust gibt!
			 */
			for (Euro verlust : findeDarueberHinaus(key, quoten)) {
				verluste = verluste.add(verlust);
			}
			// if ((verluste.intValue()) > TenorTexter.BERUFUNGS_GRENZE)
			if (verluste.greaterThan(TenorTexter.BERUFUNGS_GRENZE))
				return new Vollstreckbarkeit();
		}
		/*
		 * Wenn wir hier angekommen sind (also aus der Schleife vorher nicht
		 * ausgestiegen sind), ist das Urteil unanfechtbar
		 */
		return new Vollstreckbarkeit(true, false, false, false, true);
	}

	/**
	 * Die Methode findet alle Verluste, in denen die in der {@link List}
	 * vorhandenen beteiligte vertreten sind, dann gucken, ob sie > 0% tragen müssen
	 * 
	 * @param beteiligte die fraglichen Schuldner in der {@link VerlusteBank}
	 * @param verluste   die {@link VerlusteBank} mit den Verlusten aller
	 *                   Beteiligten
	 * @return eine {@link List} mit allen Quoten, die die Beteiligten sonst noch
	 *         tragen müssen, sofern Verlust > 0
	 */
	private List<Euro> findeDarueberHinaus(final List<Beteiligter> beteiligte, final VerlusteBank verluste) {
		List<Euro> liste = new ArrayList<>();
		if (beteiligte == null || verluste == null || beteiligte.size() < 1) {
			// do nothing, jump to return statement
		} else {

			liste.addAll(//
					verluste.keySet().stream()
							// "eigenes" VerlustKonto überspringen
							.filter(verlustKonto -> !verlustKonto.equals(beteiligte))
							// sonstige Verlustquote durchwinken
							.filter(verlustKonto -> verlustKonto.containsAll(beteiligte))
							// sofern nicht leer
							.filter(verlustKonto -> verluste.get(verlustKonto).greaterThan(Euro.ZERO_CENTS))
							// in Euro umwandeln
							.map(verluste::get)
							// und zur Liste hinzufügen
							.collect(Collectors.toList())//
			);
		}
		return liste;
	}

	/**
	 * Die Methode dient dazu, den Streitwert zu ermitteln, indem der Streitwert auf
	 * den höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
	 * 
	 * @param angriffe eine (ungruppierte) {@link List} aus {@link Angriff}en
	 * 
	 * @return {@link List} aus {@link Long}: wenn eine Widerklage enthalten ist,
	 *         dann eine mit drei Elementen, bei dem {@link List#get(int)}(0) der
	 *         Streitwert insgesamt, Element 1 der Klagestreitwert und Element 2 der
	 *         Widerklagestreitwert sind; sonst mit nur einem ELement, nämlich nur
	 *         mit dem Streitwert.
	 */ // List<Long>
	private List<Euro> errechneStreitwert(List<Angriff> angriffe) {
		final List<List<Angriff>> gruppen = AngriffListe.gruppiereAngriffe(angriffe);
		// List<Long>
		List<Euro> streitwerte = new ArrayList<>(gruppen.size());
		Euro hoechsteKlage = Euro.ZERO_CENTS, hoechsteWiderklage = Euro.ZERO_CENTS;
		for (List<Angriff> gruppe : gruppen) {
			final Euro wert = hoechterWert(gruppe);
			if (AngriffListe.klageOderWiderklage(gruppe) == AngriffArt.WIDERKLAGE) {
				hoechsteWiderklage = wert.max(hoechsteWiderklage);
			} else {
				hoechsteKlage = wert.max(hoechsteKlage);
			}
		}
		/* den höchten Klagewert braucht man immer */
		streitwerte.add(hoechsteKlage);// (Long.valueOf(hoechsteKlage.longValue()));
		/*
		 * wenn mit Widerklage, dann drei Elemente, bei dem 0 der Streitwert insgesamt
		 * (davor schieben), 1 der Klagestreitwert (s.o.) und 2 der Widerklagestreitwert
		 * sind;
		 */
		if (hoechsteWiderklage.getCents() > 0L) {
			/*
			 * Wenn wirtschaftlichIdentisch, dann den höchsten beider Werte (Long#max),
			 * sonst die Summe beider Werte (Euro#add)
			 */
			streitwerte.add(0, //
					this.streit.p45i3GKG//
							? Euro.max(hoechsteKlage, hoechsteWiderklage)
							: hoechsteKlage.add(hoechsteWiderklage)
//							? Long.max(hoechsteKlage.longValue(), hoechsteWiderklage.longValue())//
//							: Long.valueOf(hoechsteKlage.add(hoechsteWiderklage).longValue())//
			);
			streitwerte.add(hoechsteWiderklage);// (((Long.valueOf(hoechsteWiderklage.longValue()));
		}

		return streitwerte;
	}

	/**
	 * Die Methode findet den höchsten {@link Angriff#getAntragCent()} (Streitwert)
	 * einer gruppierten {@link List} von {@link Angriff}en
	 * 
	 * @param gruppe gruppierte {@link List} von {@link Angriff}en
	 * @return den höchsten {@link Angriff#getAntrag()} als {@link Euro}
	 */
	private Euro hoechterWert(List<Angriff> gruppe) {
		Euro hoechterWert = Euro.ZERO_CENTS;
		for (Angriff angriff : gruppe) {
			final Euro wert = angriff.getAntrag();
			hoechterWert = wert.max(hoechterWert);
		}
		return hoechterWert;
	}

	/**
	 * Die {@link Function} baut eine einzelne {@link BaumbachBeteiligtenListe} aus
	 * den dreien einer {@link Streitgenossen}-{@link UnterContainerKlasse} um
	 * 
	 */
	static class StreitgenossenAngriffConverter implements Function<Streitgenossen, StreitgenossenAngriffe> {

		/**
		 * Die Methode wandelt eine {@link Streitgenossen}-{@link UnterContainerKlasse}
		 * in {@link StreitgenossenAngriffe} um.
		 * 
		 * @param streitgenossenContainer eine
		 *                                {@link Streitgenossen}-{@link UnterContainerKlasse}
		 * 
		 * @return eine {@link StreitgenossenAngriffe}-{@link UnterContainerKlasse}
		 */
		@Override
		public StreitgenossenAngriffe apply(Streitgenossen streitgenossenContainer) {
			final List<Angriff> angriffe = new AngriffListe();
			angriffe.addAll(AngriffListe.from(baueListe(streitgenossenContainer)));// baumbachFormel.
			Collections.sort(angriffe);

			final StreitgenossenAngriffe angriffContainer = new StreitgenossenAngriffe();
			angriffContainer.angriffe = angriffe;
			angriffContainer.beteiligte = AngriffListe.alleBeteiligteAus(angriffe);
			angriffContainer.p45i3GKG = streitgenossenContainer.wirtschaftlichIdentisch;

			return angriffContainer;
		}

		/**
		 * Die Methode baut eine einzelne {@link BaumbachBeteiligtenListe} aus den
		 * dreien einer {@link Streitgenossen}-{@link UnterContainerKlasse} um
		 * 
		 * @param streitgenossenContainer eine
		 *                                {@link Streitgenossen}-{@link UnterContainerKlasse}
		 * @return eine einzelne {@link BaumbachBeteiligtenListe}
		 */
		private BaumbachBeteiligtenListe baueListe(Streitgenossen streitgenossenContainer) {
			BaumbachBeteiligtenListe baumbachBeteiligtenListe = new BaumbachBeteiligtenListe();
			baumbachBeteiligtenListe.addAll(streitgenossenContainer.klaeger);
			/* Drittwiderbeklagte nur hinzufügen, wenn vorhanden */
			if (streitgenossenContainer.drittwiderbeklagte != null
					&& streitgenossenContainer.drittwiderbeklagte.size() > 0)
				baumbachBeteiligtenListe.addAll(streitgenossenContainer.drittwiderbeklagte);
			baumbachBeteiligtenListe.addAll(streitgenossenContainer.beklagte);
			return baumbachBeteiligtenListe;
		}

	}

	private static class EntscheidungsListenElementeAbgleicher implements
			BiFunction<List<Beteiligter>, EntscheidungsListenElemente<? extends ProzessVerhaeltnis>, List<Beteiligter>> {

		@Override
//		public List<Beteiligter> apply(List<Beteiligter> t,
//				EntscheidungsListenElemente<? extends ProzessVerhaeltnis> u) {
//			// TODO Auto-generated method stub
//			return null;
//		}

		/**
		 * Die Methode achtet darauf, dass nicht verschiedene
		 * Mehrfach-/Gerichtskosten-Beteiligte in {@link StreitgenossenAngriffe#pseudo}
		 * bzw {@link EntscheidungsListenElemente#prozessverhaeltnisse} kommen, damit
		 * {@link XmlID} und {@link XmlIDREF} funktionieren.
		 * 
		 * @param pseudo
		 * @param entscheidungsListenElemente
		 * @return
		 */ // (List<Beteiligter> pseudo, EntscheidungsListenElemente<? extends
			// ProzessVerhaeltnis>)
		public List<Beteiligter> apply(List<Beteiligter> pseudo,
				EntscheidungsListenElemente<? extends ProzessVerhaeltnis> entscheidungsListenElemente) {
			for (ProzessVerhaeltnis v : entscheidungsListenElemente.prozessverhaeltnisse) {

				pseudoAbgleichen(pseudo, v.getGlaeubiger(), b -> v.setGlaeubiger(b));
				pseudoAbgleichen(pseudo, v.getSchuldner(), b -> v.setSchuldner(b));
			}

			return pseudo;
		}

		/**
		 * @param pseudo
		 * @param beteiligter
		 * @param setter
		 */
		private void pseudoAbgleichen(List<Beteiligter> pseudo, Beteiligter beteiligter, Consumer<Beteiligter> setter) {
			// Abfrage: nur MehrfachBeteiligter und Gerichtskostenbeteiligter!
			if (!(beteiligter instanceof MehrfachBeteiligter) && !(beteiligter instanceof GerichtsKostenBeteiligter))
				return;

			// es muss immer derselbe (Mehrfach-/Gerichtskosten-)Beteiligte in das
			// Prozessverhältnis!
			if (pseudo.contains(beteiligter)) {
				setter.accept(pseudo.get(pseudo.indexOf(beteiligter)));
			} else {
				pseudo.add(beteiligter);
			}

		}

	}
//
//	private StreitgenossenAngriffConverter converter;
//	private EntscheidungsListenElementeAbgleicher converter2;

//	private BaumbachBeteiligtenListe baueListe(Streitgenossen streitgenossenContainer) {
//		return converter.baueListe(streitgenossenContainer);
//	}

//
//	private static final String BESCHREIBUNG = "Tenor bei Streitgenossen nach der sog. Baumbach'schen Formel (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor sowie Streitwertbeschluss).";
//	private static final String GRUENDE_91_92_100II = "Die Kostenentscheidung beruht auf §§ 91, 92, 100 Abs. 2 ZPO und der sogenannten Baumbach'schen Kostenformel,"
//			+ " entsprechend der unterschiedlichen Beteiligung am Rechtsstreit: \n";
//	private static final String KOPFZEILE_BEGINN = "Einzelangriffe / Verluste";
//	private static final String ERGEBNIS_ZEILE = "Kostenquote hinsichtlich eines fiktiven Streitwerts von %,.2f EUR";
//	private static final String VERLUST_GESAMT = "Verlust gesamt";
//	final BaumbachBeteiligtenListe baumbachBeteiligtenListe;
//	private boolean wirtschaftlichIdentisch;
//	private double[][] kosten = null;
//	private final StringBuilder gruende = new StringBuilder();
//	private TabulierendZeilen zeilen;
//	private VollstreckbarkeitsListe vollstreckbarkeit;
//	private final TenorDatenContainer container;
//	private final BaumbachTexter texter;
//	private Pruefend<Streitgenossen> pruefer = new StreitgenossenPruefer();
//	/**
//	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
//	 * und {@link Allgemein#unwesentlich92ii.}
//	 */
//	private Allgemein allgemein;
//
//	/**
//	 */
//	public BaumbachFormel(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
//		super();
//		/*
//		 * aus den drei Listen im Parameterobjekt eine BaumbachBeteiligtenListe bauen,
//		 * in der Reihenfolge Kläger, Drittwiderbeklagte, Beklagte
//		 */
//		Streitgenossen streitgenossen = verfahrensDaten.streitgenossen;
//		pruefer.pruefeEingabe(streitgenossen);
//		baumbachBeteiligtenListe = new BaumbachBeteiligtenListe();
//		baumbachBeteiligtenListe.addAll(streitgenossen.klaeger);
//		/* Drittwiderbeklagte nur hinzufügen, wenn vorhanden */
//		if (streitgenossen.drittwiderbeklagte != null && streitgenossen.drittwiderbeklagte.size() > 0)
//			baumbachBeteiligtenListe.addAll(streitgenossen.drittwiderbeklagte);
//		baumbachBeteiligtenListe.addAll(streitgenossen.beklagte);
//		// this.b92ii = .b92ii;
//		this.allgemein = verfahrensDaten.allgemein;
//		this.wirtschaftlichIdentisch = streitgenossen.wirtschaftlichIdentisch;
//		this.container = new TenorDatenContainer(verfahrensDaten);
//		this.texter = new BaumbachTexter(baumbachBeteiligtenListe);
//	}
//
//	/**
//	 * Die Methode erzeugt den Hauptsachetenor, davon ausgehend, dass die
//	 * Verurteilungen auf Zahlung lauten.
//	 * 
//	 * @return den Hauptsachetenor als
//	 *         {@link EntscheidungsListenElemente<HauptsacheVerhaeltnis>}, vor allem
//	 *         den Tenor als String mit abschließendem Zeilenumbruch
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
//	 */
//	@Override
//	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung() {
//		EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
//		hauptsache.prozessverhaeltnisse = baueHauptsacheVerhaeltnisse();
//		texter.setImUebrigen(baueImUebrigen());
//		hauptsache.text = texter.texteHauptsache(hauptsache);
//		return hauptsache;
//	}
//
//	/**
//	 * Gibt den ganzen Kostentenor.
//	 * 
//	 * @return den Kostentenor als {@link KostenGrundEntscheidungsElemente}, vor
//	 *         allem den Text des Kostentenors mit abschließendem Zeilenumbruch
//	 * 
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
//	 */
//	@Override
//	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung() {
//		EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();
//		final double[][] quoten = getKosten();
//		kostenEntscheidung.prozessverhaeltnisse = baueKostenTragungsVerhaeltnisse(quoten);
//
//		texter.setQuoten(quoten);
//		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);
//
//		return kostenEntscheidung;
//	}
//
//	/**
//	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit.
//	 * 
//	 * @return einen {@link VollstreckbarkeitsEntscheidungsElemente}
//	 */
//	@Override
//	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung() {
//		EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();
//		if (vollstreckbarkeit == null)
//			vollstreckbarkeit = berechneVollstreckbarkeit();
//		erweitereGruende(vollstreckbarkeit.toString());
//		vollEntscheidung.prozessverhaeltnisse = baueVollstreckungsVerhaeltnisse(vollstreckbarkeit);
//		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeit);
//		return vollEntscheidung;
//	}
//
//	/**
//	 * Die Methode erzeugt eine Streitwertfestsetzung, indem der Streitwert auf den
//	 * höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
//	 * 
//	 * @return {@link StreitwertEntscheidungsElemente}
//	 * 
//	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#erzeugeStreitwertEntscheidung()
//	 */
//	@Override
//	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
//		StreitwertEntscheidungsElemente streitwert = new StreitwertEntscheidungsElemente();
//		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
//			double[] streitwerte = errechneStreitwert(true);
//			// Bei einer Widerklage müssen beide Streitwerte ausgegeben werden
//			for (double sw : streitwerte) {
//				streitwert.streitwerte.add(Euro.ofEuros(sw));// (new Long(new Double(sw).longValue()));
//			}
//		} else {
//			// Ohne Widerklage muss nur ein Streitwert ausgegeben werden
//			streitwert.streitwerte.add(Euro.ofEuros(baumbachBeteiligtenListe.getHoechstenStreitwert(BEKLAGTE)));
////					new Long(
////					new Double(baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.BEKLAGTE)).longValue()));
//		}
//		streitwert.text = texter.texteStreitwert(streitwert);
//		return streitwert;
//	}
//
//	/**
//	 * Die Methode erzeugt einen leeren sonstigen Tenor.
//	 * 
//	 * @return ein {@link SonstigeEntscheidungsElemente}
//	 * 
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
//	 */
//	@Override
//	public EntscheidungsElemente erzeugeSonstigeEntscheidung() {
//		EntscheidungsElemente sonstige = new EntscheidungsElemente();
//		sonstige.text = "";
//		return sonstige;
//	}
//
//	/**
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
//	 */
//	@Override
//	public String getBerechnungsTyp() {
//		return BESCHREIBUNG;
//	}
//
//	/**
//	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
//	 * 
//	 * @return den Text Gründe/Entscheidungsgründe als String
//	 * 
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
//	 */
//	@Override
//	public String getGruende() {
//		return gruende.toString();
//	}
//
//	/**
//	 * Die Methode erweitert die bisherigen Gründe
//	 * 
//	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
//	 */
//	public void erweitereGruende(CharSequence text) {
//		gruende.append(text);
//	}
//
//	/**
//	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
//	 * 
//	 * @return das Ergebnis von erzeugeHauptsachetenor() + erzeugeKostentenor() +
//	 *         erzeugeStreitwertFestsetzung()
//	 * 
//	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
//	 */
//	@Override
//	public TenorDatenContainer erzeugeContainer() {
//		container.berechnungsTyp = getBerechnungsTyp();
//		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
//		container.kostenEntscheidung = erzeugeKostenEntscheidung();
//		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
//		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
//		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
//		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
//		container.allgemein.selectedPanel = Streitgenossen.TAB_ID;
//		return container;
//	}
//
//	/**
//	 * Berechnet die Verteilung der Gerichtskosten
//	 * 
//	 * @return ein eindimensionales Array double[] mit einer Spalte für jeden
//	 *         Beteiligten
//	 */
//	private double[] ermittleGerichtskostenQuoten() {
//		double[] zwischen;
//		double fiktiverStreitwert;
//		zwischen = new double[baumbachBeteiligtenListe.size()];
//		// fiktiven Streitwert berechnen
//		fiktiverStreitwert = baumbachBeteiligtenListe.addiereStreitwerte();
//		// Unterliegen des Klägers: Anderer fiktiver Streitwert nötig: nur die
//		// Summe der Bk-Streitwerte!
//		double klaegerUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(BEKLAGTE);
//		// Bei Widerklage: zusätzlich noch das eigene Unterliegen des Klägers
//		// hinsichtlich Widerklage
//		if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(0))
//			klaegerUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(0, false, false);
//		zwischen[0] = (klaegerUnterliegen) * 100.0 / fiktiverStreitwert;
//		// Wenn DWBk vorhanden: Unterliegen jedes DwBk berechnen
//		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() > 0) {
//			int[] dwbkBereich = baumbachBeteiligtenListe.minUndMax(DRITTWIDERBEKLAGTE);
//			for (int i = dwbkBereich[0]; i <= dwbkBereich[1]; i++) {
//				double dwbkUnterliegen = 0.0;
//				// Dwbk können BaumbachGesamtschuldner sein, deshalb:
//				// beiGesamtschuldnernMultiplizieren = true
//				dwbkUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
//				zwischen[i] = dwbkUnterliegen * 100.0 / fiktiverStreitwert;
//			}
//		}
//		// Unterliegen jedes Beklagten
//		int[] beklagtenBereich = baumbachBeteiligtenListe.minUndMax(BEKLAGTE);
//		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
//			// Erst einmal das eigene Unterliegen hinsichtlich der Klage
//			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
//			zwischen[i] = bkUnterliegen * 100.0 / fiktiverStreitwert;
//		}
//		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
//			// Um Unterliegen bzgl. Widerklage bei jedem Widerbeklagten erhöhen,
//			// sofern auch Widerkläger
//			if (baumbachBeteiligtenListe.isEinzelpersonUndAnWiderklageBeteiligt(i)) {
//				double bkUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(KLAEGER)
//						+ baumbachBeteiligtenListe.addiereAngreiferUnterliegen(DRITTWIDERBEKLAGTE);
//				zwischen[i] += bkUnterliegen * 100.0 / fiktiverStreitwert;
//			}
//		}
//		baueAngriffsTabelle(zwischen, fiktiverStreitwert);
//		baueGruende();
//		return zwischen;
//	}
//
//	/**
//	 * Berechnet die Verteilung der außergerichtlichen Kosten des Klägers.
//	 * 
//	 * @return ein zweidimensionales Array double[][] mit einer Zeile je Kläger und
//	 *         einer Spalte für jeden Beteiligten.
//	 */
//	private double[][] ermittleKlaegerkostenQuoten() {
//		double[][] zwischen;
//		double fiktiverStreitwert;
//		// TODO Geht so nur für einen (!) Kläger
//		final int kl = 0;
//		zwischen = new double[1][baumbachBeteiligtenListe.size()];
//		// fiktiven Streitwert berechnen
//		// der fiktive Streitwert ist die Summe aller Bk-Streitwerte
//		// sowie des kl.eigenen Sw
//		fiktiverStreitwert = baumbachBeteiligtenListe.getStreitwertFor(kl)
//				+ baumbachBeteiligtenListe.addiereStreitwerte(BEKLAGTE);
//		// Unterliegen des Klägers: Anderer fiktiver Streitwert nötig: nur die
//		// Summe der Bk-Streitwerte!
//		double klUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(BEKLAGTE);
//		// Bei Widerklage muss hier noch das eigene Unterliegen hinzugerechnet
//		// werden.
//		if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(kl))
//			klUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(kl, false, true);
//		zwischen[kl][kl] = klUnterliegen * 100 / fiktiverStreitwert;
//		int[] beklagtenBereich = baumbachBeteiligtenListe.minUndMax(BEKLAGTE);
//		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
//			// Inwieweit war der jeweilige Beklagte hinsichtlich
//			// der Klage unterlegen?
//			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
//			// Bei Widerklage muss hier noch mehr hinzugerechnet werden:
//			// Inwieweit war der Beklagte (sofern Widerkläger und nicht
//			// Gesamtschuldnervereinigung) mit seiner Widerklage unterlegen?
//			if (baumbachBeteiligtenListe.isEinzelpersonUndAnWiderklageBeteiligt(i)
//					&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(kl))
//				bkUnterliegen += baumbachBeteiligtenListe.addiereKlaegerUnterliegenFor(kl, false, true);
//			zwischen[kl][i] = (bkUnterliegen / fiktiverStreitwert) * 100.0;
//		}
//		return zwischen;
//	}
//
//	/**
//	 * Berechnet die Verteilung der außergerichtlichen Kosten des
//	 * Drittwiderbeklagten
//	 * 
//	 * @return ein zweidimensionales Array double[][] mit einer Zeile je
//	 *         Drittwiderbeklagten und einer Spalte für jeden Beteiligten
//	 */
//	private double[][] ermittleDrittwiderbeklagtenkostenQuoten() {
//		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() < 1)
//			return null;
//		double[][] zwischen = new double[baumbachBeteiligtenListe.zaehleDrittwiderbeklagte()][baumbachBeteiligtenListe
//				.size()];
//		final int[] dwbkBereich = baumbachBeteiligtenListe.minUndMax(DRITTWIDERBEKLAGTE);
//		final int[] bkBereich = baumbachBeteiligtenListe.minUndMax(BEKLAGTE);
//		for (int zeile = 0; zeile < baumbachBeteiligtenListe.zaehleDrittwiderbeklagte(); zeile++) {
//			for (int beteiligter = 0; beteiligter < baumbachBeteiligtenListe.size(); beteiligter++) {
//				// Der Dwbk, dem die zeile "gehört"
//				if (beteiligter == zeile + dwbkBereich[0]
//						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(zeile + dwbkBereich[0]))
//					zwischen[zeile][beteiligter] = baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, false, true)
//							* 100.0 / baumbachBeteiligtenListe.getStreitwertFor(beteiligter);
//				// TODO: das klappt so nur, wenn es nur einen
//				// widerklagenden Beklagten gibt.
//				else if (beteiligter >= bkBereich[0] && baumbachBeteiligtenListe.isAnWiderklageBeteiligt(beteiligter)
//						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(zeile + dwbkBereich[0])) {
//					// Wenn es die Spalte des widerklagenden Beklagten ist
//					zwischen[zeile][beteiligter] = (baumbachBeteiligtenListe.getStreitwertFor(zeile + dwbkBereich[0])
//							- baumbachBeteiligtenListe.getUnterliegenFor(zeile + dwbkBereich[0], false, true)) * 100.0
//							/ baumbachBeteiligtenListe.getStreitwertFor(zeile + dwbkBereich[0]);
//				} else
//					// Rest mit 0.0 auffüllen
//					zwischen[zeile][beteiligter] = 0.0;
//			}
//		}
//		return zwischen;
//	}
//
//	/**
//	 * Berechnet die Verteilung der außergerichtlichen Kosten der Beklagten
//	 * 
//	 * @return Array double[][] mit so vielen Zeilen wie Beklagte und der
//	 *         Einfachheit halber (Einheitlichkeit) so vielen Spalten wie Parteien
//	 *         insgesamt.
//	 */
//	private double[][] ermittleBeklagtenkostenQuoten() {
//		double[][] zwischen = null;
//		// Array muss so viele Zeilen wie Beklagte und der Einfachheit halber
//		// (Einheitlichkeit) so viele Spalten wie Parteien insgesamt haben.
//		zwischen = new double[baumbachBeteiligtenListe.zaehleBeklagte()][baumbachBeteiligtenListe.size()];
//		final int[] bkBereich = baumbachBeteiligtenListe.minUndMax(BEKLAGTE);
//		final int[] klDwbkBereich = new int[] { baumbachBeteiligtenListe.minUndMax(KLAEGER)[0],
//				baumbachBeteiligtenListe.minUndMax(DRITTWIDERBEKLAGTE)[1] };
//		// zeilen-Schleife (i): für jeden Beklagten eine Zeile
//		for (int zeile = 0; zeile < baumbachBeteiligtenListe.zaehleBeklagte(); zeile++) {
//			// int betroffenerBeklagter: Index des betroffenen Beklagten
//			final int betroffenerBeklagter = zeile + bkBereich[0];
//			// Fiktiven Streitwert für den betroffenen Beklagten berechnen
//			double fiktiverStreitwert = baumbachBeteiligtenListe.getStreitwertFor(betroffenerBeklagter);
//			/*
//			 * Das Unterliegen des Beklagten selbst setzt sich zusammen aus seinem
//			 * Unterliegen hinsichtlich der Klage (und ggfls. der Wk) Zusätzliches
//			 * Unterliegen wird einberechnet
//			 */
//			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(betroffenerBeklagter, false, true);
//			/*
//			 * Wenn er Widerkläger ist: fiktiven Streitwert um Streitwert von jedem
//			 * widerbeklagten Kl./Dwbk erhöhen
//			 */
//			if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(betroffenerBeklagter))
//				for (int klDwbk = klDwbkBereich[0]; klDwbk <= klDwbkBereich[1]; klDwbk++)
//					if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(klDwbk))
//						fiktiverStreitwert += baumbachBeteiligtenListe.getStreitwertFor(klDwbk);
//			/*
//			 * Spalten-Schleife int beteiligter: Für jede Partei eine Spalte
//			 */
//			for (int beteiligter = 0; beteiligter <= bkBereich[1]; beteiligter++) {
//				/*
//				 * Wenn der betroffeneBeklagte ein BaumbachGesamtschuldner ist, wird er
//				 * übersprungen (sonst: Division durch 0)
//				 */
//				if (beteiligter >= klDwbkBereich[0] && beteiligter <= klDwbkBereich[1]
//						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(betroffenerBeklagter)) {
//					/*
//					 * Hier wird's bei Gesamtschuldnerischer Widerklage-Verurteilung kritisch: Bei
//					 * Gesamtschuldnerschaft muss multipliziert werden (deshalb true), sonst trägt
//					 * eine Kl/Dwbk-Gesamtschuldnerschaft zu wenig. Aber das eigene Unterliegen des
//					 * Klägers darf nicht hinzugerechnet werden, wenn er selbst an der Reihe ist,
//					 * sonst trägt er zuviel; die Kl/Dwbk-Gesamtschuldnerschaft kommt ja auch noch
//					 * an die Reihe. Deshalb dort false.
//					 */
//					final double klUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, true, false);
//					/*
//					 * Kläger: Zusätzliches Unterliegen wird einberechnet. Hier braucht man auch das
//					 * Unterliegen des Beklagten hinsichtlich einer gesamtschuldnerischen
//					 * Verurteilung, weil die Bk-Gesamtschuldnerschaft mit den
//					 * Bk-Außergerichtskosten nichts zu schaffen hat.
//					 */
//					if (baumbachBeteiligtenListe.getTypFor(beteiligter) == KLAEGER)
//						zwischen[zeile][beteiligter] = (baumbachBeteiligtenListe.getStreitwertFor(betroffenerBeklagter)
//								- (baumbachBeteiligtenListe.getUnterliegenFor(betroffenerBeklagter, false, true))) * 100
//								/ fiktiverStreitwert;
//					else
//						/*
//						 * den Dwbk geht das Unterliegen des Bk hinsichtlich der Klage nichts an
//						 */
//						zwischen[zeile][beteiligter] = 0.0;
//					/*
//					 * Wenn Bk und Kl/Dwbk an Wk beteiligt sind, wird das bkUnterliegen
//					 * heraufgesetzt...
//					 * 
//					 * (hier muss bei Gesamtschuldnerschaften multiplziert werden, sobald diese an
//					 * der Reihe ist. Bei den einfachen Beteiligten darf aber nicht deren
//					 * Unterliegen in der Gesamtschuldnerschaft hinzugerechnet werden (die kommen ja
//					 * auch noch dran, da bkUnterliegen über die Spalten-Schleife hinweg fortlebt)
//					 */
//					if ((baumbachBeteiligtenListe.isAnWiderklageBeteiligt(betroffenerBeklagter)
//							&& (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(beteiligter)))) {
//						bkUnterliegen += (baumbachBeteiligtenListe.getStreitwertFor(beteiligter)
//								- baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, true, false));
//						/*
//						 * ...und beim Kl./Dwbk wird das Unterliegen hinsichtlich der Wk hinzugerechnet
//						 */
//						zwischen[zeile][beteiligter] += (klUnterliegen * 100.0 / fiktiverStreitwert);
//					}
//				} else if (beteiligter == betroffenerBeklagter
//						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(betroffenerBeklagter)) {
//					// Der betreffende Beklagte, dem die Zeile "gehört"
//					zwischen[zeile][beteiligter] = bkUnterliegen;
//					zwischen[zeile][beteiligter] *= 100.0;
//					zwischen[zeile][beteiligter] /= fiktiverStreitwert;
//				} else {
//					/*
//					 * Rest mit Nullen auffüllen (Beklagte müssen untereinander nix tragen)
//					 */
//					zwischen[zeile][beteiligter] = 0.0;
//				}
//			}
//		}
//		return zwischen;
//	}
//
//	/**
//	 * Berechnet alle Kostenquoten, die gerichtlichen und die außergerichtlichen
//	 * aller Beteiligten.
//	 * 
//	 * @return ein zweidimensionales Array double[][] mit einer Zeile für die
//	 *         Gerichtskosten, und je einer weiteren Zeile für jeden Beteiligten. In
//	 *         jeder Zeile hat jeder Beteiligte eine Spalte.
//	 */
//	private double[][] ermittleQuoten() {
//		int beteiligte = baumbachBeteiligtenListe.size();
//		double[][] zwischen = new double[beteiligte + 1][beteiligte];
//		int zaehler = 0;
//		zwischen[zaehler] = ermittleGerichtskostenQuoten();
//		for (double[] quoten : ermittleKlaegerkostenQuoten()) {
//			zaehler++;
//			zwischen[zaehler] = quoten;
//		}
//		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() > 0)
//			for (double[] quoten : ermittleDrittwiderbeklagtenkostenQuoten()) {
//				zaehler++;
//				zwischen[zaehler] = quoten;
//			}
//		for (double[] quoten : ermittleBeklagtenkostenQuoten()) {
//			zaehler++;
//			zwischen[zaehler] = quoten;
//		}
//		return zwischen;
//	}
//
//	/**
//	 * Die Methode prüft, ob die Klage/Widerklage "im Übrigen" abgewiesen werden
//	 * muss.
//	 * 
//	 * @return ein {@link BaumbachImUebrigen}
//	 */
//	private BaumbachImUebrigen baueImUebrigen() {
//		// boolean[][] ganzZugesprochenOderAbgewiesen = new boolean[][]
//		// { { false, true}, { false, true } };
//		final BaumbachImUebrigen imUebrigen = new BaumbachImUebrigen(false, true, false, true);
//		/*
//		 * Wenn die Summe der Streitwerte so hoch ist wie die Summe des Unterliegens,
//		 * dann hat der Kläger voll obsiegt.
//		 */
//		// [0][0]
//		final double bkUnterliegen = baumbachBeteiligtenListe.addiereUnterliegen(BEKLAGTE);
//		imUebrigen.setKlageVollBegruendet(baumbachBeteiligtenListe.addiereStreitwerte(BEKLAGTE) <= bkUnterliegen);
//		/*
//		 * Sobald eine Verurteilung vorliegt, wurde die Klage jedenfalls nicht mehr ganz
//		 * abgewiesen.
//		 */
//		// [0][1] = false;
//		if (bkUnterliegen > 0.0) {
//			imUebrigen.setKlageGanzUnbegruendet(false);
//		}
//		if (!baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
//			return imUebrigen;
//		}
//		/*
//		 * Wenn die Summe der Streitwerte so hoch ist wie die Summe des Unterliegens,
//		 * dann hat der Widerkläger voll obsiegt.[1][0]
//		 */
//		final double wbkUnterliegen = baumbachBeteiligtenListe.addiereUnterliegen(KLAEGER)
//				+ baumbachBeteiligtenListe.addiereUnterliegen(DRITTWIDERBEKLAGTE);
//		imUebrigen.setWiderklageVollBegruendet((baumbachBeteiligtenListe.addiereStreitwerte(KLAEGER)
//				+ baumbachBeteiligtenListe.addiereStreitwerte(DRITTWIDERBEKLAGTE)) <= wbkUnterliegen);
//		/*
//		 * Sobald eine Verurteilung vorliegt, wurde die Widerklage jedenfalls nicht mehr
//		 * ganz abgewiesen.[1][1] = false;
//		 */
//		if (wbkUnterliegen > 0.0) {
//			imUebrigen.setWiderklageGanzUnbegruendet(false);
//		}
//		return imUebrigen;
//	}
//
//	/**
//	 * Die Methode listet alle Hauptsacheverhältnisse der Beteiligten untereinander
//	 * auf
//	 * 
//	 * @return eine {@link List}&lt;{@link HauptsacheVerhaeltnis }>
//	 */
//	private List<HauptsacheVerhaeltnis> baueHauptsacheVerhaeltnisse() {
//		List<HauptsacheVerhaeltnis> verhaeltnisse = new ArrayList<HauptsacheVerhaeltnis>();
//		BaumbachBeteiligter kl = baumbachBeteiligtenListe.get(baumbachBeteiligtenListe.minUndMax(KLAEGER)[0]);
//		BaumbachBeteiligter wkl = null;
//		// die Liste in "Tenorreihenfolge" durchgehen (Beklagte vor übrigen,
//		// Gesamtschuldner vor einfachen)
//		for (int index : baumbachBeteiligtenListe.beteiligteInTenorReihenfolge(false)) {
//			BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(index);
//			BaumbachBeteiligter glaeubiger = null;
//			// Wenn Typ Beklagter ist, ist der Gläubiger der Kläger, sonst der widerklagende
//			// Beklagte
//			// Da die Beklagten in der Liste vor den übrigen stehen (Tenorreihenfolge), muss
//			// der Widerkläger
//			// schon ermittelt sein, bevor es zum Kläger/den Widerbeklagten kommt
//			if (schuldner.getTyp() == BEKLAGTE) {
//				glaeubiger = kl;
//				if (schuldner.isAnWiderklageBeteiligt())
//					wkl = schuldner;
//			} else {
//				glaeubiger = wkl;
//			}
//			if (schuldner.getUnterliegen() > 0.0) {
//				HauptsacheVerhaeltnis verhaeltnis = new HauptsacheVerhaeltnis(glaeubiger, schuldner,
//						baumbachBeteiligtenListe.hatDarueberHinaus(index), Euro.ofEuros(schuldner.getUnterliegen()));
//				verhaeltnisse.add(verhaeltnis);
//			}
//		}
//		return verhaeltnisse;
//	}
//
//	/**
//	 * Die Methode listet alle Kostentragungsverhältnisse der Beteiligten
//	 * untereinander auf.
//	 * 
//	 * @param quoten siehe {@link #ermittleQuoten()}
//	 * @return eine {@link List}&lt;{@link KostenTragungsVerhaeltnis}>
//	 */
//	private List<KostenTragungsVerhaeltnis> baueKostenTragungsVerhaeltnisse(double[][] quoten) {
//		List<KostenTragungsVerhaeltnis> verhaeltnisse = new ArrayList<KostenTragungsVerhaeltnis>(
//				1 + baumbachBeteiligtenListe.size());
//		for (int zeile = 0; zeile < quoten.length; zeile++) {
//			Beteiligter glaeubiger;
//			if (zeile == 0) {
//				// die erste Zeile betrifft die Gerichtskosten
//				glaeubiger = new GerichtsKostenBeteiligter();
//			} else {
//				glaeubiger = baumbachBeteiligtenListe.get(zeile - 1);
//			}
//			/*
//			 * In Tenor-Reihenfolge bringen, d.h.: Die Spalten mit den
//			 * BaumbachGesamtschuldner müssen an den Anfang der Liste ihres Beteiligtentyps
//			 */
//			for (int spalte : baumbachBeteiligtenListe.beteiligteInTenorReihenfolge(true)) {
//				BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(spalte);
//				double kostenBruchteil = quoten[zeile][spalte];
//				if (kostenBruchteil > 0.0) {
//					/*
//					 * hier muss der Kostenbruchteil durch 100 geteilt werden, um den Anforderungen
//					 * des TenorDatenContainers zu genügen.
//					 */
//					KostenTragungsVerhaeltnis verhaeltnis = new KostenTragungsVerhaeltnis(glaeubiger, schuldner,
//							hatDarueberHinaus(spalte, quoten[zeile]), kostenBruchteil / 100.0);
//					verhaeltnisse.add(verhaeltnis);
//				}
//			}
//		}
//		return verhaeltnisse;
//	}
//
//	/**
//	 * Die Methode konvertiert die {@link #vollstreckbarkeit} in eine Liste der
//	 * {@link VollstreckungsVerhaeltnis}se. Die Liste ist sortiert nach Maßgabe von
//	 * {@link VollstreckungsVerhaeltnis#compareTo(eu.gronos.kostenrechner.data.tenordaten.ProzessVerhaeltnis)}.
//	 * 
//	 * @param liste eine {@link List}&lt;{@link Vollstreckbarkeit}&gt;
//	 * @return eine {@link List}&lt;{@link VollstreckungsVerhaeltnis}&gt;
//	 */
//	private List<VollstreckungsVerhaeltnis> baueVollstreckungsVerhaeltnisse(VollstreckbarkeitsListe liste) {
//		if (liste != null && !liste.isEmpty() && liste.sindAlle708iVm713()) {
//			return new VollstreckbarkeitsHelfer(container.hauptsacheEntscheidung.prozessverhaeltnisse,
//					container.kostenEntscheidung.prozessverhaeltnisse)
//					.errechneAlleVollstreckungsVerhaeltnisse(liste.get(0));
//		}
//		List<VollstreckungsVerhaeltnis> verhaeltnisse = new ArrayList<VollstreckungsVerhaeltnis>();
//		for (Vollstreckbarkeit vo : liste) {
//			if (vo instanceof BaumbachVollstreckbarkeit) {
//				BaumbachVollstreckbarkeit bvo = (BaumbachVollstreckbarkeit) vo;
//				// Vor dem Hinzufügen müssen "leere" VollstreckbarkeitsObjekte und das Tragen
//				// eigener Kosten herausgefiltert werden
//				if (bvo.hatEintraege() && !bvo.getGlaeubiger().equals(bvo.getSchuldner())) {
//					VollstreckungsVerhaeltnis verhaeltnis = new VollstreckungsVerhaeltnis(bvo);
//					verhaeltnisse.add(verhaeltnis);
//				}
//			}
//		}
//		Collections.sort(verhaeltnisse);
//		return verhaeltnisse;
//	}
//
//	/**
//	 * Wenn eine der Quoten unter 10% ist, muss sie den verbleibenden sinnvoll
//	 * "aufgeschlagen" werden:
//	 * 
//	 * § 92 Abs. 2 ZPO: Das Gericht kann der einen Partei die gesamten Prozesskosten
//	 * auferlegen, wenn 1. die Zuvielforderung der anderen Partei verhältnismäßig
//	 * geringfügig war und keine oder nur geringfügig höhere Kosten veranlasst hat
//	 * oder 2. der Betrag der Forderung der anderen Partei von der Festsetzung durch
//	 * richterliches Ermessen, von der Ermittlung durch Sachverständige oder von
//	 * einer gegenseitigen Berechnung abhängig war.
//	 * 
//	 * @param quoten ein Array double[] mit den quoten im Urzustand
//	 * @return ein Array double[] mit den quoten unter Berücksichtigung von § 92
//	 *         Abs. 2 ZPO
//	 */
//	private double[] entferneUnwesentlichesUnterliegen(double[] quoten) {
//		double[] bereinigteQuoten = new double[quoten.length];
//		double summe = 0.0;
//		// Erster Schritt: alle Quoten durchgehen
//		for (int index = 0; index < bereinigteQuoten.length; index++) {
//			// ... und Einträge unter 10% ("verhältnismäßig gering") nicht
//			// übernehmen ... Caveat: Bei Baumbach rechne ich mit "ganzen" Prozenten, also
//			// nicht durch 100.0 teilen!
//			if (quoten[index] < allgemein.vomHundert92ii)// 10.0(100.0 * )
//				bereinigteQuoten[index] = 0.0;
//			else {
//				// / ... andere hingegen aufsummieren
//				bereinigteQuoten[index] = quoten[index];
//				summe += quoten[index];
//			}
//		}
//		// Wenn es keine Einzelquoten über 10% gab, dann das Array unversehrt
//		// zurückgeben; ebenso, wenn die summe schon über 100% geht
//		if (summe <= 0.0 || summe > 100.0)
//			return quoten;
//		// Zweiter Schritt: Sonst das Verhältnis der Summe zu 100% feststellen
//		final double verhaeltnis = 100.0 / summe;
//		// Dritter Schritt: Alle Quoten des neuen Arrays mit dem verhaeltnis
//		// multiplizieren
//		for (int index = 0; index < bereinigteQuoten.length; index++) {
//			bereinigteQuoten[index] = bereinigteQuoten[index] * verhaeltnis;
//		}
//		return bereinigteQuoten;
//	}
//
//	/**
//	 * Findet über die BaumbachBeteiligtenListe alle Gesamtschuldnerschaften, in
//	 * denen der BaumbachBeteiligte mit dem Index beteiligter ist, dann gucken, ob
//	 * sie > 0% tragen müssen
//	 * 
//	 * @param index   die Position in der BaumbachBeteiligtenListe
//	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
//	 *                Zeile
//	 * @return true, wenn der BaumbachBeteiligte unter index in einer
//	 *         Gesamtschuldnerschaft ist, die eine Kostenquote > 0 tragen muss
//	 */
//	private boolean hatDarueberHinaus(int index, double[] spalten) {
//		final ArrayList<Integer> darueberHinaus = findeDarueberHinaus(index, spalten);
//		return darueberHinaus != null && darueberHinaus.size() > 0 && spalten != null;
//	}
//
//	/**
//	 * Findet über die BaumbachBeteiligtenListe alle Gesamtschuldnerschaften, in
//	 * denen der BaumbachBeteiligte mit dem Index beteiligter ist, dann gucken, ob
//	 * sie > 0% tragen müssen
//	 * 
//	 * @param index   die Position in der BaumbachBeteiligtenListe
//	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
//	 *                Zeile
//	 * @return eine ArrayList mit den Indexen von allen Gesamtschuldnerschaften, die
//	 *         eine Kostenquote > 0 tragen müssen, in denen der BaumbachBeteiligte
//	 *         unter index in einer Gesamtschuldnerschaft verurteilt ist
//	 */
//	private ArrayList<Integer> findeDarueberHinaus(int index, double[] spalten) {
//		ArrayList<Integer> liste = baumbachBeteiligtenListe.findeDarueberHinaus(index);
//		ArrayList<Integer> darueberHinaus = new ArrayList<Integer>();
//		if (liste != null && liste.size() > 0 && spalten != null)
//			for (int i : liste) {
//				if (spalten[i] > 0.0)
//					darueberHinaus.add(new Integer(i));
//			}
//		return darueberHinaus;
//	}
//
//	/**
//	 * Die Methode dient dazu, die gesamte mögliche Quote zurückzugeben. Es wird die
//	 * zusätzliche Quote zurückgegeben, die sich aus den BaumbachGesamtschuldnern
//	 * ergibt, die auch den referenzierten BaumbachBeteiligten repräsentieren.
//	 * 
//	 * {@link BaumbachBeteiligtenListe#getUnterliegenFor(int, boolean, boolean)}
//	 * 
//	 * @param index   die Position in der BaumbachBeteiligtenListe
//	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
//	 *                Zeile
//	 * @return die Summe als double
//	 */
//	private double berechneGanzeQuote(int index, double[] spalten) {
//		/* Erst einmal trägt er, was er selbst trägt */
//		double ganzeQuote = spalten[index];
//		final ArrayList<Integer> darueberHinaus = findeDarueberHinaus(index, spalten);
//		/* Und dann, wozu er gesamtschuldnerisch mit anderen verurteilt wurde */
//		if (darueberHinaus != null && darueberHinaus.size() > 0) {
//			for (Integer i : darueberHinaus)
//				ganzeQuote += spalten[i];
//		}
//		return ganzeQuote;
//	}
//
//	/**
//	 * Der Getter für {@link #kosten} gibt den Inhalt selbigen Felds zurück. Ist
//	 * dieses noch <code>null</code>, so wird vorher noch {@link #ermittleQuoten()}
//	 * aufgerufen, um es zu füllen und
//	 * {@link #entferneUnwesentlichesUnterliegen(double[])
//	 * entferneUnwesentlichesUnterliegen}, um ggfls. ein unwesentliches Unterliegen
//	 * zu entfernen.
//	 * 
//	 * @return gibt {@link #kosten} als double[][] zurück
//	 */
//	double[][] getKosten() {
//		if (kosten == null) {
//			kosten = ermittleQuoten();
//			if (allgemein.b92ii)
//				for (int zeile = 0; zeile < kosten.length; zeile++) {
//					/*
//					 * Erst einmal die Quoten untersuchen und unwesentliches Unterliegen anderweitig
//					 * verteilen (wenn ausgewählt)
//					 */
//					kosten[zeile] = entferneUnwesentlichesUnterliegen(kosten[zeile]);
//				}
//
//			if (Kostenrechner.debug) {
//				StringBuilder sb = new StringBuilder();
//				for (double d[] : kosten) {
//					double summe = 0.0;
//					for (double dd : d) {
//						sb.append(String.format("%,.2f; ", dd));
//						summe += dd;
//					}
//					sb.append(String.format("(Summe: %,.2f).%n", summe));
//				}
//				Kostenrechner.getLogger().info(sb.toString());
//			}
//
//		}
//		return kosten;
//	}
//
//	/**
//	 * Die Methode dient dazu, den Streitwert zu ermitteln, indem der Streitwert auf
//	 * den höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
//	 * 
//	 * @param mitWiderklage wenn eine
//	 *                      {@link BaumbachBeteiligtenListe#enthaeltAnWiderklageBeteiligte()
//	 *                      Widerklage vorhanden} ist
//	 * @return wenn mitWiderklage, dann ein Array double[3], bei dem 0 der
//	 *         Streitwert insgesamt, 1 der Klagestreitwert und 2 der
//	 *         Widerklagestreitwert sind; sonst double[1] nur mit dem Streitwert.
//	 */
//	private double[] errechneStreitwert(boolean mitWiderklage) {
//		// Bei einer Widerklage müssen beide Streitwerte ermittelt werden
//		if (mitWiderklage) {
//			final double klSw = baumbachBeteiligtenListe.getHoechstenStreitwert(BEKLAGTE);
//			final double wkSw = baumbachBeteiligtenListe.getHoechstenStreitwert(KLAEGER) > baumbachBeteiligtenListe
//					.getHoechstenStreitwert(DRITTWIDERBEKLAGTE)
//							? baumbachBeteiligtenListe.getHoechstenStreitwert(KLAEGER)
//							: baumbachBeteiligtenListe.getHoechstenStreitwert(DRITTWIDERBEKLAGTE);
//			final double sw = !wirtschaftlichIdentisch ? klSw + wkSw : klSw > wkSw ? klSw : wkSw;
//			return new double[] { sw, klSw, wkSw };
//		} else {
//			/* Ohne Widerklage muss nur ein Streitwert ermittelt werden */
//			return new double[] { baumbachBeteiligtenListe.getHoechstenStreitwert(BEKLAGTE) };
//		}
//	}
//
//	/**
//	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
//	 * beginnt sie neu mit dem übergebenen Text.
//	 * 
//	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
//	 *             Gründe.
//	 */
//	private void starteGruende(CharSequence text) {
//		if (gruende.length() > 0)
//			gruende.delete(0, gruende.length());
//		erweitereGruende(text);
//	}
//
//	/**
//	 * Die Methode dient dazu, über {@link #MISSING()} die Gründe für die
//	 * BaumbachFormel zu basteln.
//	 */
//	private void baueGruende() {
//		starteGruende(GRUENDE_91_92_100II);
//		zeilen.toStringBuilder(gruende);
//	}
//
//	/**
//	 * Die Methode dient dazu, eine "Tabelle" (also eine ArrayList aus Arrays
//	 * Object[]) zu erzeugen, das dann zur Begründung anhand der Einzelangriffe
//	 * genutzt werden kann.
//	 * 
//	 * @param quoten             die Gerichtskostenquoten als double[][]
//	 * @param fiktiverStreitwert ein double mit dem fiktiven Streitwert der
//	 *                           Gerichtskosten
//	 */
//	private void baueAngriffsTabelle(double[] quoten, double fiktiverStreitwert) {
//		/*
//		 * Durch die Verwendung einer ArrayList muss die Anzahl der Zeilen nicht
//		 * ermittelt werden, aber eine Liste der Angriffe muss über
//		 * beteiligteWieVerklagt ermittelt werden.
//		 */
//		zeilen = new TabulierendZeilen();
//		final ArrayList<Integer> beteiligtenIndicesZeilen = baumbachBeteiligtenListe.beteiligteWieVerklagt();
//		/*
//		 * Liste der Beteiligten/Gesamtschuldnerschaften über
//		 * beteiligteInTenorReihenfolge für Zahl der Spalten ermitteln
//		 */
//		final ArrayList<Integer> beteiligtenIndicesSpalten = baumbachBeteiligtenListe
//				.beteiligteInTenorReihenfolge(true);
//		/* Hierin werden die Verluste der Beteiligten zum Summieren gespeichert */
//		final double[] verluste = new double[beteiligtenIndicesSpalten.size()];
//		for (int i = 0; i < verluste.length; i++)
//			verluste[i] = 0.0;
//		/* Kopfzeile und Ergebniszeile bauen und der ArrayList hinzufügen */
//		final List<String> columnHeaders = new ArrayList<String>(beteiligtenIndicesSpalten.size() + 1);
//		/*
//		 * Kopfzeile mithilfe einer For-Schleife füllen; die Liste enthält eine Spalte
//		 * mehr als beteiligteInTenorReihenfolge, weil zuerst noch KOPFZEILE_BEGINN
//		 * eingefügt wird.
//		 */
//		columnHeaders.add(KOPFZEILE_BEGINN);
//		for (Integer index : beteiligtenIndicesSpalten) {
//			columnHeaders.add(baumbachBeteiligtenListe.parteiBezeichner(GENITIV, index.intValue()));
//		}
//		zeilen.add(columnHeaders);
//		/*
//		 * ArrayList füllen: äußere Schleife: für jede Zeile eine ArrayList<Double>
//		 * anlegen mit 1*String und 1..n*Double
//		 */
//		for (int zeile = 0; zeile < beteiligtenIndicesZeilen.size(); zeile++) {
//			/*
//			 * Die ArrayList<Double> mit den Spalten für die aktuelle Zeile
//			 */
//			DoubleDataRows doubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
//			final int a = beteiligtenIndicesZeilen.get(zeile).intValue();
//			final BaumbachBeteiligter bbtA = baumbachBeteiligtenListe.get(a);
//			/* Die "Kopfspalte" links sagt Angriff wer gegen wen */
//			final String rowHeader = baumbachBeteiligtenListe.benenneAngriffGegen(a);
//			for (int spalte = 0; spalte < beteiligtenIndicesSpalten.size(); spalte++) {
//				final int b = beteiligtenIndicesSpalten.get(spalte).intValue();
//				final BaumbachBeteiligter bbtB = baumbachBeteiligtenListe.get(b);
//				double wert = 0.0;
//				/*
//				 * Wert für innere Tabelle nach BeteiligtenTyp von a und b ermitteln
//				 */
//				if (a == b) {
//					/*
//					 * Wenn a und b identisch (dasselbe Element in der bbtl), dann bezieht sich das
//					 * Unterliegen von b auf den Angriff gegen b, ergo: das Unterliegen von b ist
//					 * hier richtig.
//					 */
//					wert = bbtB.getUnterliegen();
//				} else if (bbtB.isGesamtschuldnerschaft() && baumbachBeteiligtenListe.getAlsGesamtschuldner(b).contains(
//						bbtA.getTyp(), baumbachBeteiligtenListe.ermittleLaufendeNummerFuerBeteiligtenTyp(a))) {
//					/*
//					 * Wenn b eine gesamtschuldnerische Verurteilung ist und a (gegen den sich der
//					 * Angriff richtet) darin enthalten ist, dann muss das Unterliegen der
//					 * Gesamtschuldnerschaft in die Spalte.
//					 */
//					wert = bbtB.getUnterliegen();
//				} else if (bbtA.getTyp() == BEKLAGTE) {
//					/*
//					 * Wenn a Beklagter und b Kläger ist, dann Unterliegen berechnen
//					 */
//					if (bbtB.getTyp() == KLAEGER)
//						wert = bbtA.getStreitwert() - baumbachBeteiligtenListe.getUnterliegenFor(a, false, true);
//				} else if ((bbtA.getTyp() == KLAEGER && bbtA.isAnWiderklageBeteiligt())
//						|| bbtA.getTyp() == DRITTWIDERBEKLAGTE) {
//					/*
//					 * Wenn a Widerbeklagter ist und b Widerkläger, dann Unterliegen berechnen.
//					 */
//					if (bbtB.getTyp() == BEKLAGTE && bbtB.isAnWiderklageBeteiligt())
//						wert = bbtA.getStreitwert() - baumbachBeteiligtenListe.getUnterliegenFor(a, false, true);
//				}
//				/*
//				 * Den ermittelten Wert in die Spalte schreiben und auf die Verluste addieren.
//				 */
//				doubles.add(new Double(wert));
//				verluste[spalte] += wert;
//			}
//			zeilen.add(rowHeader, doubles);
//		}
//		/* Zeilen für Verlustsummen und Ergebnis erzeugen */
//		final DoubleDataRows ergebnisDoubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
//		final String ergebnisHeader = String.format(ERGEBNIS_ZEILE, fiktiverStreitwert);
//		final DoubleDataRows verlusteDoubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
//		final String verlusteHeader = VERLUST_GESAMT;
//		/*
//		 * Verlustsummen-Zeile und Ergebniszeile mithilfe einer For-Schleife füllen
//		 */
//		for (int spalte = 0; spalte < beteiligtenIndicesSpalten.size(); spalte++) {
//			verlusteDoubles.add(new Double(verluste[spalte]));
//			ergebnisDoubles.add(new Double(quoten[beteiligtenIndicesSpalten.get(spalte).intValue()]));
//		}
//		/* Verlustsummen-Zeile und Ergebniszeile hinzufügen */
//		zeilen.add(verlusteHeader, verlusteDoubles);
//		zeilen.add(ergebnisHeader, ergebnisDoubles);
//	}
//
//	/**
//	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
//	 * Vorläufige Vollstreckbarkeit: erst einmal gucken, ob die Hauptsache über die
//	 * Schwelle von §709 kommt. Wenn nicht, dann bei den Kosten rechnen: Array
//	 * double[][] durchgehen, Quote mit jeweiligen Kosten multiplizieren und für
//	 * jede Partei zusammen rechnen. Die jeweils eigenen Kosten dabei auslassen.
//	 * 
//	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
//	 *         Unanfechtbarkeit speichert, die weiteren Elemente speichern die
//	 *         vorläufige Vollstreckbarkeit für die
//	 *         Einzelvollstreckungsverhältnisse.
//	 */
//	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
//		VollstreckbarkeitsListe vollstreckbarkeit = new VollstreckbarkeitsListe();
//		/* Unanfechtbarkeit prüfen und der Liste hinzufügen */
//		vollstreckbarkeit.add(pruefeUnanfechtbarkeit());
//		/* Nach festgestellter Unanfechtbarkeit war's das schon. */
//		if (vollstreckbarkeit.sindAlle708iVm713())
//			return vollstreckbarkeit;
//		/*
//		 * Einzelne Vollstreckungsverhältnisse durchgehen und
//		 * BaumbachVollstreckbarkeitsObjekt der Liste hinzufügen; an quoten[][]
//		 * entlanghangeln.
//		 */
//		double[][] quoten = getKosten();
////		final long streitwert = (long) errechneStreitwert(baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0];
//		final Euro streitwert = Euro
//				.ofEuros(errechneStreitwert(baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0]);
//		for (int glaeubigerIndex = 0; glaeubigerIndex < baumbachBeteiligtenListe.size(); glaeubigerIndex++) {
//			for (int schuldnerIndex = 0; schuldnerIndex < baumbachBeteiligtenListe.size(); schuldnerIndex++) {
//				BaumbachBeteiligter glaeubiger = baumbachBeteiligtenListe.get(glaeubigerIndex);
//				BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(schuldnerIndex);
//				if (glaeubigerIndex != schuldnerIndex && !(glaeubiger instanceof BaumbachGesamtschuldnerschaft)
//						&& !(schuldner instanceof BaumbachGesamtschuldnerschaft)) {
//					double hauptsache = 0;
//					Euro vollstreckbareKosten = Euro.ZERO_CENTS;// double vollstreckbareKosten = 0;
//					if (glaeubiger.getTyp() == KLAEGER) {
//						/* Bei einem Kläger vorher die Gerichtskosten drauf KostenVerzeichnis.KV1210 */
//						vollstreckbareKosten = vollstreckbareKosten.add(//
//								GebuehrenAuflistung.errechneGebuehr(TenorTexter.getKostenKlaeger().get(0), streitwert)//
//										.multiply(berechneGanzeQuote(schuldnerIndex, quoten[0]) / 100.0));
////						vollstreckbareKosten += berechneGanzeQuote(schuldnerIndex, quoten[0]) / 100.0
////								* GebuehrenAuflistung.errechneGebuehr(TenorTexter.getKostenKlaeger().get(0),
////										streitwert);
//					}
//					/*
//					 * Die Anwaltskosten (KOSTEN_BEKLAGTER) immer - auch bei Kl und Dwbk
//					 */
//					vollstreckbareKosten = vollstreckbareKosten.add(//
//							(TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwert))//
//									.multiply(berechneGanzeQuote(schuldnerIndex, quoten[glaeubigerIndex + 1]) / 100.0));
////					vollstreckbareKosten += berechneGanzeQuote(schuldnerIndex, quoten[glaeubigerIndex + 1]) / 100.0
////							* (long) (TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwert));
//					// hauptsache berechnen, wenn es in der "Paarung" eine
//					// Hauptsache gibt
//					if ((KLAEGER == glaeubiger.getTyp() && BEKLAGTE == schuldner.getTyp())
//							|| (BEKLAGTE == glaeubiger.getTyp() && glaeubiger.isAnWiderklageBeteiligt()
//									&& schuldner.isAnWiderklageBeteiligt()
//									&& (KLAEGER == schuldner.getTyp() || DRITTWIDERBEKLAGTE == schuldner.getTyp()))) {
//						hauptsache = baumbachBeteiligtenListe.getUnterliegenFor(schuldnerIndex, false, true);
//					}
//					vollstreckbarkeit.add(BaumbachVollstreckbarkeit.pruefeSicherheitsleistung((long) hauptsache,
//							vollstreckbareKosten.longValue(), glaeubiger, schuldner));
//				}
//			}
//		}
//		return vollstreckbarkeit;
//	}
//
//	/**
//	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
//	 * 
//	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
//	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
//	 */
//	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
//		/*
//		 * Wenn der Streitwert schon nicht über 600 EUR geht, ist das Urteil
//		 * unanfechtbar.
//		 */
//		if (errechneStreitwert(
//				baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0] <= TenorTexter.BERUFUNGS_GRENZE
//						.doubleValue()) {
//			return new Vollstreckbarkeit(true, false, false, false, true);
//		} else if (errechneStreitwert(baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0] <= 2
//				* TenorTexter.BERUFUNGS_GRENZE.doubleValue()) {
//			/*
//			 * Beschwer jedes Einzelbeteiligten überprüfen, ob nicht über Berufungsgrenze
//			 */
//			for (int index = 0; index < baumbachBeteiligtenListe.size(); index++)
//				if (!baumbachBeteiligtenListe.isGesamtschuldnerschaft(index)) {
//					/* erstmal das "eigene" "ganze" Unterliegen */
//					double unterliegen = baumbachBeteiligtenListe.getUnterliegenFor(index, false, true);
//					/*
//					 * Wenn der beteiligte ein "Angreifer" ist:
//					 */
//					/* Unterschleife beim Kläger ggü Bk. */
//					if (baumbachBeteiligtenListe.getTypFor(index) == KLAEGER) {
//						double angreiferUnterliegen = 0.0;
//						for (int angriff = 0; angriff < baumbachBeteiligtenListe.size(); angriff++) {
//							if (baumbachBeteiligtenListe.getTypFor(angriff) == BEKLAGTE
//									&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(angriff)) {
//								final double angriffsUnterliegen = baumbachBeteiligtenListe.getStreitwertFor(angriff)
//										- baumbachBeteiligtenListe.getUnterliegenFor(angriff, false, true);
//								/*
//								 * Nach Vorhandensein von Gesamtschuldnerschaften unterscheiden
//								 */
//								if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()
//										&& angreiferUnterliegen < angriffsUnterliegen) {
//									angreiferUnterliegen = angriffsUnterliegen;
//								} else if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()) {
//									// nüscht
//								} else {
//									angreiferUnterliegen += angriffsUnterliegen;
//								}
//							}
//						}
//						unterliegen += angreiferUnterliegen;
//					}
//					/* Unterschleife beim Bk/Widerkläger ggü. Widerbeklagten */
//					if (baumbachBeteiligtenListe.getTypFor(index) == BEKLAGTE
//							&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(index)) {
//						double angreiferUnterliegen = 0.0;
//						for (int angriff = 0; angriff < baumbachBeteiligtenListe.size(); angriff++) {
//							if (!baumbachBeteiligtenListe.isGesamtschuldnerschaft(angriff)
//									&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(angriff)
//									&& (baumbachBeteiligtenListe.getTypFor(angriff) == DRITTWIDERBEKLAGTE
//											|| baumbachBeteiligtenListe.getTypFor(angriff) == KLAEGER)) {
//								final double angriffsUnterliegen = baumbachBeteiligtenListe.getStreitwertFor(angriff)
//										- baumbachBeteiligtenListe.getUnterliegenFor(angriff, false, true);
//								/*
//								 * Nach Vorhandensein von Gesamtschuldnerschaften unterscheiden
//								 */
//								if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()
//										&& angreiferUnterliegen < angriffsUnterliegen) {
//									angreiferUnterliegen = angriffsUnterliegen;
//								} else if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()) {
//									// nüscht
//								} else {
//									angreiferUnterliegen += angriffsUnterliegen;
//								}
//							}
//						}
//						unterliegen += angreiferUnterliegen;
//					}
//					/*
//					 * Sobald irgend etwas höher als die Berufungsgrenze ist, ist das Urteil
//					 * anfechtbar.
//					 */
//					if (unterliegen > TenorTexter.BERUFUNGS_GRENZE.doubleValue())
//						return new Vollstreckbarkeit();
//				}
//			/*
//			 * Wenn wir hier angekommen sind (also aus der Schleife vorher nicht
//			 * ausgestiegen sind), ist das Urteil unanfechtbar
//			 */
//			return new Vollstreckbarkeit(true, false, false, false, true);
//		} else {
//			return new Vollstreckbarkeit();
//		}
//	}

}
