package eu.gronos.kostenrechner.logic.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.GENITIV;

import java.util.ArrayList;
import java.util.List;

import eu.gronos.kostenrechner.data.baumbach.Angriff;
import eu.gronos.kostenrechner.data.baumbach.AngriffListe;
import eu.gronos.kostenrechner.data.baumbach.MehrfachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.StreitgenossenAngriffe;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.TenorToken;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.VerlusteBank;

/**
 * Etwas Ähliches wie {@link TenorTexter}, nur für die
 * {@link TenorVorbereitend#getGruende()}
 *
 * @author Peter Schuster (setrok)
 * @date 21.11.2021
 *
 */
public class BegruendungsTexter {

	private final StringBuilder gruende = new StringBuilder();
	private BegruendungsZahlenTabelle zeilen;
	private final StreitgenossenAngriffe streit;

	public BegruendungsTexter(StreitgenossenAngriffe streit) {
		super();
		this.streit = streit;
	}

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
	 */
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * Die Methode hilft einen {@link TenorDatenContainer} zu befüllen. Dazu liefert
	 * es die nötigen Daten für ein {@link BegruendungsElemente} zurück.
	 * 
	 * @return das erstellte {@link BegruendungsElemente}
	 */
	public BegruendungsElemente begruendungsElemente() {
		return this.zeilen.toBegruendungsElemente(getGruende());
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht ohnehin schon
	 * leer).
	 * 
	 */
	void starteGruende() {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

	/**
	 * Die Methode dient dazu, über
	 * {@link BegruendungsZahlenTabelle#toStringBuilder(StringBuilder)} die Gründe für die
	 * BaumbachFormel zu basteln.
	 * 
	 * @param quoten die Gerichtskostenquoten als {@link VerlusteBank}
	 * @param zeilen die {@link BegruendungsZahlenTabelle} aus
	 *               {@link #baueAngriffsTabelle(VerlusteBank)}
	 * 
	 * @return einen {@link StringBuilder}
	 */
	StringBuilder baueGruende(VerlusteBank quoten) {
		this.zeilen = baueAngriffsTabelle(quoten);
		starteGruende();
		erweitereGruende(TenorToken.GRUENDE_EINLEITUNG);
		erweitereGruende(TenorToken.GRUENDE_91_92_100II);
		erweitereGruende(TenorToken.DOPPELPUNKT_ABSATZ);
		zeilen.toStringBuilder(gruende);
		return gruende;
	}

	/**
	 * Die Methode dient dazu, eine "Tabelle" (also eine {@link BegruendungsZahlenTabelle}
	 * aus {@link String}s und {@link DoubleDataRows}), das dann zur Begründung
	 * anhand der Einzelangriffe genutzt werden kann.
	 * 
	 * @param quoten             die Gerichtskostenquoten als {@link VerlusteBank}
	 * @param fiktiverStreitwert ein double mit dem fiktiven Streitwert der
	 *                           Gerichtskosten
	 */
	private BegruendungsZahlenTabelle baueAngriffsTabelle(VerlusteBank quoten) {
		/*
		 * Durch die Verwendung einer TabulierendZeilen-ArrayList muss die Anzahl der
		 * Zeilen nicht ermittelt werden, aber eine Liste der Angriffe muss über
		 * beteiligteWieVerklagt ermittelt werden.
		 */
		// TODO final TabulierendZeilen zeilen = new TabulierendZeilen();
//		final BegruendungsTabelle zeilen = new BegruendungsTabelle();
		final BegruendungsZahlenTabelle zeilen = new BegruendungsZahlenTabelle();
		final List<List<Angriff>> gruppen = AngriffListe.gruppiereAngriffe(this.streit.angriffe);
		// final List<Beteiligter> betList = AngriffListe.alleBeteiligteAus(angriffe);

		/*
		 * Hierin werden die Verluste der Beteiligten zum Summieren gespeichert.
		 * Kopfzeile und Ergebniszeile bauen und der ArrayList hinzufügen
		 */
		final List<String> columnHeaders = new ArrayList<String>();
		columnHeaders.add(TenorToken.KOPFZEILE_BEGINN.toString());
		final List<List<Beteiligter>> beteiligtenSpalten = fuelleBeteiligtenSpalten(columnHeaders);
		zeilen.add(columnHeaders);

		for (List<Angriff> gruppe : gruppen) {
			// TODO BegruendungsTabelle
			fuelleZeilenFuerGruppe(zeilen, gruppe, this.streit.beteiligte, beteiligtenSpalten);
		}

		/*
		 * Nach Ende der äußeren Schleife: Zeilen für Verlustsummen und Ergebnis
		 * erzeugen
		 */
		final String ergebnisHeader = TenorToken.ERGEBNIS_ZEILE.toString() + quoten.getFiktiverStreitwert()
				+ TenorToken.LEER + TenorToken.EURO;
		final String verlusteHeader = TenorToken.VERLUST_GESAMT.toString();

		/* Verlustsummen-Zeile und Ergebniszeile hinzufügen */
		// TODO BegruendungsTabelle
		zeilen.add(verlusteHeader, fuelleVerlustsummen(quoten, beteiligtenSpalten));
		// TODO BegruendungsTabelle
		zeilen.add(ergebnisHeader, fuelleErgebnisse(quoten, beteiligtenSpalten));
		return zeilen;
	}

	/**
	 * Nach Ende der äußeren Schleife: Zeilen für Verlustsummen und Ergebnis
	 * erzeugen. Verlustsummen-Zeile und Ergebniszeile mithilfe einer For-Schleife
	 * füllen
	 * 
	 * @param quoten             die Gerichtskostenquoten als {@link VerlusteBank}
	 * @param beteiligtenSpalten {@link List} der {@link Beteiligter}n für Zahl der
	 *                           Spalten
	 */ // TitledNumberRow
	private BegruendungsZahlenZeile fuelleErgebnisse(VerlusteBank quoten,
			final List<List<Beteiligter>> beteiligtenSpalten) {
		// final TitledNumberRow ergebnisRows = new
		// TitledNumberRow(beteiligtenSpalten.size());
		final BegruendungsZahlenZeile ergebnisRows = new BegruendungsZahlenZeile(beteiligtenSpalten.size());
		for (int spalte = 0; spalte < beteiligtenSpalten.size(); spalte++) {
			if (quoten.get(beteiligtenSpalten.get(spalte)) != null) {
				ergebnisRows.add(quoten.getFractionFor(beteiligtenSpalten.get(spalte)));
				// Double.valueOf(100.0 * getQuotaFor)
			} else {
				ergebnisRows.add(Fraction.ZERO);// 0.0
			}
		}
		return ergebnisRows;
	}

	/**
	 * Nach Ende der äußeren Schleife: Zeilen für Verlustsummen und Ergebnis
	 * erzeugen. Verlustsummen-Zeile und Ergebniszeile mithilfe einer For-Schleife
	 * füllen
	 * 
	 * @param quoten             die Gerichtskostenquoten als {@link VerlusteBank}
	 * @param beteiligtenSpalten {@link List} der {@link Beteiligter}n für Zahl der
	 *                           Spalten
	 */ // TitledNumberRow
	private BegruendungsZahlenZeile fuelleVerlustsummen(VerlusteBank quoten,
			final List<List<Beteiligter>> beteiligtenSpalten) {
		// final TitledNumberRow verlusteRows = new
		// TitledNumberRow(beteiligtenSpalten.size());
		final BegruendungsZahlenZeile verlusteRows = new BegruendungsZahlenZeile(beteiligtenSpalten.size());
		for (int spalte = 0; spalte < beteiligtenSpalten.size(); spalte++) {
			if (quoten.get(beteiligtenSpalten.get(spalte)) != null) {
				verlusteRows.add(quoten.get(beteiligtenSpalten.get(spalte)));
				// Double.valueOf(.doubleValue())
			} else {
				verlusteRows.add(Euro.ZERO_CENTS);// 0.0
			}
		}
		return verlusteRows;
	}

	/**
	 * TabulierendZeilen füllen: äußere Schleife: für jede Zeile eine DoubleDataRows
	 * anlegen mit 1*String und 1..n*Double
	 * 
	 * 
	 * 
	 * @param zeilen             die {@link BegruendungsZahlenTabelle} zum Befüllen
	 * @param gruppe             eine Antragsgruppe als {@link List} von
	 *                           {@link Angriff}en
	 * @param beteiligte         die Gesamt-{@link List}e aller
	 *                           {@link Beteiligter}n, auch die nicht in den
	 *                           Bezeichner aufzunehmenden, für eine Prüfung, ob sie
	 *                           eindeutig sind, für
	 *                           {@link MehrfachBeteiligter#istEinzigerSeinerArt(Beteiligter, List)
	 * @param beteiligtenSpalten {@link List} der {@link Beteiligter}n für Zahl der
	 *                           Spalten
	 */ // BegruendungsTabelle
	private void fuelleZeilenFuerGruppe(final BegruendungsZahlenTabelle zeilen, List<Angriff> gruppe,
			final List<Beteiligter> beteiligte, final List<List<Beteiligter>> beteiligtenSpalten) {
		final List<Angriff> uebersprungen = new ArrayList<>();
		for (Angriff angriff : gruppe) {
			if (angriff.getGegner().size() > 1) {
				if (angriff.getErfolg().getCents() > 0L) {
					uebersprungen.add(angriff);
				}
				continue;
			}
			/*
			 * Die "Kopfspalte" links sagt Angriff wer gegen wen *
			 *
			 * Den ermittelten Wert in die Spalte schreiben
			 */
			zeilen.add(Angriff.benenneAngriff(angriff, beteiligte),
					fuelleEinzelverluste(beteiligtenSpalten, uebersprungen, angriff));
		}
	}

	/**
	 * Die Methode erzeugt eine {@link DoubleDataRows} mit den Spalten für die
	 * aktuelle Zeile
	 * 
	 * @param beteiligtenSpalten {@link List} der {@link List} der
	 *                           {@link Beteiligter}n für Zahl der Spalten
	 * @param uebersprungen      eine {@link List} von übersprungenen
	 *                           {@link Angriff}en für
	 *                           {@link #findeWeiterenErfolg(List, List)}
	 * @param angriff            der {@link Angriff} für
	 *                           {@link #fuelleEinzelverlust(List, Angriff, List)}
	 * @return die {@link DoubleDataRows} mit den Spalten für die aktuelle Zeile
	 */ // TitledNumberRow
	private BegruendungsZahlenZeile fuelleEinzelverluste(final List<List<Beteiligter>> beteiligtenSpalten,
			final List<Angriff> uebersprungen, Angriff angriff) {
		final BegruendungsZahlenZeile rows = new BegruendungsZahlenZeile();
		// TitledNumberRow rows = new TitledNumberRow();
		/*
		 * innere Schleife spaltenSchleife: Wert für innere Tabelle nach BeteiligtenTyp
		 * von a und b ermitteln
		 */
		for (List<Beteiligter> beteiligter : beteiligtenSpalten) {
			rows.add(fuelleEinzelverlust(uebersprungen, angriff, beteiligter));
		}
		return rows;
	}

	/**
	 * Die Methode tut Wert für innere Tabelle nach BeteiligtenTyp von a und b
	 * ermitteln
	 * 
	 * @param uebersprungen eine {@link List} von übersprungenen {@link Angriff}en
	 *                      für {@link #findeWeiterenErfolg(List, List)}
	 * @param angriff       der {@link Angriff}, aus dem die Methode den
	 *                      {@link Angriff#getGegner()} und
	 *                      {@link Angriff#getErfolg()} nimmt.
	 * @param beteiligter   eine {@link List} von {@link Beteiligter} als eine Zelle
	 *                      der {@code beteiligtenSpalten}
	 * @return ein {@link Euro}, ehemals {@code double}, für die
	 *         {@link TitledNumberRow}
	 */
	private Euro fuelleEinzelverlust(final List<Angriff> uebersprungen, Angriff angriff,
			List<Beteiligter> beteiligter) {
		if (beteiligter.equals(angriff.getGegner())) {
			/*
			 * Wenn beteiligter mit angriff#getGegner identisch ist, dann ist der Erfolg
			 * gegen den Gegner der Verlust für diese Spalte
			 */
			return angriff.getErfolg();// .doubleValue()
		} else if (beteiligter.size() > 1) {
			return findeWeiterenErfolg(beteiligter, uebersprungen, angriff.getGegner());// .doubleValue()
		} else if (beteiligter.equals(angriff.getAngreifer())) {
			/*
			 * Wenn beteiligter ein Kläger oder Widerkläger ist, dann Unterliegen berechnen
			 */
			Euro einzelErfolg = angriff.getErfolg();
			einzelErfolg = einzelErfolg.add(findeWeiterenErfolg(uebersprungen, angriff.getGegner()));
			return angriff.getAntrag().subtract(einzelErfolg);// .doubleValue()
		} else {
			return Euro.ZERO_CENTS;// 0.0
		}
	}

	/**
	 * Liste der Beteiligten/Gesamtschuldnerschaften über
	 * beteiligteInTenorReihenfolge für Zahl der Spalten ermitteln; dabei Kopfzeile
	 * füllen; die Liste enthält eine Spalte mehr als beteiligteInTenorReihenfolge,
	 * weil zuerst noch KOPFZEILE_BEGINN eingefügt wird.
	 * 
	 * Füllt außerdem die {@link BegruendungsZahlenTabelle#getColumnHeaders()}, die als
	 * {@code columnHeaders} übergeben werden.
	 * 
	 * @param columnHeaders die {@link List} von {@link String}s zum Befüllen für
	 *                      {@link BegruendungsZahlenTabelle#getColumnHeaders()}
	 * 
	 * @return die {@link List} der {@link Beteiligter}n für Zahl der Spalten
	 */
	private List<List<Beteiligter>> fuelleBeteiligtenSpalten(final List<String> columnHeaders) {
		final List<List<Beteiligter>> beteiligtenSpalten = new ArrayList<>();
		for (Angriff angriff : this.streit.angriffe) {
			/*
			 * partei muss bei Angreifern anders gebildet werden als bei Gegnern (im Punkt
			 * "gesamtschuldnerisch"), deshalb die if-Abfrage
			 */
			List<Beteiligter> spalte = null;
			Beteiligter partei = null;
			if (!beteiligtenSpalten.contains(angriff.getAngreifer())) {
				spalte = angriff.getAngreifer();
				partei = MehrfachBeteiligter.from(spalte, false);
				fuelleBeteiligtenSpalte(columnHeaders, beteiligtenSpalten, spalte, partei);
			}
			if (!beteiligtenSpalten.contains(angriff.getGegner())) {
				spalte = angriff.getGegner();
				partei = MehrfachBeteiligter.from(spalte, angriff.isGesamtSchuldnerisch());
				fuelleBeteiligtenSpalte(columnHeaders, beteiligtenSpalten, spalte, partei);
			}

		}
		return beteiligtenSpalten;
	}

	/**
	 * Die Methode erzeugt einen einzelnen Eintrag in
	 * {@link BegruendungsZahlenTabelle#getColumnHeaders()} und der {@link List} von
	 * {@link List}s aus {@link Beteiligter}
	 * 
	 * @param columnHeaders      die {@link BegruendungsZahlenTabelle#getColumnHeaders()}
	 *                           als {@link List} von {@link String}, in die der
	 *                           Eintrag aufgenommen werden soll.
	 * @param beteiligtenSpalten die {@link List} von {@link List}s aus
	 *                           {@link Beteiligter}, in die der Eintrag aufgenommen
	 *                           werden soll.
	 * @param spalte             eine aufzunehmende Spalte als {@link List} von
	 *                           {@link Beteiligter}
	 * @param partei             der {@link Beteiligter} oder
	 *                           {@link MehrfachBeteiligter}, aus dem ein Bezeichner
	 *                           für die
	 *                           {@link BegruendungsZahlenTabelle#getColumnHeaders()}
	 *                           erzeugt werden soll.
	 */
	private void fuelleBeteiligtenSpalte(final List<String> columnHeaders,
			final List<List<Beteiligter>> beteiligtenSpalten, List<Beteiligter> spalte, Beteiligter partei) {
		if (spalte != null && partei != null) {
			columnHeaders.add(partei.parteiBezeichner(GENITIV, partei.getLfdNr(),
					MehrfachBeteiligter.istEinzigerSeinerArt(partei, this.streit.beteiligte)));
			beteiligtenSpalten.add(spalte);
		}
	}

	/**
	 * Wenn in <code>uebersprungen</code> eine gesamtschuldnerische Verurteilung
	 * gegen <code>beteiligter</code> ist und <code>gegner</code> (gegen den sich
	 * der Angriff richtet) darin enthalten ist, dann gibt die Methode
	 * {@link Angriff#getErfolgCent()} zurück.
	 * 
	 * @param beteiligter   eine {@link List} aus mehreren {@link Beteiligter}n
	 * @param uebersprungen eine {@link List} von übersprungenen {@link Angriff}en
	 * @param gegner        eine {@link List} aus einem {@link Beteiligter}n
	 * @return den zusätzlichen {@link Angriff#getErfolgCent()}
	 */
	private Euro findeWeiterenErfolg(List<Beteiligter> beteiligter, List<Angriff> uebersprungen,
			List<Beteiligter> gegner) {
		return uebersprungen.stream()
				// wenn derselbe Beteiligte und alle (übrigen) Gegner in Liste
				// enthalten
				.filter(angriff -> angriff.getGegner().equals(beteiligter) && angriff.getGegner().containsAll(gegner))
				// den Erfolg ziehen
				.map(Angriff::getErfolg)
				// Erfolg zusammenrechnen
				.reduce(Euro.ZERO_CENTS, (bisher, neu) -> bisher.add(neu));
	}
//  long erfolg = 0L;
//  for (Angriff angriff : uebersprungen) {
//      if (angriff.getGegner().equals(beteiligter)
//              && angriff.getGegner().containsAll(gegner))
//          erfolg += angriff.getErfolg().getCents();
//  }
//System.out.println(erfolg.getCents() == erfolg2);
//  return Euro.ofCents(erfolg);

	/**
	 * Die Methode dient dazu, für einen <code>gegner</code> eines {@link Angriff}s
	 * den zusätzlichen Erfolg zurückzugeben, der sich aus den zusammengerechneten
	 * {@link Angriff#getErfolgCent()}en der {@link List} von übersprungenen
	 * {@link Angriff}en ergibt, die auch den referenzierten Beteiligten enthalten
	 * 
	 * @param uebersprungen eine {@link List} von übersprungenen {@link Angriff}en
	 * @param gegner        eine {@link List} aus einem {@link Beteiligter}n
	 * @return den zusätzlichen {@link Angriff#getErfolg()} als {@link Euro}
	 */
	private Euro findeWeiterenErfolg(List<Angriff> uebersprungen, List<Beteiligter> gegner) {
		return uebersprungen.stream()
				// wenn alle Gegner in Liste enthalten
				.filter(angriff -> angriff.getGegner().containsAll(gegner))
				// den Erfolg ziehen
				.map(Angriff::getErfolg)
				// Erfolg zusammenrechnen
				.reduce(Euro.ZERO_CENTS, (bisher, neu) -> bisher.add(neu));
	}
//  long erfolg = 0L;
//  for (Angriff angriff : uebersprungen) {
//      if (angriff.getGegner().containsAll(gegner))
//          erfolg += angriff.getErfolg().getCents();
//  }
//System.out.println(erfolg == erfolg2.getCents());
	// return Euro.ofCents(erfolg);

}
