/**
 * GebuehrenBerechnungsAufstellung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.ArrayList;
import java.util.Arrays;

import eu.gronos.kostenrechner.data.gebuehren.AnwaltsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenErhoehungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GerichtsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.MehrfachPauschalTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.MehrwertsteuerTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.gebuehren.GebuehrenBerechnungPruefer;

/**
 * Klasse zur Berechnung von Gebühren
 *
 * @author Peter Schuster (setrok)
 * @date 24.08.2014
 *
 */
public class GebuehrenBerechnungsAufstellung implements TenorVorbereitend/* , Begruendend, Tenorierend */ {

	private static final String BESCHREIBUNG = "Berechnung von Gebühren";
	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand", "Satz", "Prozent (%)",
			"Gebührenbetrag (EUR)" };
	private final ArrayList<GebuehrenTatbestand> values;
	private final Euro streitwert;
	private final TenorDatenContainer container;
	private BegruendungsZahlenTabelle zeilen;
	private StringBuilder gruende;
	private Pruefend<GebuehrenBerechnung> pruefer = new GebuehrenBerechnungPruefer();

	/**
	 * Der einzige Konstruktor.
	 * 
	 * @param gebuehrenBerechnung ein Parameterobjekt {@link GebuehrenBerechnung}
	 *                            mit den GebuehrenSatzTatbeständen,
	 *                            AuslagenTatbeständen und
	 *                            MehrwertsteuerTatbeständen; darf nicht null sein,
	 *                            und ein long mit dem Streitwert, muss
	 *                            größer/gleich 0 sein.
	 * @throws NullPointerException     wenn values null ist
	 * @throws IllegalArgumentException wenn streitwert kleiner 0 ist
	 */
	public GebuehrenBerechnungsAufstellung(VerfahrensDatenContainer verfahrensDaten)
			throws NullPointerException, IllegalArgumentException {
		super();
		GebuehrenBerechnung gebuehrenBerechnung = verfahrensDaten.gebuehrenBerechnung;
		pruefer.pruefeEingabe(gebuehrenBerechnung);
		/*
		 * Arrays.asList generiert eine starre Liste, die nicht erweitert werden kann;
		 * deshalb sicherheitshalber noch addAll hinterher.
		 */
		this.values = new ArrayList<GebuehrenTatbestand>();
		values.addAll(gebuehrenBerechnung.gebuehren);

		this.streitwert = gebuehrenBerechnung.streitwert;
		container = new TenorDatenContainer(verfahrensDaten);
	}

	/**
	 * Die Methode wandelt über {@link StringBuilder#toString()} die Gründe in einen
	 * {@link String} um, hier eine Aufstellung mit den Einzelwerten der Gebühren.
	 * 
	 * @return ein {@link String}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
	 */
	@Override
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * Die Methode gibt einen leeren Hauptsachetenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung() {
		EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
		hauptsache.text = "";
		return hauptsache;
	}

	/**
	 * Die Methode gibt einen leeren Kostentenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung() {
		EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();
		kostenEntscheidung.text = "";
		return kostenEntscheidung;
	}

	/**
	 * Die Methode gibt einen leeren Vollstreckbarkeitstenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeVollstreckbarkeitsEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung() {
		EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();
		vollEntscheidung.text = "";
		return vollEntscheidung;
	}

	/**
	 * Die Methode gibt statt einer wirklichen Streitwertentscheidung die
	 * Feststellung zurück, dass der Streitwert mal auf den übergebenen Betrag
	 * festgesetzt wurde. Da Hauptsacheentscheidung, Kostengrundentscheidung und
	 * Vollstreckbarkeitsentscheidung leer sind, ist das der erste Satz des
	 * Gesamttenors.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente swe = new StreitwertEntscheidungsElemente();
		swe.text = String.format(TenorTexter.STREITWERT_FESTGESETZT_VERGANGENHEIT, streitwert);// (double)
		swe.streitwerte.add(streitwert);
		return swe;
	}

	/**
	 * Die Methode gibt als {@link SonstigeEntscheidungsElemente} eine
	 * Gebührenaufstellung zurück.
	 * 
	 * @return {@link EntscheidungsElemente} für Sonstige
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public EntscheidungsElemente erzeugeSonstigeEntscheidung() {
		EntscheidungsElemente sonstige = new EntscheidungsElemente();
		sonstige.text = berrechneGebuehrenAufstellung();
		return sonstige;
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return einen befüllten {@link TenorDatenContainer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		container.allgemein.selectedPanel = GebuehrenBerechnung.TAB_ID;
		return container;
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Die Methode soll die Summe aller Gebühren der Tabelle liefern.
	 * 
	 * @return einen String mit der Aufstellung aller GebührenTatbestände und ihres
	 *         Betrags für diesen Streitwert
	 */
	private String berrechneGebuehrenAufstellung() {
		Euro summe = Euro.ZERO_CENTS;// double summe = 0;//
		Euro rvgsumme = Euro.ZERO_CENTS;// double rvgsumme = 0;
		zeilen = new BegruendungsZahlenTabelle();
		zeilen.add(Arrays.asList(COLUMN_HEADERS));
		for (GebuehrenTatbestand gt : values) {
//			final DoubleDataRows doubles = new DoubleDataRows(3);
			final BegruendungsZahlenZeile zahlen = new BegruendungsZahlenZeile(3);
			/*
			 * errechneGebuehr ignoriert MehrwertsteuerTatbestände; diese muss deshalb extra
			 * gerechnet werden.
			 */
			if (gt instanceof MehrwertsteuerTatbestand) {
				/* Bei Mehrwertsteuer ist das erste Double-Feld (der Gebührensatz) leer (0.0) */
				// doubles.add(Double.valueOf(0.0));// new Double(0.0)
				zahlen.add(Double.valueOf(0.0));
				final MehrwertsteuerTatbestand mwst = (MehrwertsteuerTatbestand) gt;
				final Euro einzelposition = mwst.errechneSteuer(rvgsumme);// double
				/* Ins zweite Double-Feld kommt die Prozentangabe */
				// doubles.add(new Double(mwst.getSteuerSatz() * 100.0));
				zahlen.add(mwst.getSteuerSatz() * 100.0);
				// summe += einzelposition;
				summe = summe.add(einzelposition);
				/*
				 * Ins dritte Feld (EUR) kommt die Einzelposition als Double - geht nicht als
				 * Euro
				 */
				// doubles.add(einzelposition.doubleValue());// new Double(einzelposition)
				zahlen.add(einzelposition);
				/*
				 * Wenn die Gebühren isoliert werden, die RVG-Summe wieder auf 0 setzen.
				 */
				if (mwst.isIsoliert())
					rvgsumme = Euro.ZERO_CENTS;// 0.0;
				/* Und die Zeile in die Tabelle nehmen */
				zeilen.add(gt.langBezeichnung(), zahlen);// doubles);// getBezeichnung()
			} else {
				/* summe um die Gebühr erhöhen */
				final Euro einzelposition = GebuehrenAuflistung.errechneGebuehr(gt, streitwert);// double
				String ergaenzung = "";
				if (gt instanceof MehrfachPauschalTatbestand)
					ergaenzung = String.format(" x %d %s", ((MehrfachPauschalTatbestand) gt).getAnzahl(),
							((MehrfachPauschalTatbestand) gt).getEinheit());
				if (gt instanceof GebuehrenErhoehungsTatbestand)
					ergaenzung = String.format(" x %d", ((GebuehrenErhoehungsTatbestand) gt).getAnzahl());
				if (gt instanceof GebuehrenSatzTatbestand) {
					//
				}
				if (gt instanceof GebuehrenAnrechnungsTatbestand) {
					//
				}
				if (gt instanceof GebuehrenSatzTatbestand) {
					/* Bei allem, was einen Satz hat, kommt der in die erste Double-Spalte */
					// doubles.add(new Double(((GebuehrenSatzTatbestand) gt).getSatz()));
					zahlen.add(((GebuehrenSatzTatbestand) gt).getSatz());
				} else {
					/* Sonst kommt da 0.0 rein, weil es hier kein Steuer-TB ist */
//					doubles.add(0.0);// new Double(0.0)
					zahlen.add(Double.valueOf(0.0));
				}
				/* die zweite Spalte (%) ist auf jeden Fall 0.0 */
				//doubles.add(new Double(0.0));
				zahlen.add(Double.valueOf(0.0));
				summe = summe.add(einzelposition);// summe += einzelposition;
				//doubles.add(einzelposition.doubleValue());// new Double(einzelposition)
				zahlen.add(einzelposition);
				/*
				 * rvgsumme nur erhöhen, wenn RVG-Gebühr (wird für MwSt gebraucht).
				 */
				if (gt.getGebuehrenKlasse() == AnwaltsGebuehrenTabelle.class)
					rvgsumme = rvgsumme.add(GebuehrenAuflistung.errechneGebuehr(gt, streitwert));
				// rvgsumme += GebuehrenAuflistung.errechneGebuehr(gt, streitwert);
				/* Und die Zeile in die Tabelle nehmen */
				zeilen.add(gt.langBezeichnung() + ergaenzung, zahlen);//doubles);
			}
		}
		//final DoubleDataRows doublesumme = new DoubleDataRows(3);
		final BegruendungsZahlenZeile summenzahlen = new BegruendungsZahlenZeile();
//		doublesumme.add(0.0);// new Double(0.0)
//		doublesumme.add(0.0);// new Double(0.0)
		summenzahlen.add(Double.valueOf(0.0));
		summenzahlen.add(Double.valueOf(0.0));
		//doublesumme.add(new Double(summe.doubleValue()));
		summenzahlen.add(summe);
		zeilen.add("Gesamt", summenzahlen);//doublesumme);

		final String summenFormat = "Danach fallen Gebühren in Höhe von insgesamt %s EUR an.%n";// %,.2f
		baueGruende();
		return String.format(summenFormat, summe);
	}

	/**
	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
	 */
	private void baueGruende() {
		gruende = new StringBuilder();
		final GebuehrenSatzTatbestand vv = new GebuehrenSatzTatbestand("", "", 1.0, AnwaltsGebuehrenTabelle.class);
		final GebuehrenSatzTatbestand kv = new GebuehrenSatzTatbestand("", "", 1.0, GerichtsGebuehrenTabelle.class);
		gruende.append(String.format(
				"Bei einem Streitwert von %s EUR beträgt eine Gerichtsgebühr %s EUR und eine Anwaltsgebühr %s EUR. %nAufstellung: %n",
				streitwert, GebuehrenAuflistung.errechneGebuehr(kv, streitwert), // %,.2f (double) %,.2f %,.2f
				GebuehrenAuflistung.errechneGebuehr(vv, streitwert)));
		zeilen.toStringBuilder(gruende);
	}
}