/**
 * MehrkostenMethode.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRueckgabe;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.logic.TenorToken;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.VerlusteBank;

/**
 * MehrkostenMethode berechnet das Unterliegen des Klägers dadurch, dass die
 * Mehrkosten, die auf den zurückgenommenen Teil entfallen, errechnet und diese
 * in Relation zu den tatsächlichen entstandenen Kosten des Rechtsstreits
 * gesetzt werden (sog. Mehrkostenmethode, Anders/Gehle, A-198, dies. Antrag und
 * Entscheidung im Zivilprozess, Teil B Rn. 404, 410 ff.).
 * 
 * @author Peter Schuster (setrok)
 * @date 25.04.2014
 * 
 */
public class MehrkostenMethode extends StreitwertReduktion {

	public MehrkostenMethode(VerfahrensDatenContainer verfahrensDaten
	/* , StreitwertReduktionen swrn */) throws IllegalArgumentException {
		super(verfahrensDaten/* , swrn */);
	}

	@Override
	public String getBerechnungsTyp() {
		return "Tenor nach der sog. Mehrkostenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss).";
	}

	/**
	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen. Berücksichtigt
	 * jetzt auch, dass der Kläger nach Teilklagerücknahme noch mit einem weiteren
	 * Teil unterliegen kann...
	 * 
	 * @return
	 * 
	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
	 *         Unterliegen (der Klägerpartei) ausdrückt. TODO sollte
	 *         {@link Fraction} zurückgeben.
	 * 
	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#berechneUnterliegensQuote()
	 */
	@Override // double
	protected Fraction berechneUnterliegensQuote() {
		final Fraction klVerlustTkn = unterliegenTeilKlageruecknahme();
		final int i = 2;
		Fraction gewichtungVerbleibend = gewichtungVerbleibend(klVerlustTkn, i, 0);
		/*
		 * Dann auch die Werte für den verbliebenen Streitgegenstand den Summen
		 * hinzurechnen. gewichtungVerbleibend = s.o., entweder 1.0 oder gemäß Anteil am
		 * Rechtsstreit
		 */
		Fraction sonstigerVerlust = verteilung(i - 1).getFractionFor(klaegerlein).multiply(gewichtungVerbleibend);

		super.zeilen = baueBegruendungsTabelle(verteilung(0), gewichtungVerbleibend, sonstigerVerlust);
		super.starteGruende(baueGruende(verteilung(0), super.zeilen));
		// Abwägung muss hinein
		return klVerlustTkn.add(sonstigerVerlust);// .doubleValue();
	}

	/**
	 * Die Methode dient dazu, die Unterliegensquote nach Maßgabe der dem
	 * Konstruktor übergebenen Werte zu berechnen.
	 * 
	 * @return eine {@link Fraction}, die die Unterliegensquote (der Klägerseite)
	 *         bezogen auf eine Teilklagerücknahme bedeutet.
	 */
	private Fraction unterliegenTeilKlageruecknahme() {
		/*
		 * Äußere Zähler-Schleife: Anhand der Liste swrn.reduktionen eine Schleife
		 * durchlaufen, war früher alles vorher, dann alles nachher. De facto wird die
		 * Schleife genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben,
		 * ist doof. :)
		 */
		for (int i = 0; i < reduktionen().size(); i++) {
			List<GebuehrenTatbestand> tatbestaende = tatbestaende(i);
			if (tatbestaende != null) {
				/* Alle Gebührentatbestände für diesen Streitwert durchgehen */
				for (GebuehrenTatbestand gt : tatbestaende) {
					final GebuehrenRueckgabe durchgang = gebuehrenZeile(i, gt);
					/*
					 * verarbeitungen[0] enthält die Differenz zwischen den ersten beiden Elementen
					 */
					verteilung(0).add(beklagte, durchgang.unterliegensBetragBk);
					verteilung(0).add(klaegerlein, durchgang.unterliegensBetragKl);
				}
			}
		}
		final Fraction klVerlustTkn = verteilung(0).getFractionFor(this.klaegerlein);
		return klVerlustTkn;
	}

	/**
	 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
	 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
	 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 7.
	 * 
	 * @param verluste              die {@link VerlusteBank}
	 * @param gewichtungVerbleibend eine {@link Fraction} dazu, wie der verbleibende
	 *                              Teil gewichtet wird
	 * @param sonstigerVerlust      die zusätzliche Verlustquote als
	 *                              {@link Fraction} hinsichtlich des verbliebenen
	 *                              Streitgegenstands mit faktor.
	 * @return die zusammengestellten {@link BegruendungsZahlenTabelle}
	 */
	private BegruendungsZahlenTabelle baueBegruendungsTabelle(VerlusteBank verluste, Fraction gewichtungVerbleibend,
			Fraction sonstigerVerlust) {
		/*
		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 7.
		 */
		final BegruendungsZahlenTabelle zeilen = new BegruendungsZahlenTabelle();
//		final BegruendungsTabelle zeilen = new BegruendungsTabelle();
		zeilen.add(Arrays.asList("Gebührentatbestand", "Tatsächlicher Streitwert (EUR)",
				"Tatsächliche Gebührenhöhe (EUR)", "Reduzierter Streitwert (EUR)", "Fiktive Gebührenhöhe (EUR)",
				"Mehrkosten (EUR)", "Differenz (%)"));

		/*
		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
		 * doof. :)
		 */
		for (int i = 0; i < reduktionen().size() - 1; i++) {
			List<GebuehrenTatbestand> tatbestaende = tatbestaende(i);
			if (tatbestaende != null) {
				/* Alle Gebührentatbestände für diesen Streitwert durchgehen */
				for (GebuehrenTatbestand gt : tatbestaende) {
					// TODO besser als Stream?
					// final TitledNumberRow numbers = new TitledNumberRow(6);
					final BegruendungsZahlenZeile numbers = new BegruendungsZahlenZeile(6);
					final GebuehrenRueckgabe durchgang = gebuehrenZeile(i, gt);
					/* Tatsächlicher Streitwert und tatsächliche Gebührenhöhe */
					numbers.add(streitwert(i));
					numbers.add(durchgang.gebuehrTatsaechlich);
					/* Reduzierter Streitwert und fiktive Gebührenhöhe */
					numbers.add(streitwert(1));
					numbers.add(durchgang.unterliegensBetragBk);
					numbers.add(durchgang.unterliegensBetragKl);
					/* Mehrkosten und prozentuale Differenz */
					if (durchgang.gebuehrTatsaechlich.getCents() == 0L)
						numbers.add(Fraction.ZERO);
					else
						numbers.add(durchgang.unterliegensBetragKl.divideAsFraction(durchgang.gebuehrTatsaechlich));
//                                Fraction.reducedOf(
//                                durchgang.unterliegensBetragKl.getCents(),
//                                durchgang.gebuehrTatsaechlich.getCents())
					/* Und die Zeile in die Tabelle nehmen */
					zeilen.add(gt.langBezeichnung(), numbers);
				}
			}
		}
		/*
		 * Zum Schluss noch eine Zeile für den verbliebenen Streitgegenstand. Falls von
		 * dem reduzierten Streitwert dann nicht alles zugesprochen wurde, ist der
		 * Verlust noch einmal höher...
		 */
		final int i = 2;
		if (zusatz(i).begruendet.lessThan(differenz(i - 1))) {
			final String zwischenUeberschrift = "Zwischensumme";
			final String verbliebeneUeberschrift = String.format("Verbliebener Streitgegenstand (gewichtet zu %,.2f%%)",
					gewichtungVerbleibend.toPercent());
			/*
			 * TODO (doch richtig?) Wenn wir aufs Kosteninteresse abstellen müssen, muss das
			 * auch in der Tabelle stehen); dafür wird nicht nach Anteilen gewichtet (faktor
			 * = 1.0) TODO (doch richtig?) Wenn wir nicht aufs Kosteninteresse abstellen,
			 * dann müssen die Zwischensumme und der verbliebene Streitgegenstand gemäß der
			 * Anteile am Rechtsstreit gewichtet werden. Das Gewichten geht über die
			 * Multiplikation mit anteileAnRechtsstreit...
			 */
			/*
			 * Damit die ausgegebene Tabelle (halbwegs) nachvollziehbar bleibt, brauchen wir
			 * eine Zwischensummen-Zeile
			 */
			// final DoubleDataRows zwischenWerte = new DoubleDataRows(6);
			// final TitledNumberRow zwischenZeile = new TitledNumberRow(6);
			final BegruendungsZahlenZeile zwischenZeile = new BegruendungsZahlenZeile(6);
			zwischenZeile.add(verluste.getFiktiverStreitwert());
			zwischenZeile.add(streitwert(1));
			zwischenZeile.add(verluste.get(beklagte));
			zwischenZeile.add(verluste.get(klaegerlein));
			zwischenZeile.add(verluste.getFractionFor(klaegerlein));
			zeilen.add(zwischenUeberschrift, zwischenZeile);
			/*
			 * Dann auch die Zeile für den verbliebenen Streitgegenstand erzeugen. faktor =
			 * s.o., entweder 1.0 oder gemäß Anteil am Rechtsstreit
			 */
//			final TitledNumberRow verbliebenZeile = new TitledNumberRow(6);
			final BegruendungsZahlenZeile  verbliebenZeile = new BegruendungsZahlenZeile(6);
			verbliebenZeile.add(streitwert(1));
			verbliebenZeile.add(streitwert(1));
			verbliebenZeile.add(zusatz(i).begruendet);
			verbliebenZeile.add(zusatz(i).begruendet);
			verbliebenZeile.add(differenz(i - 1).subtract(zusatz(i).begruendet));
			verbliebenZeile.add(sonstigerVerlust);
			zeilen.add(verbliebeneUeberschrift, verbliebenZeile);
		}
		/* Und die Fußzeile muss hinzu... */
//		final TitledNumberRow gesamtWerte = new TitledNumberRow(6);
		final BegruendungsZahlenZeile   gesamtWerte = new BegruendungsZahlenZeile(6);
		gesamtWerte.add(streitwert(0));
		/*
		 * Fiktiver Streitwert enthält die Summe aller tatsächlich angefallenen Gebühren
		 * ... und hinsichtlich des verbliebenen Streitgegenstands mit faktor.
		 */
		gesamtWerte.add(verluste.getFiktiverStreitwert());
		gesamtWerte.add(streitwert(1));
		gesamtWerte.add(verluste.get(beklagte));
		gesamtWerte.add(verluste.get(klaegerlein));
		gesamtWerte.add(verluste.getFractionFor(klaegerlein).add(sonstigerVerlust));
		zeilen.add("Gesamtsumme", gesamtWerte);

		return zeilen;
	}

	/**
	 * Die Methode errechnet die Differenz, die der Kläger durch seine anfängliche
	 * Gier verursacht hat, also die Mehrkosten für diese Gebühr.
	 * 
	 * @param zeile index der Streitwerte-Liste
	 * @param gt    ein {@link GebuehrenTatbestand}
	 * @return eine {@link GebuehrenRueckgabe}
	 */
	private GebuehrenRueckgabe gebuehrenZeile(int zeile, GebuehrenTatbestand gt) {
		final GebuehrenRueckgabe durchgang = new GebuehrenRueckgabe();
		/*
		 * Welche Gebühr ist tatsächlich angefallen? Die tatsächlichen Kosten beinhalten
		 * die vor der Reduktion angefallenen Gebühren nach dem alten, höheren
		 * Streitwert, also streitwerte[i], und die danach entstandenen Gebühren nach
		 * dem neuen, niedrigeren Streitwert.
		 */
		durchgang.gebuehrTatsaechlich = // Euro.ofEuros(.longValue())
				GebuehrenAuflistung.errechneGebuehr(gt, streitwert(zeile));
		/*
		 * Welche Gebühr wäre bei dem niedrigeren Streitwert angefallen? Die Kosten, die
		 * bei anfänglichem Einklagen nur des verbleibenden Betrags entstandenen wären
		 * (fiktive). Das sind die vor der Reduktion angefallenen Gebühren und die
		 * danach entstandenen, beide nach dem niedrigeren Streitwert, also
		 * streitwerte[1].
		 */
		durchgang.unterliegensBetragBk = GebuehrenAuflistung.errechneGebuehr(gt, super.swrn.streitwerte.get(1));
		// Euro .ofEuros()
		/*
		 * Die Differenz hat der Kläger durch seine anfängliche Gier verursacht, also
		 * die Mehrkosten für diese Gebühr.
		 */
		durchgang.unterliegensBetragKl = durchgang.gebuehrTatsaechlich.subtract(durchgang.unterliegensBetragBk);
		return durchgang;
	}

	/**
	 * Die Methode dient dazu, über
	 * {@link BegruendungsZahlenTabelle#toStringBuilder(StringBuilder)} die Gründe zu
	 * basteln.
	 * 
	 * @param verluste die {@link VerlusteBank}
	 * @param zeilen   die zusammengestellten {@link BegruendungsZahlenTabelle}
	 * 
	 * @return einen {@link StringBuilder}
	 */
	private StringBuilder baueGruende(VerlusteBank verluste, BegruendungsZahlenTabelle zeilen) {
		final StringBuilder gruende = new StringBuilder(TenorToken.GRUENDE_EINLEITUNG);
		if (verluste.getQuotaFor(klaegerlein) == 0.0 || verluste.getQuotaFor(klaegerlein) == 0.0) {
			gruende.append(TenorToken.GRUENDE_EINLEITUNG_91);
			gruende.append("\n");// TODO
		} else {
			gruende.append(TenorToken.GRUENDE_EINLEITUNG_92_269);
			gruende.append(TenorToken.WEITER_GRUENDE_MEHRK);
			gruende.append(TenorToken.PUNKT_ABSATZ);
			gruende.append(TenorToken.GRUENDE_BERECHNUNG_MEHRKOSTEN);
			// streitwerte[1] > verurteilung .getCents() > .getCents()
			if (super.swrn.streitwerte.get(1).greaterThan(reduktionen().get(2).zusatz.begruendet)) {
				// TODO zu stark verdrahtet
				gruende.append(TenorToken.GRUENDE_VERBLIEBENER_GEGENSTAND);
				gruende.append(TenorToken.GRUENDE_VERBLIEBENER_GEGENSTAND_SONST);
			}
			gruende.append(TenorToken.GRUENDE_EINLEITUNG_TABELLE);
			gruende.append(TenorToken.DOPPELPUNKT_ABSATZ);
		}
		return zeilen.toStringBuilder(gruende);
	}

//	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand",
//			"Tatsächlicher Streitwert (EUR)", "Tatsächliche Gebührenhöhe (EUR)", "Reduzierter Streitwert (EUR)",
//			"Fiktive Gebührenhöhe (EUR)", "Mehrkosten (EUR)", "Differenz (%)" };
//	private static final String BESCHREIBUNG = "Tenor nach der sog. Mehrkostenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss).";
//	private static final String WEITER_GRUENDE = "der sogenannten Mehrkostenmethode.\n";
//	private static final String GRUENDE_BERECHNUNG_MEHRKOSTEN = "Danach ist bei einer teilweisen Klagerücknahme die Kostenquote dadurch zu ermitteln, dass die durch den zurückgenommenen Teil entstandenen Mehrkosten in Relation zu den tatsächlich entstandenen Kosten des Rechtsstreits zu setzen sind. ";
//	private static final String GRUENDE_EINLEITUNG_TABELLE = "Im einzelnen errechnen sich diese Punkte wie folgt:\n";
//	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND = "Da die Klage hinsichtlich des verbliebenen Streitgegenstands nicht vollständig erfolgreich war";
//	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND_WERTVOLLER = " und der Wert des verbliebenen Streitwerts die angefallenen Kosten des Rechtsstreits übersteigt, ist für den zurückgenommenen Teil das Kosteninteresse anzusetzen. ";
//	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND_SONST = ", ist auch das Unterliegen diesbezüglich zu berücksichtigen. ";
//	/**
//	 * Soll für den zurückgenommenen Teil nur auf das Kosteninteresse abgestellt
//	 * werden?
//	 */
//	private boolean aufKosteninteresseAbstellen = false;
//
//	/**
//	 * Der Konstruktor ruft den Konstruktor der Oberklasse auf.
//	 * 
//	 * 
//	 * @param teilklageruecknahme Parameterojbkekt vom Typ
//	 *                            {@link Teilklageruecknahme}, das enthält die
//	 *                            {@link Teilklageruecknahme#parteien} klaeger
//	 *                            Informationen zur Klägerseite als Beteiligter
//	 *                            (wird zur Tenorierung benötigt) und beklagter
//	 *                            Informationen zur Beklagten als Beteiligter (wird
//	 *                            zur Tenorierung benötigt), die Arrays
//	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
//	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
//	 *                            erst nach der "Streitwertreduktion"
//	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
//	 *                            Gebühren, die
//	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
//	 *                            eine ArrayList<Long> mit streitwerteUndObsiegen[0]
//	 *                            als dem alten (höheren) Streitwert,
//	 *                            streitwerteUndObsiegen[1] als dem neuen,
//	 *                            niedrigeren Streitwert und die
//	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
//	 *                            Höhe der Kläger letzlich obsiegt hat.
//	 * 
//	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
//	 *                            b92ii Soll § 92 II ZPO angewandt werden
//	 *                            (verhältnismäßig geringe Kostentragungen werden
//	 *                            unterschlagen)
//	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
//	 *                                  folgenden Plausibilitäten verletzt ist: a)
//	 *                                  der frühere Streitwert muss größer als der
//	 *                                  spätere (oder gleich) sein, b) der spätere
//	 *                                  Streitwert muss größer als die Verurteilung
//	 *                                  (oder gleich) sein.
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#StreitwertReduktion(Allgemein,
//	 *      Teilklageruecknahme)
//	 * 
//	 */
//	public MehrkostenMethode(VerfahrensDatenContainer vefahrensDaten) throws IllegalArgumentException {
//		super(vefahrensDaten);
//	}
//
//	/**
//	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getBerechnungsTyp()
//	 */
//	@Override
//	public String getBerechnungsTyp() {
//		return BESCHREIBUNG;
//	}
//
//	/**
//	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
//	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen. Berücksichtigt
//	 * jetzt auch, dass der Kläger nach Teilklagerücknahme noch mit einem weiteren
//	 * Teil unterliegen kann...
//	 * 
//	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
//	 *         Unterliegen ausdrückt.
//	 * 
//	 */
//	@Override
//	protected double berechneUnterliegensQuote() {
//		Euro tatsaechlicheSumme = Euro.ZERO_CENTS;// double 0.0
//		Euro fiktiveSumme = Euro.ZERO_CENTS;// double = 0.0;
//		Euro mehrkosten = Euro.ZERO_CENTS; // double = 0.0;
//		/*
//		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
//		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
//		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 7.
//		 */
//		zeilen = new TabulierendZeilen();
//		zeilen.add(Arrays.asList(COLUMN_HEADERS));
//		/*
//		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
//		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
//		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
//		 * doof. :)
//		 */
//		for (int i = 0; i < streitwerte.length; i++) {
//			List<GebuehrenTatbestand> tatbestaende = i == 0 ? vorher : nachher;
//			if (tatbestaende != null) {
//				/* Alle Gebührentatbestände für diesen Streitwert durchgehen */
//				for (GebuehrenTatbestand gt : tatbestaende) {
//					final DoubleDataRows doubles = new DoubleDataRows(6);
//					/*
//					 * Welche Gebühr ist tatsächlich angefallen? Die tatsächlichen Kosten beinhalten
//					 * die vor der Reduktion angefallenen Gebühren nach dem alten, höheren
//					 * Streitwert und die danach entstandenen Gebühren nach dem neuen, niedrigeren
//					 * Streitwert. Ausrechnen und zur Summe hinzurechnen.
//					 */
//					doubles.add(streitwerte[i].doubleValue());// (new Double(streitwerte[i]));
//					final Euro gebuehrTatsaechlich = GebuehrenAuflistung.errechneGebuehr(gt, streitwerte[i]);//
//					doubles.add(gebuehrTatsaechlich.doubleValue());// (new Double(gebuehrTatsaechlich));
//					tatsaechlicheSumme = tatsaechlicheSumme.add(gebuehrTatsaechlich);// ERROR (tatsaechlicheSumme); ?
//					// summeTatsaechlicheEUR += gebuehrTatsaechlich;
//					/*
//					 * Welche Gebühr wäre bei dem niedrigeren Streitwert angefallen? Die Kosten, die
//					 * bei anfänglichem Einklagen nur des verbleibenden Betrags entstandenen wären
//					 * (fiktive). Das sind die vor der Reduktion angefallenen Gebühren und die
//					 * danach entstandenen, beide nach dem niedrigeren Streitwert, also
//					 * streitwerte[1]. Ausrechnen und zur Summe hinzurechnen.
//					 */
//					doubles.add(streitwerte[1].doubleValue());// (new Double(streitwerte[1]));
//					final Euro gebuehrFiktiv = GebuehrenAuflistung.errechneGebuehr(gt, streitwerte[1]);// double
//					doubles.add(gebuehrFiktiv.doubleValue());// (new Double(gebuehrFiktiv));
//					// summeFiktiveEUR += gebuehrFiktiv;
//					fiktiveSumme = fiktiveSumme.add(gebuehrFiktiv);
//					/*
//					 * Die Differenz hat der Kläger durch seine anfängliche Gier verursacht, also
//					 * die Mehrkosten für diese Gebühr. Diese dann auch zur Summe der Mehrkosten
//					 * hinzurechnen.
//					 */
//					// double differenzEUR = gebuehrTatsaechlich - gebuehrFiktiv;
//					final Euro differenzEUR = gebuehrTatsaechlich.subtract(gebuehrFiktiv);
//					// mehrkostenEUR += differenzEUR;
//					mehrkosten = mehrkosten.add(differenzEUR);
//					System.out.println("Mehrkosten:" + mehrkosten);
//					doubles.add(differenzEUR.doubleValue());// (new Double(differenzEUR));
//					final double differenzQuote = differenzEUR.divide(gebuehrTatsaechlich);// differenzEUR /
//																							// gebuehrTatsaechlich;
//					doubles.add(differenzQuote * 100.0 /* Prozent */);// new Double()
//					/* Und die Zeile in die Tabelle nehmen */
//					zeilen.add(gt.langBezeichnung(), doubles);// toString()
//				}
//			}
//		}
//		/*
//		 * Zum Schluss noch eine Zeile für den verbliebenen Streitgegenstand. Falls von
//		 * dem reduzierten Streitwert dann nicht alles zugesprochen wurde, ist der
//		 * Verlust noch einmal höher...
//		 */
//		double unterliegensQuote;
//		// verurteilung < streitwerte[1]
//		if (verurteilung.lessThan(streitwerte[1])) {
//			final double anteileAnRechtsstreit = super.berechneAnteileAnRechtsstreit()[0];
//			/*
//			 * Böse Trickserei: Wenn der Kläger gar nicht gewonnen hat oder der verbliebene
//			 * Streitgegenstand wertvoller ist als die tatsächlich angefallenen Kosten, soll
//			 * sich die Verteilung im wesentlich nach dem verbliebenen Streitgegenstand
//			 * richten. Das wird dadurch erreicht, dass die Mehrkosten den tatsächlichen
//			 * Kosten gleichgesetzt werden. So zählt der zurückgenommene Anteil soviel wie
//			 * seine Kosten.
//			 */
//			aufKosteninteresseAbstellen = verurteilung.equals(Euro.ZERO_CENTS)
//					|| streitwerte[1].greaterThan(tatsaechlicheSumme);
//			// verurteilung == 0L || streitwerte[1] > summeTatsaechlicheEUR;
//			String zwischenUeberschrift;
//			String verbliebeneUeberschrift;
//			double faktor;
//			if (aufKosteninteresseAbstellen) {
//				/*
//				 * Wenn wir aufs Kosteninteresse abstellen müssen, muss das auch in der Tabelle
//				 * stehen); dafür wird nicht nach Anteilen gewichtet (faktor = 1.0)
//				 */
//				mehrkosten = tatsaechlicheSumme;
//				zwischenUeberschrift = "Zwischenergebnis (Mehrkosten = Kosteninteresse)";
//				faktor = 1.0;
//				verbliebeneUeberschrift = "Verbliebener Streitgegenstand";
//			} else {
//				/*
//				 * Wenn wir nicht aufs Kosteninteresse abstellen, dann müssen die Zwischensumme
//				 * und der verbliebene Streitgegenstand gemäß der Anteile am Rechtsstreit
//				 * gewichtet werden. Das Gewichten geht über die Multiplikation mit
//				 * anteileAnRechtsstreit...
//				 */
//				// mehrkostenEUR *= anteileAnRechtsstreit;
//				mehrkosten = mehrkosten.multiply(anteileAnRechtsstreit);
//				// summeTatsaechlicheEUR *= anteileAnRechtsstreit;
//				tatsaechlicheSumme = tatsaechlicheSumme.multiply(anteileAnRechtsstreit);
//				// summeFiktiveEUR *= anteileAnRechtsstreit;
//				fiktiveSumme = fiktiveSumme.multiply(anteileAnRechtsstreit);
//				zwischenUeberschrift = String.format("Zwischensumme (gewichtet zu %,.2f%%)",
//						anteileAnRechtsstreit * 100);
//				/*
//				 * ... und hinsichtlich des verbliebenen Streitgegenstands mit faktor.
//				 */
//				faktor = super.berechneAnteileAnRechtsstreit()[1];
//				verbliebeneUeberschrift = String.format("Verbliebener Streitgegenstand (gewichtet zu %,.2f%%)",
//						faktor * 100);
//			}
//			/*
//			 * Damit die ausgegebene Tabelle (halbwegs) nachvollziehbar bleibt, brauchen wir
//			 * eine Zwischensummen-Zeile
//			 */
//			unterliegensQuote = mehrkosten.divide(tatsaechlicheSumme);// mehrkostenEUR / summeTatsaechlicheEUR;
//			final DoubleDataRows zwischenWerte = new DoubleDataRows(6);
//			zwischenWerte.add(streitwerte[0].doubleValue());// (new Double(streitwerte[0]));
//			zwischenWerte.add(tatsaechlicheSumme.doubleValue());// (new Double(summeTatsaechlicheEUR));
//			zwischenWerte.add(streitwerte[1].doubleValue());// (new Double(streitwerte[1]));
//			zwischenWerte.add(fiktiveSumme.doubleValue());// (new Double(summeFiktiveEUR));
//			zwischenWerte.add(mehrkosten.doubleValue());// (new Double(mehrkostenEUR));
//			zwischenWerte.add(unterliegensQuote * 100.0 /* Prozent */);// (new Double(unterliegensQuote * 100.0 /*
//																		// Prozent */));
//			zeilen.add(zwischenUeberschrift, zwischenWerte);
//			/*
//			 * Dann auch die Werte für den verbliebenen Streitgegenstand den Summen
//			 * hinzurechnen. faktor = s.o., entweder 1.0 oder gemäß Anteil am Rechtsstreit
//			 */
//			// summeTatsaechlicheEUR += ((double) streitwerte[1]) * faktor;
//			tatsaechlicheSumme = tatsaechlicheSumme.add(streitwerte[1].multiply(faktor));
//			System.out.println("tatsaechlicheSumme: " + tatsaechlicheSumme);
//			// summeFiktiveEUR += (double) verurteilung * faktor;
//			// double (double) (streitwerte[1] - verurteilung);
//			final Euro restUnterliegen = streitwerte[1].subtract(verurteilung);
//			// mehrkostenEUR += restUnterliegen * faktor;
//			mehrkosten = mehrkosten.add(restUnterliegen.multiply(faktor));
//			final double restQuote = restUnterliegen.divide(streitwerte[1]);
//			// restUnterliegen / (double) streitwerte[1];
//			/* Und die Zeile erzeugen */
//			final DoubleDataRows verbliebeneWerte = new DoubleDataRows(6);
//			verbliebeneWerte.add(streitwerte[1].doubleValue());// (new Double(streitwerte[1]));
//			verbliebeneWerte.add(streitwerte[1].doubleValue());// (new Double(streitwerte[1]));
//			verbliebeneWerte.add(verurteilung.doubleValue());// (new Double(verurteilung));
//			verbliebeneWerte.add(verurteilung.doubleValue());// (new Double(verurteilung));
//			verbliebeneWerte.add(restUnterliegen.doubleValue());// (new Double(restUnterliegen));
//			verbliebeneWerte.add(restQuote * 100.0 /* Prozent */);// (new Double(restQuote * 100.0 /* Prozent */));
//			zeilen.add(verbliebeneUeberschrift, verbliebeneWerte);
//		}
//		/* Und die Fußzeile muss hinzu... */
//		unterliegensQuote = mehrkosten.divide(tatsaechlicheSumme);// mehrkostenEUR / summeTatsaechlicheEUR;
//		System.out.println("Mehrkosten:" + mehrkosten + ";tatsaechlicheSumme:" + tatsaechlicheSumme);
//		final DoubleDataRows gesamtWerte = new DoubleDataRows(6);
//		gesamtWerte.add(streitwerte[0].doubleValue());// (new Double(streitwerte[0]));
//		gesamtWerte.add(tatsaechlicheSumme.doubleValue());// (new Double(summeTatsaechlicheEUR));
//		gesamtWerte.add(streitwerte[1].doubleValue());// (new Double(streitwerte[1]));
//		gesamtWerte.add(fiktiveSumme.doubleValue());// (new Double(summeFiktiveEUR));
//		gesamtWerte.add(mehrkosten.doubleValue());// (new Double(mehrkostenEUR));
//		gesamtWerte.add(unterliegensQuote * 100.0 /* Prozent */);
//		// (new Double(unterliegensQuote * 100.0 /* Prozent // */));
//		zeilen.add("Gesamtsumme", gesamtWerte);
//		baueGruende(unterliegensQuote);
//		return unterliegensQuote;
//	}
//
//	/**
//	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
//	 * 
//	 * @param unterliegensQuote ein double zwischen 0.0 und 1.0, der in Prozent
//	 *                          umgerechnet das Unterliegen ausdrückt.
//	 */
//	private void baueGruende(double unterliegensQuote) {
//		if (unterliegensQuote == 0.0 || unterliegensQuote == 1.0) {
//			super.starteGruende(GRUENDE_EINLEITUNG_91);
//			super.erweitereGruende("\n");// TODO
//		} else {
//			super.starteGruende(GRUENDE_EINLEITUNG_92_269 + WEITER_GRUENDE);
//			super.erweitereGruende(GRUENDE_BERECHNUNG_MEHRKOSTEN);
//			// streitwerte[1] > verurteilung
//			if (streitwerte[1].greaterThan(verurteilung)) {
//				if (aufKosteninteresseAbstellen)
//					super.erweitereGruende(
//							GRUENDE_VERBLIEBENER_GEGENSTAND + GRUENDE_VERBLIEBENER_GEGENSTAND_WERTVOLLER);
//				else
//					super.erweitereGruende(GRUENDE_VERBLIEBENER_GEGENSTAND + GRUENDE_VERBLIEBENER_GEGENSTAND_SONST);
//			}
//			super.erweitereGruende(GRUENDE_EINLEITUNG_TABELLE);
//		}
//		zeilen.toStringBuilder(gruende);
//	}

}