/**
 * QuotenMethode.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRueckgabe;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.logic.TenorToken;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.VerlusteBank;

/**
 * QuotenMethode berechnet das Unterliegen des Klägers nach Teilklagerücknahme
 * oder Teilerledigung analog § 92 Abs. 1 ZPO ähnlich einem teilweisen
 * Unterliegen. Nach Schneider (Kosten S. 94, 197 ff.; MüKo § 269 Rn 57) ist für
 * jede Gebühr eine gesonderte Quote zu ermitteln, da einige Gebühren nur nach
 * dem verringerten Streitwert anfallen (Anders/Gehle, A-198).
 * 
 * @author Peter Schuster (setrok)
 * @date 27.04.2014
 */
public class QuotenMethode extends StreitwertReduktion {

	public QuotenMethode(VerfahrensDatenContainer verfahrensDaten
	/* , StreitwertReduktionen swrn */) throws IllegalArgumentException {
		super(verfahrensDaten/* , swrn */);
	}

	@Override
	public String getBerechnungsTyp() {
		return "Tenor nach der sog. Quotenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss)";
	}

	/**
	 * 
	 * @return
	 * @return einen {@code double} zwischen 0.0 und 1.0, der in Prozent umgerechnet
	 *         das Unterliegen des Klägers ausdrückt.
	 * 
	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#berechneUnterliegensQuote()
	 */
	@Override // double
	protected Fraction berechneUnterliegensQuote() {
		final Fraction klVerlustTkn = unterliegenTeilKlageruecknahme();

		System.out.println("klVerlustTkn = " + klVerlustTkn.toString() + " = " + klVerlustTkn.doubleValue());

		final int i = 2;
		Fraction gewichtungVerbleibend = gewichtungVerbleibend(klVerlustTkn, i, 0);

		System.out.println("gewichtungVerbleibend = " + gewichtungVerbleibend.toString() + " = "
				+ gewichtungVerbleibend.doubleValue());

		/*
		 * Dann auch die Werte für den verbliebenen Streitgegenstand den Summen
		 * hinzurechnen. gewichtungVerbleibend = s.o., entweder 1.0 oder gemäß Anteil am
		 * Rechtsstreit
		 */
		Fraction sonstigerVerlust = verteilung(i - 1).getFractionFor(klaegerlein).multiply(gewichtungVerbleibend);

		System.out
				.println("sonstigerVerlust = " + sonstigerVerlust.toString() + " = " + sonstigerVerlust.doubleValue());

		super.zeilen = baueBegruendungsTabelle(verteilung(0), gewichtungVerbleibend, sonstigerVerlust);
		super.starteGruende(baueGruende(verteilung(0), super.zeilen));

		// wenn es keinen Unterschied zwischen den beiden Streitwerten gibt, aber nur
		// zum Teil verurteilt wurde, dann gib
		// das Verhältnis der Verurteilung zum letzten Streitwert zurück.
		if (streitwerte.length == 2 && streitwerte[0].equals(streitwerte[1]) && !streitwerte[1].equals(verurteilung)) {
			return Fraction.ONE.subtract(Fraction.valueOf(verurteilung.getCents(), streitwerte[1].getCents()));
		}

		// Abwägung muss hinein?
		// Nein, bei der Quotenmethode wird sie in die Berechnung schon einbezogen.
		return klVerlustTkn;// .add(sonstigerVerlust);// .doubleValue();
	}

	/**
	 * Die Methode dient dazu, die Unterliegensquote nach Maßgabe der dem
	 * Konstruktor übergebenen Werte zu berechnen.
	 * 
	 * @return eine {@link Fraction}, die die Unterliegensquote (der Klägerseite)
	 *         bezogen auf eine Teilklagerücknahme bedeutet.
	 */
	private Fraction unterliegenTeilKlageruecknahme() {
		/*
		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
		 * doof. :)
		 */
		for (int zeile = 0; zeile < reduktionen().size() - 1; zeile++) {
			List<GebuehrenTatbestand> tatbestaende = tatbestaende(zeile);
			/*
			 * Die Verlustquote als Bruch zwischen 0/1 und 1/1 zeigt das Verhältnis des
			 * ggfs. höheren (aktuellen) ggü dem niedrigeren (streitwerte[1]) Streitwert,
			 * ggfs. 1/1, wenn die Schleife gerade durch den niedrigeren Wert geht
			 */
//			double verlustQuote = 1.0 - (verurteilung.divide(streitwerte[i])).doubleValue();
//			final Fraction verlustQuote = new Fraction(streitwert(1).getCents(), streitwert(zeile).getCents());
			final Fraction verlustQuote = Fraction.ONE.subtract(// also 100% abzüglich
					Fraction.valueOf(verurteilung.getCents(), streitwert(zeile).getCents())//
			);// beim Fehler war's streitwert(1) - warum? Muss Veruteilung sein, oder?
			System.out.println("Zeile " + zeile + ". Verlustquote Kläger für " + streitwert(zeile)
					+ ", im Verhältnis zu " + verurteilung + " = " + verlustQuote);
			if (tatbestaende != null) {
				/*
				 * Innere Schleife: alle GebührenTB des jeweiligen Arrays GebuehrenTatbestand[]
				 * durchgehen,
				 */
				for (GebuehrenTatbestand gt : tatbestaende) {
					final GebuehrenRueckgabe gr = gebuehrenZeile(zeile, verlustQuote, gt);
					verteilung(0).add(beklagte, gr.unterliegensBetragBk);
					verteilung(0).add(klaegerlein, gr.unterliegensBetragKl);
					//
					System.out.println(gt.getVorschrift() + "=" + gr.gebuehrTatsaechlich + ", Kläger: "
							+ gr.unterliegensBetragKl + ", Bk: " + gr.unterliegensBetragBk);
				}
			}
		}
		final Fraction klVerlustTkn = verteilung(0).getFractionFor(this.klaegerlein);
		// System.out.println(klVerlustTkn.toString() + " = " +
		// klVerlustTkn.doubleValue());
		return klVerlustTkn;
	}

	/**
	 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
	 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
	 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 5.
	 * 
	 * @param verluste die {@link VerlusteBank}
	 * @return die zusammengestellten {@link BegruendungsZahlenTabelle}
	 */
	private BegruendungsZahlenTabelle baueBegruendungsTabelle(VerlusteBank verluste, Fraction gewichtungVerbleibend,
			Fraction sonstigerVerlust) {
		/*
		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 5.
		 */
		final BegruendungsZahlenTabelle zeilen = new BegruendungsZahlenTabelle();
//		final BegruendungsTabelle zeilen = new BegruendungsTabelle();
		zeilen.add(Arrays.asList(new String[] { "Gebührentatbestand", "Streitwert (EUR)", "Gebührenbetrag (EUR)",
				"Verlustquote Kl.", "Unterliegen Kl. (EUR)" }));
		/*
		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
		 * doof. :)
		 */
		for (int zeile = 0; zeile < reduktionen().size() - 1; zeile++) {
			List<GebuehrenTatbestand> tatbestaende = tatbestaende(zeile);
			/*
			 * Die Verlustquote als Bruch zwischen 0/1 und 1/1 zeigt das Verhältnis des
			 * ggfs. höheren (aktuellen) ggü dem niedrigeren (streitwerte[1]) Streitwert,
			 * ggfs. 1/1, wenn die Schleife gerade durch den niedrigeren Wert geht
			 */ // warum:
//			final Fraction verlustQuote = streitwert(1).divideAsFraction(streitwert(zeile));
//                    Fraction.reducedOf(streitwert(1).getCents(),
//                    streitwert(zeile).getCents())
			final Fraction verlustQuote = Fraction.ONE.subtract(// also 100% abzüglich
					Fraction.valueOf(verurteilung.getCents(), streitwert(zeile).getCents())//
			);// beim Fehler war's streitwert(1) - warum? Muss Veruteilung sein, oder?
			if (tatbestaende != null) {
				/*
				 * Innere Schleife: alle GebührenTB des jeweiligen Arrays GebuehrenTatbestand[]
				 * durchgehen,
				 */
				for (GebuehrenTatbestand gt : tatbestaende) {
//					final TitledNumberRow numbers = new TitledNumberRow(4);
					final BegruendungsZahlenZeile numbers = new BegruendungsZahlenZeile(4);
					final GebuehrenRueckgabe gr = gebuehrenZeile(zeile, verlustQuote, gt);
					numbers.add(streitwert(zeile));
					/*
					 * Betrag der Gebühr berechnen und dem Summenfeld hinzu addieren.
					 */
					numbers.add(gr.gebuehrTatsaechlich);
					/*
					 * Das Unterliegen des Klägers für diesen Gebührentatbestand errechnet sich aus
					 * diesem Betrag im Verhältnis zur Quote dieser Zeile. Auch hier zum Summenfeld
					 * addieren.
					 */
					numbers.add(verlustQuote);
					numbers.add(gr.unterliegensBetragKl);
					/* Und die Zeile in die Tabelle nehmen */
					zeilen.add(gt.langBezeichnung(), numbers);
				}
			}
		}

		final int i = 2;
		if (zusatz(i).begruendet.lessThan(differenz(i - 1))) {
			final String zwischenUeberschrift = "Zwischensumme";
			final String verbliebeneUeberschrift = String.format("Verbliebener Streitgegenstand (gewichtet zu %s)",
					gewichtungVerbleibend.toString());
			/*
			 * Damit die ausgegebene Tabelle (halbwegs) nachvollziehbar bleibt, brauchen wir
			 * eine Zwischensummen-Zeile
			 */
//			final TitledNumberRow zwischenZeile = new TitledNumberRow(4);
			final BegruendungsZahlenZeile zwischenZeile = new BegruendungsZahlenZeile(4);
			zwischenZeile.add(Euro.ZERO_CENTS);
			zwischenZeile.add(verluste.getFiktiverStreitwert());
			zwischenZeile.add(verluste.getFractionFor(klaegerlein));
			zwischenZeile.add(verluste.get(klaegerlein));
			zwischenZeile.add(verluste.getFractionFor(klaegerlein));
			zeilen.add(zwischenUeberschrift, zwischenZeile);
			/*
			 * Dann auch die Zeile für den verbliebenen Streitgegenstand erzeugen. faktor =
			 * s.o., entweder 1.0 oder gemäß Anteil am Rechtsstreit
			 */
//			final TitledNumberRow verbliebenZeile = new TitledNumberRow(4);
			final BegruendungsZahlenZeile verbliebenZeile = new BegruendungsZahlenZeile(4);
			verbliebenZeile.add(streitwert(1));
			verbliebenZeile.add(streitwert(1));
			verbliebenZeile.add(sonstigerVerlust);
			verbliebenZeile.add(differenz(i - 1).subtract(zusatz(i).begruendet));
			zeilen.add(verbliebeneUeberschrift, verbliebenZeile);
		}
		/*
		 * Keine Prozentangabe, sondern ein Bruch zwischen 0.0 und 1.0: Summe der
		 * Verlustbeträge durch Summe aller Gebühren
		 */
//		final TitledNumberRow gesamtWerte = new TitledNumberRow(4);
		final BegruendungsZahlenZeile gesamtWerte = new BegruendungsZahlenZeile(4);
		gesamtWerte.add(Euro.ZERO_CENTS);
		gesamtWerte.add(verluste.getFiktiverStreitwert());
		gesamtWerte.add(verluste.getFractionFor(klaegerlein));
		gesamtWerte.add(verluste.get(klaegerlein));
		/* Ergebniszeile in die Tabelle */
		zeilen.add("Gesamt:", gesamtWerte);

		return zeilen;
	}

	/**
	 * Die Methode errechnet, welche Beträge sich aus der Quote mit dem
	 * Gebührentatbestand ergeben.
	 * 
	 * @param zeile        index der Streitwerte-Liste
	 * @param verlustQuote ein Bruch mit dem Unterliegen des Klägers
	 * @param gt           ein {@link GebuehrenTatbestand}
	 * @return eine {@link GebuehrenRueckgabe}
	 */
	private GebuehrenRueckgabe gebuehrenZeile(int zeile, Fraction verlustQuote, GebuehrenTatbestand gt) {
		final GebuehrenRueckgabe gr = new GebuehrenRueckgabe();
		/*
		 * Betrag der Gebühr berechnen.
		 */
		gr.gebuehrTatsaechlich = // Euro.ofEuros(.longValue())
				GebuehrenAuflistung.errechneGebuehr(gt, streitwert(zeile));
		/*
		 * Das Unterliegen des Klägers für diesen Gebührentatbestand errechnet sich aus
		 * diesem Betrag im Verhältnis zur Quote dieser Zeile. Das Unterliegen des
		 * Beklagten ist die Differenz davon zur tatsächlichen Gebühr.
		 */
		gr.unterliegensBetragKl = gr.gebuehrTatsaechlich.multiply(verlustQuote);
		gr.unterliegensBetragBk = gr.gebuehrTatsaechlich.subtract(gr.unterliegensBetragKl);
//				gr.gebuehrTatsaechlich.multiply(Fraction.ONE.subtract(verlustQuote));
		return gr;
	}

	/**
	 * Die Methode dient dazu, über
	 * {@link BegruendungsZahlenTabelle#toStringBuilder(StringBuilder)} die Gründe
	 * zu basteln.
	 * 
	 * @param verluste          die {@link VerlusteBank}
	 * @param tabulierendZeilen die zusammengestellten
	 *                          {@link BegruendungsZahlenTabelle}
	 * @return einen {@link StringBuilder}
	 */
	private CharSequence baueGruende(VerlusteBank verluste, BegruendungsZahlenTabelle zeilen) {
		final StringBuilder gruende = new StringBuilder(TenorToken.GRUENDE_EINLEITUNG);
		if (verluste.getQuotaFor(klaegerlein) == 0.0 || verluste.getQuotaFor(klaegerlein) == 0.0) {
			gruende.append(TenorToken.GRUENDE_EINLEITUNG_91);
			gruende.append(
					"Denn auch unter Berücksichtigung der quotalen Beteiligung der Parteien an allen Gebühren ergibt sich ein eindeutiges Ergebnis, wie sich aus folgender Tabelle ergibt:\n");
		} else {
			gruende.append(TenorToken.GRUENDE_EINLEITUNG_92_269);
			gruende.append(TenorToken.WEITER_GRUENDE_QUOTENM);
			gruende.append(TenorToken.PUNKT_ABSATZ);
			gruende.append(
					"Denn an den einzelnen Gebühren sind die Parteien mit verschiedenen Quoten beteiligt, je nachdem, ob die einzelne Gebühr vor oder nach der Reduktion angefallen ist, wie sich aus der folgenden Tabelle ergibt:\n");
			if (super.swrn.streitwerte.get(1).greaterThan(reduktionen().get(2).zusatz.begruendet)) {
				// TODO zu stark verdrahtet
				gruende.append(TenorToken.GRUENDE_VERBLIEBENER_GEGENSTAND);
				gruende.append(TenorToken.GRUENDE_VERBLIEBENER_GEGENSTAND_SONST);
			}
		}
		return zeilen.toStringBuilder(gruende);
	}

//	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand", "Streitwert (EUR)",
//			"Gebührenbetrag (EUR)", "Verlustquote Kl. (%)", "Unterliegen Kl. (EUR)" };
//
//	private static final String BESCHREIBUNG = "Tenor nach der sog. Quotenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss)";
//
//	private static final String WEITER_GRUENDE = "der sogenannten Quotenmethode.\n";
//
//	/**
//	 * Der Konstruktor ruft den Konstruktor der Oberklasse auf.
//	 * 
//	 * 
//	 * @param teilklageruecknahme Parameterojbkekt vom Typ
//	 *                            {@link Teilklageruecknahme}, das enthält die
//	 *                            {@link Teilklageruecknahme#parteien} klaeger
//	 *                            Informationen zur Klägerseite als Beteiligter
//	 *                            (wird zur Tenorierung benötigt) und beklagter
//	 *                            Informationen zur Beklagten als Beteiligter (wird
//	 *                            zur Tenorierung benötigt), die Arrays
//	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
//	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
//	 *                            erst nach der "Streitwertreduktion"
//	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
//	 *                            Gebühren, die
//	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
//	 *                            eine ArrayList<Long> mit streitwerteUndObsiegen[0]
//	 *                            als dem alten (höheren) Streitwert,
//	 *                            streitwerteUndObsiegen[1] als dem neuen,
//	 *                            niedrigeren Streitwert und die
//	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
//	 *                            Höhe der Kläger letzlich obsiegt hat.
//	 * 
//	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
//	 *                            b92ii Soll § 92 II ZPO angewandt werden
//	 *                            (verhältnismäßig geringe Kostentragungen werden
//	 *                            unterschlagen)
//	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
//	 *                                  folgenden Plausibilitäten verletzt ist: a)
//	 *                                  der frühere Streitwert muss größer als der
//	 *                                  spätere (oder gleich) sein, b) der spätere
//	 *                                  Streitwert muss größer als die Verurteilung
//	 *                                  (oder gleich) sein.
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#StreitwertReduktion(Allgemein,
//	 *      Teilklageruecknahme)
//	 * 
//	 */
//	public QuotenMethode(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
//		super(verfahrensDaten);
//	}
//
//	/**
//	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getBerechnungsTyp()
//	 */
//	@Override
//	public String getBerechnungsTyp() {
//		return BESCHREIBUNG;
//	}
//
//	/**
//	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
//	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen.
//	 * 
//	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
//	 *         Unterliegen ausdrückt.
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#berechneUnterliegensQuote()
//	 */
//	@Override
//	protected double berechneUnterliegensQuote() {
//		Euro summeUnterliegen = Euro.ZERO_CENTS;// double = 0.0;
//		Euro summeBetraege = Euro.ZERO_CENTS;// double = 0.0;
//		/*
//		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
//		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
//		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 5.
//		 */
//		zeilen = new TabulierendZeilen();
//		zeilen.add(Arrays.asList(COLUMN_HEADERS));
//		/*
//		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
//		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
//		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
//		 * doof. :)
//		 */
//		for (int i = 0; i < streitwerte.length; i++) {
//			List<GebuehrenTatbestand> tatbestaende = i == 0 ? vorher : nachher;
//			/*
//			 * Die Verlustquote (Bruch zwischen 0.0 und 1.0) muss immer daran gemessen
//			 * werden, wie hoch der Kläger letztlich obsiegt, steht also im Nenner des
//			 * Bruchs. Bei der vorher-Liste steht der (höhere) Streitwert vor der Reduktion
//			 * im Zähler, bei der nachher-Liste der (niedrigere) Streitwert nach der
//			 * Reduktion.
//			 */
//			double verlustQuote = 1.0 - (verurteilung.divide(streitwerte[i])).doubleValue();
//			// ((double) verurteilung / (double) streitwerte[i]);
//			if (tatbestaende != null) {
//				/*
//				 * Innere Schleife: alle GebührenTB des jeweiligen Arrays GebuehrenTatbestand[]
//				 * durchgehen,
//				 */
//				for (GebuehrenTatbestand gt : tatbestaende) {
//					final DoubleDataRows doubles = new DoubleDataRows(4);
//					doubles.add(streitwerte[i].doubleValue()); // (new Double(streitwerte[i]));
//					/*
//					 * Betrag der Gebühr berechnen und dem Summenfeld hinzu addieren.
//					 */
//					Euro betragGebuehr = GebuehrenAuflistung.errechneGebuehr(gt, streitwerte[i]);
//					// summeBetraegeEUR += betragGebuehr;
//					summeBetraege = summeBetraege.add(betragGebuehr);
//					doubles.add(betragGebuehr.doubleValue());// (new Double(betragGebuehr));
//					/*
//					 * Das Unterliegen des Klägers für diesen Gebührentatbestand errechnet sich aus
//					 * diesem Betrag im Verhältnis zur Quote dieser Zeile. Auch hier zum Summenfeld
//					 * addieren.
//					 */
//					final Euro betragVerlust = betragGebuehr.multiply(verlustQuote);// verlustQuote * betragGebuehr;
//					doubles.add(verlustQuote * 100.0 /* Prozent */);// (new Double(verlustQuote * 100.0 /* Prozent */));
//					doubles.add(betragVerlust.doubleValue());// (new Double(betragVerlust));
//					// summeUnterliegenEUR += betragVerlust;
//					summeUnterliegen = summeUnterliegen.add(betragVerlust);
//					/* Und die Zeile in die Tabelle nehmen */
//					zeilen.add(gt.langBezeichnung(), doubles);// toString()
//				}
//			}
//		}
//		/*
//		 * Keine Prozentangabe, sondern ein Bruch zwischen 0.0 und 1.0: Summe der
//		 * Verlustbeträge durch Summe aller Gebühren
//		 */
//		final DoubleDataRows gesamtWerte = new DoubleDataRows(4);
//		gesamtWerte.add(0.0);// new Double(0)
//		gesamtWerte.add(summeBetraege.doubleValue());// (new Double(summeBetraegeEUR));
//		double unterliegensQuote = summeUnterliegen.divide(summeBetraege);
//		// summeUnterliegenEUR / summeBetraegeEUR;
//		gesamtWerte.add(unterliegensQuote * 100.0 /* Prozent */);
//		// (new Double(unterliegensQuote * 100.0 /* Prozent */));
//		gesamtWerte.add(summeUnterliegen.doubleValue());// (new Double(summeUnterliegenEUR));
//		/* Ergebniszeile in die Tabelle */
//		zeilen.add("Gesamt:", gesamtWerte);
//		baueGruende(unterliegensQuote);
//		return unterliegensQuote;
//	}
//
//	/**
//	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
//	 * 
//	 * @param unterliegensQuote ein double zwischen 0.0 und 1.0, der in Prozent
//	 *                          umgerechnet das Unterliegen ausdrückt.
//	 */
//	private void baueGruende(double unterliegensQuote) {
//		if (unterliegensQuote == 0.0 || unterliegensQuote == 1.0) {
//			super.starteGruende(GRUENDE_EINLEITUNG_91);
//			super.erweitereGruende(
//					"Denn auch unter Berücksichtigung der quotalen Beteiligung der Parteien an allen Gebühren ergibt sich ein eindeutiges Ergebnis, wie sich aus folgender Tabelle ergibt:\n");
//		} else {
//			super.starteGruende(GRUENDE_EINLEITUNG_92_269 + WEITER_GRUENDE);
//			super.erweitereGruende(
//					"Denn an den einzelnen Gebühren sind die Parteien mit verschiedenen Quoten beteiligt, je nachdem, ob die einzelne Gebühr vor oder nach der Reduktion angefallen ist, wie sich aus der folgenden Tabelle ergibt:\n");
//		}
//		zeilen.toStringBuilder(gruende);
//	}

}