/**
 * TeilruecknahmeTeilerledigung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.ArrayList;
import java.util.List;

import eu.gronos.kostenrechner.data.KostenZweierquote;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.ReduktionStruktur;
import eu.gronos.kostenrechner.data.gebuehren.ReduktionZusatzInformation;
import eu.gronos.kostenrechner.data.gebuehren.ReduktionsGrund;
import eu.gronos.kostenrechner.data.gebuehren.ReduktionsVerarbeitung;
import eu.gronos.kostenrechner.data.gebuehren.StreitwertReduktionen;
import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.data.tenordaten.Allgemein;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.VerlusteBank;
import eu.gronos.kostenrechner.util.gebuehren.TeilklageruecknahmePruefer;

/**
 * Oberklasse für alle Klasse, die Teilklageerledigung oder Teilklagerücknahme
 * behandeln (bisher Mehrkostenmethode und Quotenmethode). Sie enthält alle
 * gemeinsam benötigten Methoden.
 * 
 * @author Peter Schuster (setrok)
 * @date 16.06.2014
 * 
 */
public abstract class StreitwertReduktion implements TenorVorbereitend/* , Begruendend */ {

	protected final StreitwertReduktionen swrn;
	protected final List<Beteiligter> klaegerlein;
	protected final List<Beteiligter> beklagte;
	protected final List<ReduktionsVerarbeitung> verarbeitungen;

	// aus der alten Version
//	protected final List<GebuehrenTatbestand> vorher;
//	protected final List<GebuehrenTatbestand> nachher;
//	private final boolean b92ii;
	protected final Euro[] streitwerte;
	protected final Euro verurteilung;
	final Beteiligter klaeger;
	final Beteiligter beklagter;
	protected final StringBuilder gruende;
	protected BegruendungsZahlenTabelle zeilen;
	/**
	 * Der quote speichert das Unterliegen des Klägers als double zwischen 0.0 und
	 * 1.0. Vor Initialisierung ist der Wert -1.0.
	 */
	private Fraction quote = Fraction.valueOf(-1.0);// double -1.0
	private VollstreckbarkeitsListe vollstreckbarkeitsListe;
	private final TenorDatenContainer container;
	private Pruefend<Teilklageruecknahme> pruefer = new TeilklageruecknahmePruefer();
	private final StreitwertReduktionTexter texter;// = new StreitwertReduktionTexter(allgemein.alsBruch);
	/**
	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
	 * und {@link Allgemein#unwesentlich92ii.}
	 */
	private Allgemein allgemein;

	/**
	 * Konstruktor:
	 * 
	 * @param teilklageruecknahme Parameterojbkekt vom Typ
	 *                            {@link Teilklageruecknahme}, das enthält die
	 *                            {@link Teilklageruecknahme#parteien} klaeger
	 *                            Informationen zur Klägerseite als Beteiligter
	 *                            (wird zur Tenorierung benötigt) und beklagter
	 *                            Informationen zur Beklagten als Beteiligter (wird
	 *                            zur Tenorierung benötigt), die Arrays
	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
	 *                            erst nach der "Streitwertreduktion"
	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
	 *                            Gebühren, die
	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
	 *                            ein Array long[3] mit streitwerteUndObsiegen[0]
	 *                            als dem alten (höheren) Streitwert,
	 *                            streitwerteUndObsiegen[1] als dem neuen,
	 *                            niedrigeren Streitwert und die
	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
	 *                            Höhe der Kläger letzlich obsiegt hat.
	 * 
	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
	 *                            b92ii Soll § 92 II ZPO angewandt werden
	 *                            (verhältnismäßig geringe Kostentragungen werden
	 *                            unterschlagen)
	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
	 *                                  folgenden Plausibilitäten verletzt ist: a)
	 *                                  der frühere Streitwert muss größer als der
	 *                                  spätere (oder gleich) sein, b) der spätere
	 *                                  Streitwert muss größer als die Verurteilung
	 *                                  (oder gleich) sein.
	 */
	public StreitwertReduktion(VerfahrensDatenContainer verfahrensDaten/* , StreitwertReduktionen swrn */)
			throws IllegalArgumentException {
//		super(verfahrensDaten);

		final Teilklageruecknahme teilklageruecknahme = verfahrensDaten.teilklageruecknahme;

		this.streitwerte = new Euro[] { teilklageruecknahme.streitwerteUndObsiegen.get(0),
				teilklageruecknahme.streitwerteUndObsiegen.get(1) };
		this.verurteilung = teilklageruecknahme.streitwerteUndObsiegen.get(2);

		// Ausgegliedert in ParsendUndBauend
		pruefer.pruefeEingabe(teilklageruecknahme);

		this.klaeger = teilklageruecknahme.parteien.get(0);
		this.beklagter = teilklageruecknahme.parteien.get(1);
//		this.vorher = teilklageruecknahme.vorher;
//		this.nachher = teilklageruecknahme.nachher;
		this.allgemein = verfahrensDaten.allgemein;
		texter = new StreitwertReduktionTexter(allgemein.alsBruch);
		
		gruende = new StringBuilder();
		container = new TenorDatenContainer(verfahrensDaten);

		this.swrn = toStreitwertReduktionen(teilklageruecknahme);// swrn;
		this.container.erweiterteKlasse = this.swrn;
		this.klaegerlein = new ArrayList<>(swrn.parteien.subList(0, 1));
		this.beklagte = new ArrayList<>(swrn.parteien.subList(1, verfahrensDaten.teilklageruecknahme.parteien.size()));

		this.verarbeitungen = new ArrayList<>();
		initVerarbeitungen();
	}

	/**
	 * Die Methode initialisiert {@link #verarbeitungen}, also
	 * {@link ReduktionsVerarbeitung}, indem sie die Werte aus dem übergebenen
	 * {@link UnterContainerKlasse} namens {@link StreitwertReduktionen} entnimmt.
	 */
	private void initVerarbeitungen() {
		for (int i = 1; i < this.swrn.reduktionen.size() && i < this.swrn.streitwerte.size(); i++) {
			final ReduktionsVerarbeitung verarbeitung = new ReduktionsVerarbeitung();
			verarbeitung.differenz = this.swrn.streitwerte.get(i - 1).subtract(this.swrn.streitwerte.get(i));
			verarbeitung.anteilAmRechtsstreit = verarbeitung.differenz.divideAsFraction(this.swrn.streitwerte.get(0));
//                    Fraction.reducedOf(
//                    verarbeitung.differenz.getCents(),
//                    this.swrn.streitwerte.get(0).getCents())
			verarbeitungen.add(verarbeitung);
		}
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf die
	 * {@link ReduktionsVerarbeitung#anteilAmRechtsstreit}
	 * 
	 * @param i den index der {@link #verarbeitungen}
	 * @return den passenden Anteil als {@link Fraction}
	 */
	protected Fraction anteilAmRechtsstreit(final int i) {
		return this.verarbeitungen.get(i).anteilAmRechtsstreit;
	}

	/**
	 * Getter {@link #swrn#reduktionen()}
	 * 
	 * @return eine {@link List} aus {@link ReduktionStruktur}
	 */
	protected List<ReduktionStruktur> reduktionen() {
		return this.swrn.reduktionen;
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf die
	 * {@link ReduktionStruktur#tatbestaende}
	 * 
	 * @param i den index der {@link #reduktionen()}
	 * @return die passende {@link GebuehrenAuflistung}
	 */
	protected GebuehrenAuflistung tatbestaende(int i) {
		return this.swrn.reduktionen.get(i).tatbestaende;
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf die {@link VerlusteBank}en der
	 * {@link #verarbeitungen}
	 * 
	 * @param i den index der {@link #verarbeitungen}
	 * 
	 * @return die passende {@link VerlusteBank}
	 */
	protected VerlusteBank verteilung(int i) {
		return this.verarbeitungen.get(i).verteilung;
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf
	 * {@link StreitwertReduktionen#streitwerte}
	 * 
	 * @param i den index der {@link #streitwert(int)}
	 * @return den {@link Euro}
	 */
	protected Euro streitwert(int i) {
		return this.swrn.streitwerte.get(i);
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf {@link ReduktionStruktur#zusatz} aus
	 * {@link StreitwertReduktionen#reduktionen}
	 * 
	 * @param i den index der {@link #reduktionen()}
	 * @return die passende {@link ReduktionZusatzInformation}
	 */
	protected ReduktionZusatzInformation zusatz(final int i) {
		return this.swrn.reduktionen.get(i).zusatz;
	}

	/**
	 * Die Methode gibt schnellen Zugriff auf die
	 * {@link ReduktionsVerarbeitung#differenz} aus den {@link #verarbeitungen}
	 * 
	 * @param i den index der {@link #verarbeitungen}
	 * @return den {@link Euro}
	 */
	protected Euro differenz(final int i) {
		return this.verarbeitungen.get(i).differenz;
	}

	/**
	 * TODO Jetzt noch Vorgehensweise für weitere Teilklageabweisung überlegen
	 * 
	 * Dazu kann man überlegen, dass der Kläger im zurückgenommenen Anteil
	 * vollständig verloren hat (er bekommt von seinem Klageantrag nix, Null,
	 * niente). Um seine Einsicht dankbar zu belohnen, wird fingiert, dass der
	 * zurückgenommene Teil am Gesamtrechtsstreit nur soviel wiegt wie der Anteil
	 * der Mehrkosten an den Gesamtkosten.
	 * 
	 * Bis zu einem gewissen Grad kann man also das weitere Unterliegen einfach
	 * draufrechnen.
	 * 
	 * Allerdings darf man nicht über 100% kommen.
	 * 
	 * Versuchen, insgesamt wieder die 100% zu erreichen, indem man den
	 * verbleibenden Teil höher gewichtet (Kehrwert vom Verhältnis der
	 * Mehrkostenquote zum Anteil am Rechtsstreit? Nö, wird sonst mehr als 100%.
	 * Besser: Differenz zwischen Anteil am Rechtsstreit (des Rücknahmeteils) und
	 * Mehrkostenquote plus Anteil am Rechtsstreit des streitig entschiedenen
	 * Teils), dann erreicht man auch: Und wenn kl gar nichts gewinnt, muss er alle
	 * Kosten tragen
	 * 
	 * @param klVerlustBisher die {@link Fraction} der Verlustquote bezogen auf die
	 *                        Teilklagerücknahme
	 * @param letzteZeile     den index der {@link #verarbeitungen}
	 * @param zeileRn         index der Rücknahme
	 * @return eine {@link Fraction}, die angibt, wie stark der verbliebene Anteil
	 *         der Klage gewichtet werden soll
	 */
	protected Fraction gewichtungVerbleibend(final Fraction klVerlustBisher, final int letzteZeile, int zeileRn) {
		final Fraction gewichtungVerbleibend = anteilAmRechtsstreit(zeileRn).subtract(klVerlustBisher)
				.add(anteilAmRechtsstreit(letzteZeile - 1));
		verteilung(letzteZeile - 1).add(beklagte, zusatz(letzteZeile).begruendet);
		verteilung(letzteZeile - 1).add(klaegerlein,
				differenz(letzteZeile - 1).subtract(zusatz(letzteZeile).begruendet));
		return gewichtungVerbleibend;
	}

	/**
	 * Die Methode wandelt eine
	 * {@link Teilklageruecknahme}-{@link UnterContainerKlasse} in eine
	 * {@link StreitwertReduktionen}-{@link UnterContainerKlasse} um
	 * 
	 * @param tkrn eine {@link Teilklageruecknahme}-{@link UnterContainerKlasse}
	 * @return eine {@link StreitwertReduktionen}-{@link UnterContainerKlasse}
	 */
	private StreitwertReduktionen toStreitwertReduktionen(Teilklageruecknahme tkrn) {
		final StreitwertReduktionen swrn = new StreitwertReduktionen();
		swrn.parteien.addAll(tkrn.parteien);

		// Vorher und Nachher umwandeln
		for (int i = 0; i < tkrn.streitwerteUndObsiegen.size() - 1; i++) {
			final ReduktionStruktur struktur = new ReduktionStruktur();
			struktur.streitwert = // Euro.ofEuros(
					tkrn.streitwerteUndObsiegen.get(i);
			swrn.streitwerte.add(struktur.streitwert);
			switch (i) {
			case 0:
				struktur.grund = ReduktionsGrund.VORHER;
				struktur.tatbestaende.addAll(tkrn.vorher);
				break;
			case 1:
				struktur.grund = ReduktionsGrund.TEILKLAGERUECKNAHME;
				struktur.tatbestaende.addAll(tkrn.nachher);
				break;
			}
			swrn.reduktionen.add(struktur);
		}

		// Letztlich obsiegt wird zur Endentscheidung
		final int i = tkrn.streitwerteUndObsiegen.size() - 1;
		final ReduktionStruktur struktur = new ReduktionStruktur();
		struktur.grund = ReduktionsGrund.ENDENTSCHEIDUNG;
		// Das letzte Element ist immer 0 EUR wert (danach sind nur noch 0 EUR
		// anhängig)
		struktur.streitwert = Euro.ZERO_CENTS;
		swrn.streitwerte.add(struktur.streitwert);
		// Letztlich obsiegt muss in die Zusatzinfo als "begründet"
		struktur.zusatz = new ReduktionZusatzInformation();
		struktur.zusatz.begruendet = // Euro.ofEuros()
				tkrn.streitwerteUndObsiegen.get(i);
		swrn.reduktionen.add(struktur);

		return swrn;
	}

	/**
	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen.
	 * 
	 * @return eine {@link Fraction} zwischen 0 und 1/1, die das Unterliegen
	 *         ausdrückt. War: einen double (zwischen 0.0 und 1.0) der in Prozent
	 *         umgerechnet
	 */
	protected abstract Fraction berechneUnterliegensQuote();// double

	/**
	 * @return die Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
	 */
	@Override
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * Die Methode erzeugeHauptsachetenor dient dazu, einen einfachen
	 * Hauptsachetenor zu erzeugen.
	 * 
	 * @return ein {@link EntscheidungsListenElemente}-Objekt für
	 *         {@link HauptsacheVerhaeltnis} mit dem Hauptsachetenor als String mit
	 *         abschließendem Zeilenumbruch.
	 */
	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung() {
		EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
		hauptsache.prozessverhaeltnisse.clear();
		hauptsache.prozessverhaeltnisse.add(new HauptsacheVerhaeltnis(klaeger, beklagter, false, verurteilung));
		// Ausgegliedert in TenorTexter
		texter.setStreitwerte(streitwerte);
		hauptsache.text = texter.texteHauptsache(hauptsache);
		return hauptsache;
	}

	/**
	 * Die Methode erzeugeKostentenor dient dazu, einen Kostentenor zu erzeugen. Die
	 * Angaben zu den Parteien werden aus den Konstruktor-Parametern
	 * <code>klaeger</code> und <code>beklagter</code> entnommen, die Quote aus der
	 * Methode {@link berechneUnterliegensQuote()} der abgeleiteten Klassen.
	 * 
	 * @return den Tenortext als String im Format "Die Kosten haben %s zu %,.2f%%
	 *         und %s zu %,.2f%% zu tragen.%n"
	 */
	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung() {
		EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();
		if (getQuote().equals(Fraction.ZERO)) {// == 0.0
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, Fraction.ONE));// 1.0
		} else if (getQuote().equals(Fraction.ONE)) {// == 1.0
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, Fraction.ONE));// 1.0
		} else {
			KostenZweierquote zweierQuote = new KostenZweierquote(getQuote());
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, zweierQuote.getKlaegerQuote()));// .doubleValue()
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, zweierQuote.getBeklagtenQuote()));// .doubleValue()
		}
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);
		return kostenEntscheidung;
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit
	 * 
	 * @return {@link EntscheidungsListenElemente} für
	 *         {@link VollstreckungsVerhaeltnis}
	 */
	@Override
	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung() {
		EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();

		if (vollstreckbarkeitsListe == null)
			vollstreckbarkeitsListe = berechneVollstreckbarkeit();
		erweitereGruende(vollstreckbarkeitsListe.toString());
		VollstreckbarkeitsHelfer helfer = new VollstreckbarkeitsHelfer(
				container.hauptsacheEntscheidung.prozessverhaeltnisse,
				container.kostenEntscheidung.prozessverhaeltnisse);
		vollEntscheidung.prozessverhaeltnisse = helfer.toVollstreckbarkeitsVerhaeltnisse(klaeger, beklagter,
				vollstreckbarkeitsListe);

		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeitsListe);
		return vollEntscheidung;
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung unter Berücksichtigung des
	 * früheren und des späteren Streitwerts. Das Datum der Reduktion muss man noch
	 * von Hand nachtragen.
	 * 
	 * @return einen String im Format "Der Streitwert wird auf %,.2f EUR bis zum
	 *         ..., danach auf %,.2f EUR festgesetzt.%n"
	 */
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente elemente = container.streitwertEntscheidung;
		elemente.streitwerte.add(streitwerte[0]);
		elemente.streitwerte.add(streitwerte[1]);
		elemente.text = texter.texteStreitwert(elemente);
		return elemente;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link EntscheidungsElemente} für Sonstige
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public EntscheidungsElemente erzeugeSonstigeEntscheidung() {
		EntscheidungsElemente sonstige = new EntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		container.allgemein.selectedPanel = Teilklageruecknahme.TAB_ID;
		return container;
	}

	/**
	 * Getter für {@link #quote}. Wenn der Wert noch nicht gesetzt wurde (dann ist
	 * er < 0.0), wird er über {@link #berechneUnterliegensQuote()} berechnet. Je
	 * nach {@link #b92ii} wird {@link #entferneUnwesentlichesUnterliegen(double)}
	 * aufgerufen.
	 * 
	 * @return gibt {@link #quote} als double zurück.
	 */ // double
	Fraction getQuote() {
		if (quote.lessThan(Fraction.ZERO)) {// < 0.0
			quote = berechneUnterliegensQuote();
			if (isB92ii())
				quote = entferneUnwesentlichesUnterliegen(quote);
		}
		return quote;
	}

	/**
	 * @return gibt {@link #b92ii} als boolean zurück, also ob § 92 II ZPO angewandt
	 *         werden soll (verhältnismäßig geringe Kostentragungen werden dann
	 *         unterschlagen).
	 */
	boolean isB92ii() {
		return allgemein.b92ii;
	}

	/**
	 * Die Methode dient dazu, &sect;&nbsp;92 Abs.&nbsp;2 ZPO anzuwenden, also eine
	 * verhältnismäßig geringe Kostentragung zu unterschlagen
	 * 
	 * @param quote die Ausgangs-{@link #getQuote() Quote}
	 * @return die bereinigte {@link #getQuote() Quote}
	 */ // double //(double quote)
	private Fraction entferneUnwesentlichesUnterliegen(Fraction quote) {
		final Fraction unwesentlich = Fraction.valueOf(allgemein.vomHundert92ii, 100);
		// if (quote > 0.0 && quote < ((double) allgemein.vomHundert92ii / 100.0))// 0.1
		// //return 0.0;
		if (quote.greaterThan(Fraction.ZERO) && quote.lessThan(unwesentlich))
			return Fraction.ZERO;
		else if (Fraction.ONE.subtract(quote).lessThan(unwesentlich))
			return Fraction.ONE;
		// else if ((1.0 - quote) < ((double) allgemein.vomHundert92ii / 100.0))// quote
		// > 0.9 return 1.0;
		else
			return quote;
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, wenn nicht unanfechtbar das 1. Element
	 *         das Verhältnis kl -> bk und das 2. Element das Verhältnis bk -> kl.
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
		VollstreckbarkeitsListe vollstreckbarkeitsListe = new VollstreckbarkeitsListe();
		vollstreckbarkeitsListe.add(pruefeUnanfechtbarkeit());
		/* Nach festgestellter Unanfechtbarkeit war's das schon. */
		if (vollstreckbarkeitsListe.sindAlle708iVm713()) {
			return vollstreckbarkeitsListe;
		}
		/*
		 * Beim Verhältnis Kl ./. Bk hinsichtlich der Hauptsache einsetzen, was der
		 * Kläger letzlich obsiegt. Hinsichtlich der Kosten die Kosten des Klägers (nach
		 * Streitwert vor Reduktion) einsetzen, soweit sie der Beklagte tragen muss.
		 */
		Vollstreckbarkeit voKl = Vollstreckbarkeit.pruefeSicherheitsleistung(verurteilung,
				// TenorTexter.getKostenKlaeger().errechneGebuehrenSumme(streitwerte[0]) * (1.0
				// - getQuote()));
				TenorTexter.getKostenKlaeger().errechneGebuehrenSumme(streitwerte[0])
						.multiply(Fraction.ONE.subtract(getQuote())));// (1.0 - getQuote()));
		vollstreckbarkeitsListe.add(voKl);
		/*
		 * Beim Verhältnis Bk ./. Kl hinsichtlich der Hauptsache 0L einsetzen,
		 * hinsichtlich der Kosten die des Bk einsetzen (Streitwert vor Reduktion),
		 * soweit sie der Kl tragen muss.
		 */
		Vollstreckbarkeit voBk = Vollstreckbarkeit.pruefeSicherheitsleistung(Euro.ZERO_CENTS, // 0L,
				// (TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwerte[0]) *
				// getQuote()));
				TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwerte[0]).multiply(getQuote()));
		vollstreckbarkeitsListe.add(voBk);
		return vollstreckbarkeitsListe;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
		/*
		 * Erst einmal das VollstreckbarkeitsObjekt "leer" initialisieren; wenn's nicht
		 * unanfechtbar.
		 */
		Vollstreckbarkeit unanfechtbarkeit = new Vollstreckbarkeit();
		/*
		 * Wenn schon der Streitwert vor Reduktion nicht über 600 EUR geht, ist das
		 * Urteil unanfechtbar.
		 */
		// streitwerte[0] <= TenorTexter.BERUFUNGS_GRENZE
		if (streitwerte[0].compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
			unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
		} else if (streitwerte[0].compareTo(TenorTexter.BERUFUNGS_GRENZE.multiply(2.0)) <= 0) {
			// } else if (streitwerte[0] <= TenorTexter.BERUFUNGS_GRENZE * 2) {
			/*
			 * Wie ist der Kläger beschwert? Das, was von seiner Klage nicht durchgedrungen
			 * ist: Differenz zwischen Streitwert vor Reduktion und letztlich obsiegt. Wie
			 * ist der Beklagte beschwert: Das wozu er verurteilt wurde ... Nur wenn beides
			 * <= 600 ist, dann ist das Urteil unanfechtbar.
			 */
			// (streitwerte[0] - verurteilung <= TenorTexter.BERUFUNGS_GRENZE
			if (streitwerte[0].subtract(verurteilung).compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0
					// verurteilung <= TenorTexter.BERUFUNGS_GRENZE
					&& verurteilung.compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
				unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
			}
		}
		return unanfechtbarkeit;
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
	 * beginnt sie neu mit dem übergebenen Text.
	 * 
	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
	 *             Gründe.
	 */
	protected void starteGruende(CharSequence text) {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
		erweitereGruende(text);
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

}

//static protected final String GRUENDE_EINLEITUNG_91 = "Die Kostenentscheidung beruht auf § 91 ZPO.";
//static protected final String GRUENDE_EINLEITUNG_92 = "Die Kostenentscheidung beruht auf §§ 91, 92 ZPO und ";
//static protected final String GRUENDE_EINLEITUNG_92_269 = "Die Kostenentscheidung beruht auf §§ 91, 92, 269 Abs. 3 ZPO und ";
//
//protected final List<GebuehrenTatbestand> vorher;
//protected final List<GebuehrenTatbestand> nachher;
//protected final Euro[] streitwerte;
//protected final Euro verurteilung;
//final Beteiligter klaeger;
//final Beteiligter beklagter;
///**
// * Der quote speichert das Unterliegen des Klägers als double zwischen 0.0 und
// * 1.0. Vor Initialisierung ist der Wert -1.0.
// */
//private double quote = -1.0;
//private VollstreckbarkeitsListe vollstreckbarkeitsListe;
//// private final boolean b92ii;
//private final TenorDatenContainer container;
//private Pruefend<Teilklageruecknahme> pruefer = new TeilklageruecknahmePruefer();
//
//
//private final StreitwertReduktionTexter texter = new StreitwertReduktionTexter();
///**
// * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
// * und {@link Allgemein#unwesentlich92ii.}
// */
//private Allgemein allgemein;
//
///**
// * Konstruktor:
// * 
// * @param teilklageruecknahme Parameterojbkekt vom Typ
// *                            {@link Teilklageruecknahme}, das enthält die
// *                            {@link Teilklageruecknahme#parteien} klaeger
// *                            Informationen zur Klägerseite als Beteiligter
// *                            (wird zur Tenorierung benötigt) und beklagter
// *                            Informationen zur Beklagten als Beteiligter (wird
// *                            zur Tenorierung benötigt), die Arrays
// *                            gebuehrenVorher und gebuehrenNachher, (Arrays
// *                            GebuehrenTatbestand[] mit allen schon vor bzw.
// *                            erst nach der "Streitwertreduktion"
// *                            (Teilklagerücknahme bzw -erledigung) entstandenen
// *                            Gebühren, die
// *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
// *                            ein Array long[3] mit streitwerteUndObsiegen[0]
// *                            als dem alten (höheren) Streitwert,
// *                            streitwerteUndObsiegen[1] als dem neuen,
// *                            niedrigeren Streitwert und die
// *                            streitwerteUndObsiegen[2], die Angabe, in welcher
// *                            Höhe der Kläger letzlich obsiegt hat.
// * 
// * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
// *                            b92ii Soll § 92 II ZPO angewandt werden
// *                            (verhältnismäßig geringe Kostentragungen werden
// *                            unterschlagen)
// * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
// *                                  folgenden Plausibilitäten verletzt ist: a)
// *                                  der frühere Streitwert muss größer als der
// *                                  spätere (oder gleich) sein, b) der spätere
// *                                  Streitwert muss größer als die Verurteilung
// *                                  (oder gleich) sein.
// */
//public StreitwertReduktion(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
//	super();
//	Teilklageruecknahme teilklageruecknahme = verfahrensDaten.teilklageruecknahme;
//	this.streitwerte = new Euro[] { teilklageruecknahme.streitwerteUndObsiegen.get(0),
//			teilklageruecknahme.streitwerteUndObsiegen.get(1) };
//	this.verurteilung = teilklageruecknahme.streitwerteUndObsiegen.get(2);
//	// Ausgegliedert in ParsendUndBauend
//	pruefer.pruefeEingabe(teilklageruecknahme);
//	this.klaeger = teilklageruecknahme.parteien.get(0);
//	this.beklagter = teilklageruecknahme.parteien.get(1);
//	this.vorher = teilklageruecknahme.vorher;
//	this.nachher = teilklageruecknahme.nachher;
//	this.allgemein = verfahrensDaten.allgemein;
//	gruende = new StringBuilder();
//	container = new TenorDatenContainer(verfahrensDaten);
//}

///**
// * Die Methode berechneAnteileAnRechtsstreit dient dazu, die Anteile des
// * zurückgenommenen und des verbleibenden Teils am ursprünglichen Streitwert und
// * damit dem Gesamtrechtsstreits zu ermitteln.
// * 
// * @return einen array double[2], mit double-Instanzen zwischen 0.0 und 1.0, bei
// *         dem double[0] dem Anteil des zurückgenommen Teils am früheren
// *         Gesamtstreitwert entspricht und double[1] dem verbleibenden Teil am
// *         früheren Gesamtstreitwert.
// */
//protected double[] berechneAnteileAnRechtsstreit() {
//	double[] zwischen = new double[] { 1.0, 0.0 };
//	zwischen[0] = // (double) (streitwerte[0] - streitwerte[1]) / (double) streitwerte[0];
//			streitwerte[0].subtract(streitwerte[1]).divide(streitwerte[0]);
//	zwischen[1] = // (double) streitwerte[1] / (double) streitwerte[0];
//			streitwerte[1].divide(streitwerte[0]);
//	return zwischen;
//}
//
///**
// * Die Methode berechneLetztenVerlust dient dazu, die Verlustquote bzgl. des
// * niedrigeren Streitwerts nach der Reduktion zu berechnen.
// * 
// * @return einen Wert zwischen 0.0 und 1.0, der in Prozente umgerechnet werden
// *         kann und der Quote entspricht.
// */
//protected double berechneLetzteVerlustQuote() {
//	double zwischen = streitwerte[1].subtract(verurteilung).divide(streitwerte[1]);
//	// (double) (streitwerte[1] - verurteilung) / streitwerte[1];
//	return zwischen;
//}
//

//
//
//	/**
//	 * @return vorher gibt den vorher als {@link List} zurück
//	 */
//	List<GebuehrenTatbestand> getVorher() {
//		return vorher;
//	}
//
//	/**
//	 * Die Methode gibt nachher als {@link List} zurück
//	 * 
//	 * @return
//	 */
//	List<GebuehrenTatbestand> getNachher() {
//		return nachher;
//	}

///**
// * @return streitwerte gibt den streitwerte als long[] zurück.
// */
//Euro[] getStreitwerte() {
//	return streitwerte;
//}