/*
 * ErweiterbaresBeteiligtenTableModel.java
 * eu.gronos.kostenrechner.model.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Euro.ZERO_CENTS;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;
import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * TableModel für die Tabellen im {@link BaumbachAssistenzDialog}. Neu
 * geschrieben in Version 0.5.9, basiert jetzt auf
 * {@link BeteiligtenTableModel}, aber die wichtigsten Methoden übernommen.
 * 
 * Zusätzliche Features: mit {@link #setColumnCount(int)} kann man einstellen,
 * wie viele Spalten derzeit angezeigt werden. Mit
 * {@link #setEachStreitwert(double)} und {@link #setEachUnterliegen(double)}
 * können viele Felder auf einmal gesetzt werden.
 * {@link #getHoechstenStreitwert()} ermittelt den höchsten Streitwert der
 * Tabelle.
 *
 * @author Peter Schuster (setrok)
 * @date 21.09.2019
 *
 */
public class ScalableBeteiligtenTableModel extends BeteiligtenTableModel {

	private static final long serialVersionUID = -7940005728950314998L;
	private int columnCount = 2;
	public static final int MAX_SPALTEN = 4;
	private final static Class<?>[] COLUMN_CLASSES = new Class<?>[] { Integer.class, Beteiligter.class, Euro.class,
			Euro.class, Boolean.class };
//			Double.class, Double.class,
	private final static String[] COLUMN_NAMES = new String[] { "#", "Parteibezeichnung (m/w)", "Streitwert",
			"Unterliegen", "Widerklage?" };

	public ScalableBeteiligtenTableModel(BeteiligtenTyp beteiligtenTyp, BaumbachBeteiligtenListe values) {
		super(beteiligtenTyp, false, values);
	}

	@Deprecated
	public ScalableBeteiligtenTableModel(int beteiligtenTyp, BaumbachBeteiligtenListe values) {
		super(beteiligtenTyp, false, values);
	}

	/**
	 * Für Gebrauch mit JList. Gibt die Beteiligten-Spalte zurück.
	 * Gesamtschuldnerschaften werden herausgefiltert.
	 * 
	 * @return die Spalte mit den Beteiligten als eindimennsionales Array
	 *         Beteiligter[]
	 */
	public Beteiligter[] getBeteiligterColumn() {
		Beteiligter[] bba = new Beteiligter[values.findeGesamtschuldnerIndices(beteiligtenTyp, false).size()];
		int row = -1;
		for (Beteiligter beteiligter : values) {
			if (!(beteiligter instanceof BaumbachGesamtschuldnerschaft)) {
				// Zähler muss seperat vom Zähler der values sein (wg. Gesamtschuldnern)
				row++;
				bba[row] = values.get(row);
			}
		}
		return bba;
	}

	/**
	 * Gibt die Größe ohne Gesamtschuldner aus.
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	@Override
	public int getRowCount() {
		return values == null ? 0 : values.findeGesamtschuldnerIndices(beteiligtenTyp, false).size();
	}

	/**
	 * @return wie viele Spalten angezeigt werden sollen als <code>int</code>. Dabei
	 *         ist Spalte 1 = bis lfd. Nr. des {@link Beteiligter}, 2 = bis
	 *         Parteibezeichnung (m/w), 3 = bis Streitwert, 4 = bis Unterliegen, 5 =
	 *         bis Widerklage.
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	@Override
	public int getColumnCount() {
		return this.columnCount;
	}

	/**
	 * @param columnCount d. {@link #getColumnCount}, d. gesetzt werden soll als
	 *                    {@link int}.
	 */
	public void setColumnCount(int columnCount) {
		if (columnCount != this.columnCount) {
			this.columnCount = columnCount;
			super.fireTableStructureChanged();
		}
	}

	/**
	 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
	 */
	@Override
	public String getColumnName(int column) {
		return COLUMN_NAMES[column];
	}

	/**
	 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(int columnIndex) {
		return COLUMN_CLASSES[columnIndex];
	}

	/**
	 * Die erste Spalte ist nie veränderlich, sonst ist immer nur die letzte Spalte
	 * beschreibbar
	 * 
	 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
	 */
	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return values != null && values.size() > rowIndex && columnCount - 1 == columnIndex;
	}

	/**
	 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int,
	 *      int)
	 */
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		if (values == null || values.size() <= rowIndex || columnIndex == 0 || COLUMN_NAMES.length <= columnIndex) {
			return;
		} else {
			boolean erfolgreich = false;
			switch (columnIndex - 1) {
			case COLUMN_BETEILIGTER:
				erfolgreich = setParteiBezeichnerAt(aValue, rowIndex);
				break;
			case COLUMN_STREITWERT:
				erfolgreich = setStreitwertAt(aValue, rowIndex);
				break;
			case COLUMN_UNTERLIEGEN:
				erfolgreich = setUnterliegenAt(aValue, rowIndex);
				break;
			case COLUMN_WIDERKLAGE:
				System.err.println("Jemand ändert anWiderklageBeteiligt!!!!");
				erfolgreich = setWiderklageAt(aValue, rowIndex);
				break;
			default:
				erfolgreich = false;
			}
			if (erfolgreich) {
				super.fireTableCellUpdated(rowIndex, columnIndex);
				super.fireTableRowsUpdated(rowIndex, rowIndex);
			}
		}
	}

	/**
	 * Setzt den Streitwert für jedes einzelne Element
	 * 
	 * @param streitwert den zu setzenden {@link BaumbachBeteiligter#getAntrag()}
	 *                   als double
	 */
	public void setEachStreitwert(Euro streitwert) {// double
		for (int rowIndex = 0; rowIndex < this.values.size(); rowIndex++)
			if (!(this.values.get(rowIndex) instanceof BaumbachGesamtschuldnerschaft))
				this.values.get(rowIndex).setAntrag(streitwert);
		fireTableRowsUpdated(0, getRowCount() - 1);
	}

	/**
	 * Die Methode ermittelt den höchsten Streitwert, der in der
	 * BaumbachBeteiligtenListe vertreten ist.
	 * 
	 * @return den gefundenen Streitwert als double
	 * 
	 * @see eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe#getHoechstenStreitwert(int)
	 */
	public HtmlRtfFormattierend getHoechstenStreitwert() {// double
//		double sw = 0;
		Euro sw = ZERO_CENTS;
		values.getHoechstenStreitwert(BeteiligtenTyp.KLAEGER);// 0
		for (int row = 0; row < getRowCount(); row++) {
			Euro d = values.get(row).getAntrag();// double
			// if (d > sw)
			if (d.greaterThan(sw))
				sw = d;
		}
		return sw;
	}

	/**
	 * Die Methode setzt das Unterliegen für jedes einzelne Element
	 * 
	 * @param unterliegen das zu setzende {@link BaumbachBeteiligter#getErfolg()}
	 *                    als double
	 */
	public void setEachUnterliegen(Euro unterliegen) {// double
		for (int rowIndex = 0; rowIndex < this.values.size(); rowIndex++)
			if (!(this.values.get(rowIndex) instanceof BaumbachGesamtschuldnerschaft))
				this.values.get(rowIndex).setErfolg(unterliegen);
		fireTableRowsUpdated(0, getRowCount() - 1);
	}

	/**
	 * Für die erste Spalte eine Nummerierung ausgeben, sonst den passenden Wert aus
	 * values (MERKER: {@link BeteiligtenTableModel#getValueAt(int, int)} hat eine
	 * Spalte weniger als {@link ScalableBeteiligtenTableModel} selbst!)
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		// Für die erste Spalte eine Nummerierung ausgeben
		if (columnIndex == 0)
			return rowIndex + 1;
		// wenn es keine Werte gibt oder der Index ausserhalb des Bereichs ist: null
		else if (values == null || values.size() < rowIndex || columnCount < columnIndex)
			return null;
		else
			// sonst den passenden Wert aus values (MERKER: super.values hat eine
			// Spalte weniger als die Tabelle selbst!)
			return super.getValueAt(rowIndex, columnIndex - 1);
	}

}
