/**
 * ForderungenTableModel.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.forderungen;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import eu.gronos.kostenrechner.data.forderungen.AufrechnungsForderung;
import eu.gronos.kostenrechner.data.forderungen.Forderung;
import eu.gronos.kostenrechner.data.forderungen.KlageForderung;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.RowHandler;
import eu.gronos.kostenrechner.interfaces.RowList;
import eu.gronos.kostenrechner.interfaces.TooltipLieferant;

/**
 * Ein AbstractTableModel zur Darstellung von Forderungen
 *
 * @author Peter Schuster (setrok)
 * @date 02.08.2014
 *
 */
public class ForderungenTableModel extends AbstractTableModel
		implements RowHandler<Forderung>, RowList<Forderung>, TooltipLieferant {

	private static final long serialVersionUID = 3439877940063327994L;
	private static final int COLUMN_COUNT = 3;
	private static final String[] COLUMN_NAMES = { "Forderungsart", "Höhe der Forderung", "Begründetheit", "Richtung" };
	private static final String[] COLUMN_TOOLTIPS = { "Hauptklageantrag oder Hilfsaufrechnung", "Höhe der Forderung",
			"Inwieweit das Angriffs- oder Verteidigungsmittel begründet ist", "Richtung" };
	private static final Class<?>[] COLUMN_CLASS = { Forderung.class, Double.class, Double.class, Boolean.class };
	/**
	 * Der ArrayList<Forderung> mit dem Namen values speichert die Werte des
	 * TableModels
	 */
	private List<Forderung> values;

	/**
	 * Der parameterlose Konstruktor legt eine leere Arraylist an.
	 * 
	 * @see javax.swing.table.AbstractTableModel
	 */
	public ForderungenTableModel() {
		super();
		values = new ArrayList<Forderung>();
	}

	/**
	 * Returns the number of rows in the model. A JTable uses this method to
	 * determine how many rows it should display. This method should be quick, as it
	 * is called frequently during rendering.
	 * 
	 * @return the number of rows in the model
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	@Override
	public int getRowCount() {
		if (values == null)
			return 0;
		else
			return values.size();
	}

	/**
	 * Returns the number of columns in the model. A JTable uses this method to
	 * determine how many columns it should create and display by default.
	 * 
	 * 
	 * @return the number of columns in the model (
	 *         {@link ForderungenTableModel#COLUMN_COUNT COLUMN_COUNT}), also die
	 *         Anzahl der Spalten. Es werden erst einmal nur 3 Spalten angezeigt, da
	 *         die Richtung (kl ./. bk oder bk ./. kl) bislang durch die
	 *         Forderungsart vorgegeben ist. Sie wäre dann die vierte Spalte.
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	@Override
	public int getColumnCount() {
		return COLUMN_COUNT;
	}

	/**
	 * Returns the name of the column at columnIndex. This is used to initialize the
	 * table's column header name. Note: this name does not need to be unique; two
	 * columns in a table can have the same name.
	 * 
	 * @param column die Spalte
	 * @return die Überschrift der Spalte (
	 *         {@link ForderungenTableModel#COLUMN_NAMES COLUMN_NAMES}), die
	 *         Spaltenüberschriften
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
	 */
	@Override
	public String getColumnName(int column) {
		return COLUMN_NAMES[column];
	}

	/**
	 * Gibt die genaueste Oberklasse für den Zellinhalt in der Spalte zurück. JTable
	 * nutzt den Wert, um einen DefaultRenderer und einen Editor für die Spalte zu
	 * initialisieren.
	 * 
	 * @param columnIndex die Spalte
	 * @return die genaueste Oberklasse für den zurückgegebenen Wert der Spalte, die
	 *         Klassen der Spalten. Die erste Spalte ist Forderung.class, um einen
	 *         entsprechenden Renderer bauen zu können.
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(int columnIndex) {
		return COLUMN_CLASS[columnIndex];
	}

	/**
	 * Die Methode sagt, ob eine Zelle bearbeitet werden kann.
	 * {@link ForderungenTableModel#setValueAt(Object, int, int) setValueAt} ändert
	 * den Wert sonst nicht und die JTable ruft den Editor nicht auf.
	 * 
	 * @param rowIndex    die Zeile
	 * @param columnIndex die Spalte
	 * @return false für nicht vorhandene Zeilen/Spalten, sonst true
	 * 
	 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
	 */
	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		if (rowIndex < 0 || rowIndex >= getRowCount())
			return false;
		if (columnIndex < 0 || columnIndex >= getColumnCount())
			return false;
		return true;
	}

	/**
	 * Die Methode dient dazu, den Wert einer Zelle zu ändern.
	 * 
	 * @param aValue      der zu setzende Inhalt für die Zeile
	 * @param rowIndex    die Zeile
	 * @param columnIndex die Spalte
	 * 
	 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int,
	 *      int)
	 */
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		if (rowIndex < 0 || rowIndex >= getRowCount())
			return;
		if (columnIndex < 0 || columnIndex >= getColumnCount())
			return;
		if (!isCellEditable(rowIndex, columnIndex))
			return;
		switch (columnIndex) {
		case 0:
			/*
			 * Bei Spalte 0 wird eine neue Forderung übergeben, in die die bisherigen Werte
			 * geklont werden müssen
			 */
			if (aValue instanceof KlageForderung || aValue instanceof AufrechnungsForderung) {
				Forderung f = (Forderung) aValue;
				f.setAntrag(getRow(rowIndex).getAntrag());
				f.setErfolg(getRow(rowIndex).getErfolg());
				values.set(rowIndex, f);
			} else
				return;
			break;
		case 1:
			/* Beim Wert kann einfach das Feld der Forderung geändert werden */
			if (aValue instanceof Double) {
				double d = ((Double) aValue).doubleValue();
				getRow(rowIndex).setAntrag(Euro.ofEuros(d));
			} else
				return;
			break;
		case 2:
			/* Beim Erfolg kann einfach das Feld der Forderung geändert werden */
			if (aValue instanceof Double) {
				double d = ((Double) aValue).doubleValue();
				getRow(rowIndex).setErfolg(Euro.ofEuros(d));
			} else
				return;
			break;
		default:
			return;
		}
		fireTableRowsUpdated(rowIndex, rowIndex);
	}

	/**
	 * Die Methode gibt den Wert für Zeile <code>rowIndex</code>, Spalte
	 * <code>columnIndex</code> zurück
	 * 
	 * @param rowIndex    die Zeile
	 * @param columnIndex die Spalte
	 * @return den Inhalt dort
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		if (rowIndex < 0 || rowIndex >= getRowCount())
			return null;
		switch (columnIndex) {
		case 0:
			return getRow(rowIndex);
		case 1:
			return getRow(rowIndex).getAntrag();
		case 2:
			return getRow(rowIndex).getErfolg();
		default:
			return null;
		}
	}

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt zurückzugeben.
	 * 
	 * @return ein Array Forderung[] mit allen Werten der Tabelle
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#toArray()
	 */
	@Override
	public Forderung[] toArray() {
		if (values != null) {
			Forderung[] array = new Forderung[values.size()];
			array = values.toArray(array);
			return array;
		} else
			return null;
	}

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt zurückzugeben, d.h. eine
	 * Referenz auf die ArrayList, die intern zur Speicherung dient.
	 * 
	 * @return eine ArrayList&lt;Forderung&gt; mit allen Werten der Tabelle
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#getAllValues()
	 */
	@Override
	public List<Forderung> getAllValues() {
		return values;
	}
	//Array

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt auszutauschen, d.h. die
	 * ArrayList, die intern zur Speicherung dient, und führt fireTableDataChanged
	 * durch.
	 * 
	 * @param forderungen eine ArrayList&lt;Forderung&gt; mit den neuen Werten
	 */
	public void setAllValues(List<Forderung> values) {
		this.values = values;
		fireTableDataChanged();
	}//ArrayList

	/**
	 * Entfernt alle Elemente aus der dem TableModel zugrunde liegenden ArrayList
	 * und führt fireTableDataChanged durch.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#clear()
	 */
	@Override
	public void clear() {
		if (values == null)
			this.values = new ArrayList<Forderung>();
		else
			values.clear();
		fireTableDataChanged();
	}

	/**
	 * Gibt eine Zeile als Parameterobjekt des Typs Forderung zurück
	 * 
	 * @param rowIndex der Index der Zeile
	 * @return das Parameterobjekt für die Zeile als <T>
	 * @throws IndexOutOfBoundsException if the index is out of range (index < 0 ||
	 *                                   index >= size()
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#getRow(int)
	 */
	@Override
	public Forderung getRow(int rowIndex) throws IndexOutOfBoundsException {
		if (values == null)
			return null;
		return values.get(rowIndex);
	}

	/**
	 * Entfernt eine Zeile aus dem Model, indem der entsprechende Eintrag aus der
	 * ArrayList&lt;Forderung&gt; gelöscht wird. Danach führt sie
	 * fireTableRowsDeleted durch.
	 * 
	 * @param rowIndex der Index der Zeile
	 * @throws IndexOutOfBoundsException if the index is out of range (index < 0 ||
	 *                                   index >= size()
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#removeRow(int)
	 */
	@Override
	public void removeRow(int rowIndex) throws IndexOutOfBoundsException, IllegalArgumentException {
		if (values == null || rowIndex < 0 || values.size() - 1 < rowIndex)
			return;
		values.remove(rowIndex);
		fireTableRowsDeleted(rowIndex, rowIndex);
	}

	/**
	 * Die Methode dient dazu, ein neues Element an die Werte anzuhängen, was eine
	 * ganze Zeile der Tabelle hinzufügt. Danach wird fireTableRowsInserted
	 * durchgeführt.
	 * 
	 * @param forderung die anzufügende Forderung
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#addRow(java.lang.Object)
	 */
	@Override
	public void addRow(Forderung forderung) throws IllegalArgumentException {
		values.add(forderung);
		fireTableRowsInserted(values.size() - 1, values.size() - 1);
	}

	/**
	 * Die Methode gibt den TooltipText für die angegebene Spalte zurück.
	 * 
	 * @param columnIndex die abgefragte Spalte
	 * @return den TooltipText als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TooltipLieferant#getTooltipText(int)
	 */
	@Override
	public String getTooltipText(int columnIndex) {
		return COLUMN_TOOLTIPS[columnIndex];
	}

	/**
	 * Die Methode wird von diesem TableModel nicht unterstützt.
	 * 
	 * @param c ein Array Forderung[]
	 * @throws NullPointerException          wenn das Array Forderung[] null ist
	 * @throws UnsupportedOperationException wird immer geworfen, weil von diesem
	 *                                       TableModel nicht unterstützt.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#addAll(List<T>)
	 */
	@Override
	public void addAll(List<Forderung> c) throws NullPointerException, UnsupportedOperationException {
	}

}
