/*
 * GebuehrenListModel.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import java.util.Collection;

import javax.swing.AbstractListModel;
import javax.swing.ListModel;
import javax.swing.event.ListDataListener;

import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenVerzeichnis;

/**
 * Ein {@link ListModel} für {@link GebuehrenTatbestand}e, die mit einem
 * Suchfeld gefiltert werden kann.
 *
 * @author Peter Schuster (setrok)
 * @date 16 Jun 2019
 *
 */
public class GebuehrenListModel extends AbstractListModel<GebuehrenTatbestand> {

	private static final long serialVersionUID = 4142435767759392682L;
	private GebuehrenVerzeichnis values;
	private GebuehrenAuflistung filtered;
	private String term = "";

	public GebuehrenListModel() {
		super();
		values = new GebuehrenVerzeichnis();
		filtered = new GebuehrenAuflistung();
		refilter();
	}


	/**
	 * Die Methode fügt alle Elemente der {@link Collection} hinzu.
	 * 
	 * @param verzeichnis ein {@link GebuehrenVerzeichnis}
	 * @return <tt>true</tt> wenn der Wertebestand sich durch den Aufruf geändert
	 *         hat.
	 */
	public void addAll(GebuehrenVerzeichnis verzeichnis) {
		values.putAll(verzeichnis);
		refilter();
	}

	/**
	 * Die Methode gibt die Länge der gefilterten Liste zurück
	 * 
	 * @return die Länge der gefilterten Liste
	 * 
	 * @see javax.swing.ListModel#getSize()
	 */
	@Override
	public int getSize() {
		return filtered.size();
	}

	/**
	 * Die Methode gibt das Element an der gewünschten Position zurück
	 * 
	 * @param index der gewünschte index
	 * @return einen {@link GebuehrenTatbestand} oder <code>null</code>, wenn die
	 *         Liste &lt; index
	 * 
	 * @see javax.swing.ListModel#getElementAt(int)
	 */
	@Override
	public GebuehrenTatbestand getElementAt(int index) {
		if (index >= 0 && index < filtered.size())
			return filtered.get(index);
		else
			return null;
	}

	/**
	 * Gibt den Suchbegriff zurück
	 * 
	 * @return gibt {@link #term} als {@link String} zurück.
	 */
	public String getTerm() {
		return term;
	}

	/**
	 * Setzt den Suchbegriff
	 * 
	 * @param term d. {@link #term}, d. gesetzt werden soll als {@link String}.
	 */
	public void setTerm(String term) {
		this.term = term;
		refilter();
	}

	/**
	 * Die Methode filtert neu. Sie setzt die {@link #filtered} zurück und fügt nur
	 * die Elemente aus {@link #values} hinzu, die den Suchbegriff
	 * {@link #getTerm()} enthalten. Groß-/Kleinschreibung wird ignoriert.
	 * 
	 * {@link ListDataListener}
	 */
	private void refilter() {
		int vorher = 0;
		if (filtered != null || !filtered.isEmpty())
			vorher = filtered.size();
		// .clear();
		filtered = values.filter(getTerm());
		int nachher = filtered.size();
		if (nachher > vorher) {
			fireIntervalAdded(this, vorher, nachher - 1);
		} else if (vorher > nachher) {
			fireIntervalRemoved(this, nachher, vorher - 1);
		}
		fireContentsChanged(this, 0, getSize() - 1);
	}

}
