/**
 * GebuehrenTatbestandTabelle.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import java.util.List;

import javax.swing.table.AbstractTableModel;

import eu.gronos.kostenrechner.data.gebuehren.AnwaltsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.AuslagenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GerichtsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.MehrwertsteuerTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.PauschalTatbestand;
import eu.gronos.kostenrechner.interfaces.RowHandler;
import eu.gronos.kostenrechner.interfaces.RowList;
import eu.gronos.kostenrechner.interfaces.TooltipLieferant;

/**
 * Ein AbstractTableModel für Gebührentatbestände, die intern als ArrayList
 * verwaltet werden.
 * 
 * @author Peter Schuster (setrok)
 * @date: 04.05.2014
 *
 * @todo TODO Reihenfolge muss sich ändern lassen
 */
public class GebuehrenTableModel extends AbstractTableModel
		implements RowHandler<GebuehrenTatbestand>, RowList<GebuehrenTatbestand>, TooltipLieferant {

	private static final int COLUMN_COUNT = 3;
	private static final long serialVersionUID = 2980582500842074125L;
	private static final int SPALTE_BEZEICHNUNG = 0;
	private static final int SPALTE_SATZ = 1;
	private static final int SPALTE_VERZEICHNIS = 2;
	private static final Class<?>[] COLUMN_CLASSES = new Class<?>[] { String.class, Double.class, String.class };
	private GebuehrenAuflistung values;
	private static final String[] COLUMN_NAMES = new String[] { "Gebührentatbestand", "Satz", "Vorschrift/Gesetz" };
	private static final String[] COLUMN_TOOLTIPS = new String[] {
			"Der Gebührentatbestand aus dem Kostenverzeichnis bzw. Vergütungsverzeichnis",
			"Der Gebührensatz bzw. die Höhe der Auslagen",
			"Die Vorschrift aus dem Kostenverzeichnis bzw. Vergütungsverzeichnis" };

	/**
	 * Baut das AbstractTableModel mit den Werten der Gebührentatbestände auf.
	 * 
	 * @param values ein eindimensionales Array GebuehrenTatbestand[]
	 * 
	 */
	public GebuehrenTableModel(GebuehrenAuflistung values) {
		super();
		this.values = new GebuehrenAuflistung();/* ArrayList<GebuehrenTatbestand> */
		this.values.addAll(values);
	}

	/**
	 * Returns the number of rows in the model. A JTable uses this method to
	 * determine how many rows it should display. This method should be quick, as it
	 * is called frequently during rendering.
	 * 
	 * @return the number of rows in the model
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	@Override
	public int getRowCount() {
		if (values != null)
			return values.size();
		else
			return 0;
	}

	/**
	 * Returns the number of columns in the model. A JTable uses this method to
	 * determine how many columns it should create and display by default.
	 * 
	 * @return the number of columns in the model
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	@Override
	public int getColumnCount() {
		return COLUMN_COUNT;
	}

	/**
	 * Die Methode getValueAt dient dazu, den Wert einer bestimmten Zelle
	 * zurückzugeben.
	 * 
	 * @param rowIndex    die gewünschte Zeile
	 * @param columnIndex die gewünschte Spalte, wobei 0 die Bezeichnung als String,
	 *                    1 der Gebührensatz als Double und 2 die Gebührentabelle
	 *                    (also KV GKG oder VV RVG) als String ist.
	 * @return den Zellinhalt
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		switch (columnIndex) {
		case SPALTE_BEZEICHNUNG:
			return getBezeichnungAt(rowIndex);
		case SPALTE_SATZ:
			return getSatzAt(rowIndex);
		case SPALTE_VERZEICHNIS:
			return getVerzeichnisAt(rowIndex);
		default:
			return null;
		}
	}

	/**
	 * Die Methode gibt zurück, ob man eine Zelle bearbeiten kann.
	 * 
	 * @param rowIndex    die gewünschte Zeile
	 * @param columnIndex die gewünschte Spalte, wobei 0 die Bezeichnung als String,
	 *                    1 der Gebührensatz als Double und 2 die Gebührentabelle
	 *                    (also KV GKG oder VV RVG) als String ist.
	 * @return derzeit noch <code>false</code>
	 * 
	 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
	 */
	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		// TODO bearbeitbar machen!
		if (SPALTE_SATZ == columnIndex) {
			final GebuehrenTatbestand gt = values.get(rowIndex);
			if (gt instanceof AuslagenTatbestand) {
//				return true; TODO bearbeitbar machen
			}
		}
		return super.isCellEditable(rowIndex, columnIndex);
	}

	/**
	 * Mit dieser Methode ändert man den Wert einer Zelle.
	 * 
	 * @param aValue      der zu setzende Wert
	 * @param rowIndex    die gewünschte Zeile
	 * @param columnIndex die gewünschte Spalte, wobei 0 die Bezeichnung als String,
	 *                    1 der Gebührensatz als Double und 2 die Gebührentabelle
	 *                    (also KV GKG oder VV RVG) als String ist.
	 * 
	 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int,
	 *      int)
	 */
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		// TODO
		if (SPALTE_SATZ == columnIndex) {
			final GebuehrenTatbestand gt = values.get(rowIndex);
			if (gt instanceof AuslagenTatbestand) {
//				((AuslagenTatbestand) gt).setBetrag((Double) aValue); TODO
			}
		}
		super.setValueAt(aValue, rowIndex, columnIndex);
	}

	/**
	 * Die Methode gibt die Spaltenüberschrift der Spalte zurück
	 * 
	 * @param columnIndex die gewünschte Spalte
	 * @return die Spaltenüberschrift der Spalte
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
	 */
	@Override
	public String getColumnName(int columnIndex) {
		if (columnIndex < COLUMN_COUNT)
			return COLUMN_NAMES[columnIndex];
		else
			return super.getColumnName(columnIndex);
	}

	/**
	 * Die Methode getColumnClass dient dazu, die Klasse des Inhalts der abgefragten
	 * Spalte zurück zu geben.
	 * 
	 * @param columnIndex die abgefragte Spalte
	 * @return die Class, die zu der Spalte passt
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(int columnIndex) {
		if (columnIndex < COLUMN_COUNT)
			return COLUMN_CLASSES[columnIndex];
		else
			return super.getColumnClass(columnIndex);

	}

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt zurückzugeben.
	 * 
	 * @return ein Array GebuehrenTatbestand[] mit allen Werten der Tabelle
	 *
	 * @see eu.gronos.kostenrechner.interfaces.RowList#toArray()
	 * @see java.util.ArrayList#toArray(T[])
	 */
	@Override
	public GebuehrenTatbestand[] toArray() {
		if (values != null) {
			GebuehrenTatbestand[] array = new GebuehrenTatbestand[values.size()];
			array = values.toArray(array);
			return array;
		} else
			return null;
	}

	/**
	 * @return gibt {@link #values} als {@link GebuehrenAuflistung} zurück.
	 */
	public GebuehrenAuflistung getValues() {
		return values;
	}

	/**
	 * Die Methode getAllValues dient dazu, den ganzen Tabelleninhalt zurückzugeben.
	 * 
	 * @return eine ArrayList<GebuehrenTatbestand> mit allen Werten der Tabelle
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#getAllValues()
	 */
	@Override
	public GebuehrenAuflistung getAllValues() {
		return values;
	}

	/**
	 * Entfernt alle Werte aus der zugrundeliegenden ArrayList oder legt eine solche
	 * an, falls noch nicht vorhanden.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#clear()
	 */
	@Override
	public void clear() {
		if (values == null)
			this.values = new GebuehrenAuflistung();
		else
			values.clear();
		fireTableDataChanged();
	}

	/**
	 * Hängt alle Werte des übergebenen Arrays der zugrundeliegenden ArrayList
	 * hinzu.
	 * 
	 * @param gebuehren alle hinzuzufügenden Werte als GebuehrenTatbestand[]
	 * @throws NullPointerException          wenn <code>array</code> leer ist.
	 * @throws UnsupportedOperationException wird nicht geworfen, weil diese
	 *                                       TableModel die Operation unterstützt.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#addAll(List<T>)
	 */
	@Override
	public void addAll(List<GebuehrenTatbestand> gebuehren) throws NullPointerException, UnsupportedOperationException {
		if (gebuehren == null)
			throw new NullPointerException("Das array darf nicht null sein!");
		values.addAll(gebuehren);
		fireTableDataChanged();
	}

	/**
	 * Hängt einen neuen GebuehrenTatbestand hinten an die values an.
	 * 
	 * @param gt der anzuhängende GebuehrenTatbestand
	 *
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#addRow(java.lang.Object)
	 */
	@Override
	public void addRow(GebuehrenTatbestand gt) throws IllegalArgumentException {
		if (gt != null) {
			values.add(gt);
			fireTableRowsInserted(values.size() - 1, values.size() - 1);
		}
	}

	/**
	 * Fügt einen neuen GebuehrenTatbestand an der gewünschten Stelle den values zu.
	 * Vorhandene Zeilen werden nach hinten verschoben.
	 * 
	 * @param rowIndex index at which the specified row is to be inserted
	 * @param gt       der anzuhängende GebuehrenTatbestand
	 * @throws IllegalArgumentException
	 * @see java.util.ArrayList#add(int, Object)
	 */
	public void addRowAt(int rowIndex, GebuehrenTatbestand gt) throws IllegalArgumentException {
		if (gt != null && rowIndex > -1 && rowIndex <= values.size()) {
			values.add(rowIndex, gt);
			fireTableRowsInserted(rowIndex, rowIndex);
		}
	}

	/**
	 * Entfernt eine Zeile aus der Tabelle
	 * 
	 * @param rowIndex die zu entfernende Zeile
	 * @throws IndexOutOfBoundsException
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#removeRow(int)
	 */
	@Override
	public void removeRow(int rowIndex) throws IndexOutOfBoundsException, IllegalArgumentException {
		if (values != null && rowIndex > -1 && rowIndex < getRowCount())
			values.remove(rowIndex);
		fireTableRowsDeleted(rowIndex, rowIndex);
	}

	/**
	 * Gibt eine Zeile als GebuehrenTatbestand zurück
	 * 
	 * @param rowIndex der Index der Zeile
	 * @return den GebuehrenTatbestand für die Zeile
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#getRow(int)
	 */
	@Override
	public GebuehrenTatbestand getRow(int rowIndex) throws IndexOutOfBoundsException {
		return values.get(rowIndex);
	}

	/**
	 * Die Methode gibt den Tooltip-Text für die angegebene Spalte zurück.
	 * 
	 * @param columnIndex der Index der Zeile
	 * @return den Tooltip-Text für die Zeile
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TooltipLieferant#getTooltipText(int)
	 */
	@Override
	public String getTooltipText(int columnIndex) {
		return COLUMN_TOOLTIPS[columnIndex];
	}

	/**
	 * Die Methode dient dazu, von getValueAt() aufgerufen zu werden.
	 * 
	 * @param rowIndex die gewünschte Zeile
	 * @return "KV GKG" oder "VV RVG" als String
	 */
	private String getVerzeichnisAt(int rowIndex) {
		if (values == null)
			return null;
		final Class<? extends GebuehrenTabelle> gebuehrenKlasse = values.get(rowIndex).getGebuehrenKlasse();
		String verzeichnis = "";
		if (GerichtsGebuehrenTabelle.class == gebuehrenKlasse) {
			verzeichnis = "GKG";
		} else if (AnwaltsGebuehrenTabelle.class == gebuehrenKlasse) {
			verzeichnis = "RVG";
		}
		return String.format("%s (%s)", values.get(rowIndex).getVorschrift(), verzeichnis);
	}

	/**
	 * Die Methode dient dazu, von getValueAt() aufgerufen zu werden.
	 * 
	 * @param rowIndex die gewünschte Zeile
	 * @return den satz eines GebuehrenTatbestand bzw. die auslagen eines
	 *         AuslagenTatbestands
	 */
	private Double getSatzAt(int rowIndex) {
		if (values != null) {
			GebuehrenTatbestand gt = values.get(rowIndex);
			if (gt instanceof AuslagenTatbestand)
				return ((AuslagenTatbestand) gt).getBetrag().doubleValue();
			else if (gt instanceof PauschalTatbestand)
				return ((PauschalTatbestand) gt).getBetrag().doubleValue();
			else if (gt instanceof GebuehrenSatzTatbestand)
				return ((GebuehrenSatzTatbestand) gt).getSatz();
			else if (gt instanceof MehrwertsteuerTatbestand)
				return ((MehrwertsteuerTatbestand) gt).getSteuerSatz();
		}
		return null;
	}

	/**
	 * Die Methode dient dazu, von getValueAt() aufgerufen zu werden.
	 * 
	 * @param rowIndex die gewünschte Zeile
	 * @return die {@link GebuehrenSatzTatbestand#getBezeichnung() Bezeichnung} des
	 *         Tatbestands in der angegebenen Zeile
	 */
	private String getBezeichnungAt(int rowIndex) {
		if (values != null)
			return values.get(rowIndex).getBezeichnung();
		return null;
	}
}
